/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2024 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.evm.gamification.scheduling.task;

import java.util.ArrayList;
import java.util.List;

import io.meeds.common.ContainerTransactional;
import io.meeds.evm.gamification.service.EvmContractTransferService;
import io.meeds.evm.gamification.utils.Utils;
import io.meeds.gamification.model.RuleDTO;

import org.apache.commons.collections4.CollectionUtils;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.web3j.protocol.exceptions.ClientConnectionException;

@Component
public class EvmContractScanTask {

  private static final Logger        LOG = LoggerFactory.getLogger(EvmContractScanTask.class);

  @Autowired
  private EvmContractTransferService evmContractTransferService;

  private int                        blockchainConnectionErrorCount = 0;

  @ContainerTransactional
  @Scheduled(cron = "${gamification.evm.transactionScan.cron:0 */5 * * * *}")
  public synchronized void scanForContractTransactions() { // NOSONAR
    List<RuleDTO> enabledRules = evmContractTransferService.getEnabledEvmRules();
    if (CollectionUtils.isNotEmpty(enabledRules)) {
      List<RuleDTO> rules = new ArrayList<>();
      enabledRules.forEach(rule -> {
        try {
          boolean isRuleExists = false;
          if (rules != null) {
            isRuleExists = rules.stream()
                                .anyMatch(ruleDTO -> ruleDTO.getEvent()
                                                            .getProperties()
                                                            .get(Utils.NETWORK_ID)
                                                            .compareTo(rule.getEvent().getProperties().get(Utils.NETWORK_ID)) == 0
                                    && StringUtils.equals(ruleDTO.getEvent().getProperties().get(Utils.CONTRACT_ADDRESS).toLowerCase(),
                                                          rule.getEvent().getProperties().get(Utils.CONTRACT_ADDRESS).toLowerCase())
                                    && StringUtils.equals(ruleDTO.getEvent().getTrigger(), rule.getEvent().getTrigger()));
          }
          if (!isRuleExists) {
            evmContractTransferService.scanForContractTransactions(rule);
            rules.add(rule);
          }
        } catch (ClientConnectionException e) {
          if (blockchainConnectionErrorCount == 0) {
            LOG.error("Error connecting to Blockchain", e);
          } else if (blockchainConnectionErrorCount < 10) {
            LOG.warn("Error connecting to Blockchain: {}", e.getMessage());
          } else if (blockchainConnectionErrorCount == 10) {
            LOG.warn("Error connecting to Blockchain (Log level will be switched to debug): {}", e.getMessage());
          } else {
            LOG.debug("Error connecting to Blockchain: {}", e.getMessage());
          }
          blockchainConnectionErrorCount++;
        } catch (Exception e) {
          LOG.error("An error occurred while rewarding for {} rule", rule.getTitle(), e);
        }
      });
    }

  }

}
