/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 Meeds Association
 * contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package io.meeds.gamification.entity;

import java.io.Serializable;
import java.util.Date;

import io.meeds.gamification.constant.EntityType;
import io.meeds.gamification.constant.IdentityType;
import io.meeds.gamification.constant.RealizationStatus;

import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.NamedQuery;
import jakarta.persistence.SequenceGenerator;
import jakarta.persistence.Table;
import lombok.Data;
import lombok.EqualsAndHashCode;

@Entity(name = "RealizationEntity")
@Table(name = "GAMIFICATION_ACTIONS_HISTORY")
@NamedQuery(
  name = "RealizationEntity.getLeaderboardRank",
  query = """
    SELECT r.rankIndex FROM (
      SELECT ROW_NUMBER() OVER() AS rankIndex, l.earnerId AS earnerId FROM (
        SELECT
          g.earnerId AS earnerId,
          SUM(g.actionScore) AS total
        FROM RealizationEntity AS g
        WHERE g.earnerType = :earnerType
        AND g.status = :status
        GROUP BY g.earnerId
        ORDER BY total DESC
     ) AS l
   ) AS r
   WHERE r.earnerId = :earnerId
  """
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardRankByDatesAndProgramIds",
  query = """
    SELECT r.rankIndex FROM (
      SELECT ROW_NUMBER() OVER() AS rankIndex, l.earnerId AS earnerId FROM (
        SELECT
          g.earnerId AS earnerId,
          SUM(g.actionScore) AS total
        FROM RealizationEntity AS g
        WHERE g.createdDate >= :fromDate
        AND g.createdDate < :toDate
        AND g.domainEntity.id IN (:programIds)
        AND g.earnerType = :earnerType
        AND g.status = :status
        GROUP BY g.earnerId
        ORDER BY total DESC
     ) AS l
   ) AS r
   WHERE r.earnerId = :earnerId
  """
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardRankByProgramIds",
  query = """
    SELECT r.rankIndex FROM (
      SELECT ROW_NUMBER() OVER() AS rankIndex, l.earnerId AS earnerId FROM (
        SELECT
          g.earnerId AS earnerId,
          SUM(g.actionScore) AS total
        FROM RealizationEntity AS g
        WHERE g.domainEntity.id IN (:programIds)
        AND g.earnerType = :earnerType
        AND g.status = :status
        GROUP BY g.earnerId
        ORDER BY total DESC
     ) AS l
   ) AS r
   WHERE r.earnerId = :earnerId
  """
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardRankByDates",
  query = """
    SELECT r.rankIndex FROM (
      SELECT ROW_NUMBER() OVER() AS rankIndex, l.earnerId AS earnerId FROM (
        SELECT
          g.earnerId AS earnerId,
          SUM(g.actionScore) AS total
        FROM RealizationEntity AS g
        WHERE g.createdDate >= :fromDate
        AND g.createdDate < :toDate
        AND g.earnerType = :earnerType
        AND g.status = :status
        GROUP BY g.earnerId
        ORDER BY total DESC
     ) AS l
   ) AS r
   WHERE r.earnerId = :earnerId
  """
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboard",
  query = "SELECT new io.meeds.gamification.model.StandardLeaderboard(g.earnerId as earnerId, SUM(g.actionScore) as total)"
    + " FROM RealizationEntity g"
    + " WHERE g.earnerType = :earnerType"
    + " AND g.status = :status"
    + " GROUP BY g.earnerId"
    + " ORDER BY total DESC, earnerId DESC"
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardByDateAndProgramIds",
  query = "SELECT new io.meeds.gamification.model.StandardLeaderboard(g.earnerId as earnerId, SUM(g.actionScore) as total)"
    + " FROM RealizationEntity g"
    + " WHERE g.createdDate >= :fromDate"
    + " AND g.createdDate < :toDate"
    + " AND g.domainEntity.id IN (:programIds)"
    + " AND g.earnerType = :earnerType"
    + " AND g.status = :status"
    + " GROUP BY  g.earnerId"
    + " ORDER BY total DESC, earnerId DESC"
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardByProgramIds",
  query = "SELECT new io.meeds.gamification.model.StandardLeaderboard(g.earnerId as earnerId, SUM(g.actionScore) as total)"
    + " FROM RealizationEntity g"
    + " WHERE g.domainEntity.id IN (:programIds)"
    + " AND g.earnerType = :earnerType"
    + " AND g.status = :status"
    + " GROUP BY g.earnerId"
    + " ORDER BY total DESC, earnerId DESC"
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardByDate",
  query = "SELECT new io.meeds.gamification.model.StandardLeaderboard(g.earnerId as earnerId, SUM(g.actionScore) as total)"
    + " FROM RealizationEntity g"
    + " WHERE g.createdDate >= :fromDate"
    + " AND g.createdDate < :toDate"
    + " AND g.earnerType = :earnerType"
    + " AND g.status = :status"
    + " GROUP BY g.earnerId"
    + " ORDER BY total DESC, earnerId DESC"
)

@NamedQuery(
  name = "RealizationEntity.getLeaderboardStatsByIdentityId",
  query = "SELECT new io.meeds.gamification.model.PiechartLeaderboard(g.domainEntity.id as domainId, SUM(g.actionScore) as total)"
      + " FROM RealizationEntity g"
      + " WHERE g.earnerId = :earnerId"
      + " AND g.domainEntity IS NOT NULL"
      + " AND g.actionScore > 0"
      + " AND g.status = :status"
      + " GROUP BY g.domainEntity.id"
      + " ORDER BY total DESC, domainId DESC"
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardStatsByIdentityIdAndSpaceId",
  query = "SELECT new io.meeds.gamification.model.PiechartLeaderboard(g.domainEntity.id as domainId, SUM(g.actionScore) as total)"
    + " FROM RealizationEntity g"
    + " WHERE g.earnerId = :earnerId"
    + " AND g.domainEntity IS NOT NULL"
    + " AND g.domainEntity.audienceId = :spaceId"
    + " AND g.actionScore > 0"
    + " AND g.status = :status"
    + " GROUP BY g.domainEntity.id"
    + " ORDER BY total DESC, domainId DESC"
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardStatsByIdentityIdAndDates",
  query = "SELECT new io.meeds.gamification.model.PiechartLeaderboard(g.domainEntity.id as domainId, SUM(g.actionScore) as total)"
    + " FROM RealizationEntity g"
    + " WHERE g.earnerId = :earnerId"
    + " AND g.domainEntity IS NOT NULL"
    + " AND g.actionScore > 0"
    + " AND g.status = :status"
    + " AND g.createdDate >= :fromDate"
    + " AND g.createdDate < :toDate"
    + " GROUP BY  g.domainEntity.id"
    + " ORDER BY total DESC, domainId DESC"
)
@NamedQuery(
  name = "RealizationEntity.getLeaderboardStatsByIdentityIdAndDatesAndSpaceId",
  query = "SELECT new io.meeds.gamification.model.PiechartLeaderboard(g.domainEntity.id as domainId, SUM(g.actionScore) as total)"
      + " FROM RealizationEntity g"
      + " WHERE g.earnerId = :earnerId"
      + " AND g.domainEntity IS NOT NULL"
      + " AND g.domainEntity.audienceId = :spaceId"
      + " AND g.actionScore > 0"
      + " AND g.status = :status"
      + " AND g.createdDate >= :fromDate"
      + " AND g.createdDate < :toDate"
      + " GROUP BY  g.domainEntity.id"
      + " ORDER BY total DESC, domainId DESC"
)

@NamedQuery(
  name = "RealizationEntity.getScoreByIdentityId",
  query = " SELECT SUM(g.actionScore) FROM RealizationEntity g" +
          " WHERE g.earnerId = :earnerId" +
          " AND g.status = :status"
)
@NamedQuery(name = "RealizationEntity.getScorePerProgramByIdentityId", query = "SELECT"
    + " new io.meeds.gamification.model.ProfileReputation(g.domainEntity.id,SUM(g.actionScore))"
    + " FROM RealizationEntity g WHERE g.earnerId = :earnerId AND g.status = :status AND g.domainEntity IS NOT NULL GROUP BY g.domainEntity.id")
@NamedQuery(
  name = "RealizationEntity.getScoreByIdentityIdAndBetweenDatesAndProgramIds",
  query = """
    SELECT SUM(g.actionScore) as total
    FROM RealizationEntity g
    WHERE g.domainEntity.id IN (:programIds)
    AND g.earnerId = :earnerId
    AND g.status = :status
    AND g.createdDate >= :fromDate
    AND g.createdDate < :toDate
  """)
@NamedQuery(name = "RealizationEntity.getScoreByIdentityIdAndBetweenDates", query = "SELECT SUM(g.actionScore) as total"
    + " FROM RealizationEntity g  WHERE g.earnerId = :earnerId AND g.status = :status AND g.createdDate >= :fromDate AND g.createdDate < :toDate")
@NamedQuery(name = "RealizationEntity.getScoreByIdentityIdsAndBetweenDates", query = "SELECT g.earnerId,SUM(g.actionScore) as total"
    + " FROM RealizationEntity g  WHERE g.earnerId IN :earnersId AND g.status = :status AND g.createdDate >= :fromDate AND g.createdDate < :toDate GROUP BY g.earnerId")
@NamedQuery(
  name = "RealizationEntity.countRealizationsInPeriod",
  query = "SELECT COUNT(a) FROM RealizationEntity a"
      + " WHERE a.ruleEntity.id = :ruleId"
      + " AND a.earnerId = :earnerId"
      + " AND a.status IN (:status)"
      + " AND (a.sendingDate >= :date OR (a.sendingDate IS NULL AND a.createdDate >= :date))"
)
@NamedQuery(
  name = "RealizationEntity.countRealizationsByRuleIdAndEarnerId",
  query = "SELECT COUNT(a) FROM RealizationEntity a"
      + " WHERE a.ruleEntity.id = :ruleId"
      + " AND a.earnerId = :earnerId"
      + " AND a.status = :status"
)
@NamedQuery(
  name = "RealizationEntity.findRealizationsByRuleIdAndEarnerIdAndReceiverAndObjectId",
  query = "SELECT g.id FROM RealizationEntity g"
      + " WHERE g.ruleEntity.id = :ruleId"
      + " AND g.earnerId = :earnerId"
      + " AND g.receiver = :receiverId"
      + " AND g.objectId = :objectId"
      + " AND g.objectType = :objectType"
      + " AND g.status IN (:status)"
      + " ORDER BY g.id DESC"
)
@NamedQuery(
  name = "RealizationEntity.getRealizationsByObjectIdAndObjectType",
  query = "SELECT g.id FROM RealizationEntity g" +
    " WHERE g.objectId = :objectId" +
    " AND g.objectType = :objectType"
)
@NamedQuery(
  name = "RealizationEntity.countParticipantsBetweenDates",
  query = "SELECT COUNT(DISTINCT g.earnerId) FROM RealizationEntity g" +
      " WHERE g.createdDate >= :fromDate" +
      " AND g.createdDate < :toDate" +
      " AND g.earnerType = :earnerType" +
      " AND g.status = :status"
)

@NamedQuery(name = "RealizationEntity.getParticipantsBetweenDates",
  query = "SELECT DISTINCT g.earnerId FROM RealizationEntity g" +
      " WHERE g.createdDate >= :fromDate" +
      " AND g.createdDate < :toDate" +
      " AND g.earnerType = :earnerType" +
      " AND g.status = :status")

@Data
@EqualsAndHashCode(callSuper = true)
public class RealizationEntity extends AbstractAuditingEntity implements Serializable {

  private static final long serialVersionUID = 2554394120711454093L;

  @Id
  @SequenceGenerator(name = "SEQ_GAMIFICATION_SCORE_HISTORY_ID", sequenceName = "SEQ_GAMIFICATION_SCORE_HISTORY_ID", allocationSize = 1)
  @GeneratedValue(strategy = GenerationType.AUTO, generator = "SEQ_GAMIFICATION_SCORE_HISTORY_ID")
  @Column(name = "ID")
  protected Long            id;

  @Column(name = "EARNER_ID", nullable = false)
  private long              earnerId;

  @Column(name = "EARNER_TYPE", nullable = false)
  private IdentityType      earnerType;

  @Column(name = "GLOBAL_SCORE", nullable = false)
  protected long            globalScore;

  @Column(name = "ACTION_TITLE", nullable = false)
  private String            actionTitle;

  @Column(name = "DOMAIN", nullable = false)
  private String            domain;

  @Column(name = "CONTEXT", nullable = true)
  private String            context;

  @Column(name = "ACTION_SCORE", nullable = false)
  private long              actionScore;

  @Column(name = "RECEIVER", nullable = false)
  private String            receiver;

  @Column(name = "OBJECT_ID")
  private String            objectId;

  @Column(name = "OBJECT_TYPE")
  private String            objectType;

  @ManyToOne
  @JoinColumn(name = "DOMAIN_ID")
  private ProgramEntity     domainEntity;

  @ManyToOne
  @JoinColumn(name = "RULE_ID")
  private RuleEntity        ruleEntity;

  @Column(name = "ACTIVITY_ID")
  private Long              activityId;

  @Column(name = "COMMENT")
  private String            comment;

  @Column(name = "CREATOR_ID")
  private Long              creator;

  @Enumerated(EnumType.ORDINAL)
  @Column(name = "STATUS", nullable = false)
  private RealizationStatus status;

  @Enumerated(EnumType.ORDINAL)
  @Column(name = "TYPE", nullable = false)
  private EntityType        type;

  @Column(name = "SENDING_DATE")
  protected Date            sendingDate;

  @Column(name = "REVIEWER_ID")
  private Long              reviewerId;

}
