/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.gatein.pc.portlet.impl.container;

import org.gatein.pc.portlet.container.object.PortletFilterObject;
import org.gatein.pc.portlet.container.managed.LifeCycleStatus;
import org.gatein.pc.portlet.container.managed.ManagedPortletFilter;
import org.gatein.pc.portlet.container.managed.ManagedObjectRegistryEventListener;
import org.gatein.pc.portlet.container.PortletFilterContext;
import org.gatein.pc.portlet.container.PortletFilter;

public class PortletFilterLifeCycle extends LifeCycle implements ManagedPortletFilter
{

   /** . */
   private PortletApplicationLifeCycle portletApplicationLifeCycle;

   /** . */
   final PortletFilterContext portletFilterContext;

   /** . */
   final PortletFilterObject portletFilter;

   PortletFilterLifeCycle(
      PortletApplicationLifeCycle portletApplicationLifeCycle,
      PortletFilterContext portletFilterContext,
      PortletFilterObject portletFilter)
   {
      this.portletApplicationLifeCycle = portletApplicationLifeCycle;
      this.portletFilterContext = portletFilterContext;
      this.portletFilter = portletFilter;
   }

   protected void invokeCreate() throws Exception
   {
      if (portletApplicationLifeCycle.getStatus().getStage() < LifeCycleStatus.CREATED.getStage())
      {
         throw new DependencyNotResolvedException("The parent application is not created");
      }

      //
      portletFilter.create();
   }

   protected void invokeStart() throws Exception
   {
      if (portletApplicationLifeCycle.getStatus().getStage() < LifeCycleStatus.STARTED.getStage())
      {
         throw new DependencyNotResolvedException("The parent application is not started");
      }

      //
      portletFilter.start();
   }

   @Override
   protected void promoteDependents(LifeCycleStatus to)
   {
      // Dependent containers may or not start (for instance if a container depends on several filters)
      for (PortletContainerLifeCycle portletContainerLifeCycle : portletApplicationLifeCycle.getDependencies(this))
      {
         portletContainerLifeCycle.promote(to);
      }
   }

   @Override
   protected void demoteDependents(LifeCycleStatus to)
   {
      for (PortletContainerLifeCycle portletContainerLifeCycle : portletApplicationLifeCycle.getDependencies(this))
      {
         portletContainerLifeCycle.demote(to);
      }
   }

   protected void invokeStop()
   {
      portletFilter.stop();
   }

   protected void invokeDestroy()
   {
      portletFilter.destroy();
   }

   public String getId()
   {
      return portletFilter.getId();
   }

   public PortletApplicationLifeCycle getManagedPortletApplication()
   {
      return portletApplicationLifeCycle;
   }

   public String toString()
   {
      return getClass().getSimpleName() + "[" + portletFilter.getId() + "]";
   }

   public PortletFilter getPortletFilter()
   {
      return portletFilter;
   }

   protected ManagedObjectRegistryEventListener getListener()
   {
      return portletApplicationLifeCycle.getListener();
   }
}
