/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.gatein.pc.test.portlet.jsr168.ext.expiringcache;

import org.gatein.pc.test.unit.PortletTestCase;
import org.gatein.pc.test.unit.PortletTestContext;
import org.gatein.pc.test.unit.Assertion;
import org.gatein.pc.test.unit.actions.PortletRenderTestAction;
import org.gatein.pc.test.unit.web.UTP10;
import org.gatein.pc.test.unit.web.UTP11;
import org.gatein.common.util.Tools;
import org.gatein.pc.test.unit.annotations.TestCase;
import org.gatein.pc.test.unit.protocol.response.Response;
import org.gatein.pc.test.unit.protocol.response.FailureResponse;
import org.gatein.pc.test.unit.protocol.response.EndTestResponse;
import org.gatein.pc.test.unit.protocol.response.InvokeGetResponse;
import static org.gatein.pc.test.unit.Assert.*;
import org.gatein.pc.test.unit.Failure;

import javax.portlet.Portlet;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletURL;
import java.util.Set;
import java.util.HashSet;
import java.io.IOException;

@TestCase({
   Assertion.EXT_EXPIRING_CACHE_3
   })
public class ExpiringCache
{

   final Set calls = new HashSet();
   String url;

   public ExpiringCache(PortletTestCase seq)
   {
      //
      seq.bindAction(0, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Content is not cached
            calls.add("0");

            // Refresh
            url = response.createRenderURL().toString();
            return new InvokeGetResponse(url);
         }
      });

      //
      seq.bindAction(1, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Should not be called
            calls.add("1");
            return null;
         }
      });
      seq.bindAction(1, UTP11.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Refresh
            return new InvokeGetResponse(url);
         }
      });

      //
      seq.bindAction(2, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Could be called depending on whether the portal
            // decides to invoke UTP2 or UTP3 first
            // so if it's called we need to disable cache otherwise the next
            // render will probably not be called
            response.setProperty(RenderResponse.EXPIRATION_CACHE, "0");
            return null;
         }
      });
      seq.bindAction(2, UTP11.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            try
            {
               assertFalse(calls.contains("1"));

               // Wait 5 seconds for the cache entry to be invalid
               Thread.sleep(5 * 1000);

               // Refresh
               return new InvokeGetResponse(url);
            }
            catch (InterruptedException e)
            {
               return new FailureResponse(Failure.createFailure(e));
            }
         }
      });

      //
      seq.bindAction(3, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Should be called
            calls.add("3");

            // Invoke the same but with different render parameter
            PortletURL tmp = response.createRenderURL();
            tmp.setParameter("abc", "def");
            url = tmp.toString();
            return new InvokeGetResponse(url);
         }
      });

      //
      seq.bindAction(4, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Should be called
            calls.add("4");

            // Refresh
            return new InvokeGetResponse(url);
         }
      });

      //
      seq.bindAction(5, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Should not be called
            calls.add("5");
            return null;
         }
      });
      seq.bindAction(5, UTP11.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Refresh
            return new InvokeGetResponse(url);
         }
      });

      //
      seq.bindAction(6, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Could be called depending on whether the portal
            // decides to invoke UTP2 or UTP3 first
            // so if it's called we need to disable cache otherwise the next
            // render will probably not be called
            response.setProperty(RenderResponse.EXPIRATION_CACHE, "0");
            return null;
         }
      });
      seq.bindAction(6, UTP11.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            try
            {
               assertFalse(calls.contains("5"));

               // Wait 5 seconds for the cache entry to be invalid
               Thread.sleep(5 * 1000);

               // Refresh
               return new InvokeGetResponse(url);
            }
            catch (InterruptedException e)
            {
               return new FailureResponse(Failure.createFailure(e));
            }
         }
      });

      //
      seq.bindAction(7, UTP10.RENDER_JOIN_POINT, new PortletRenderTestAction()
      {
         protected Response run(Portlet portlet, RenderRequest request, RenderResponse response, PortletTestContext context) throws IOException, PortletException
         {
            // Should be called
            calls.add("7");

            //
            Set expected = Tools.toSet("0", "3", "4", "7");
            assertEquals(expected, calls);

            // Refresh
            return new EndTestResponse();
         }
      });
   }
}
