package io.quarkus.runtime.graal;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

import org.graalvm.nativeimage.hosted.Feature;
import org.graalvm.nativeimage.hosted.RuntimeReflection;
import org.jboss.logging.Logger;

/**
 * Weak reflection implementation called from generated bytecode
 */
public class WeakReflection {

    static final Logger log = Logger.getLogger(WeakReflection.class);

    public static void register(Feature.BeforeAnalysisAccess analysisAccess, Class<?> clazz, boolean constructors,
            boolean methods, boolean fields) {
        analysisAccess.registerReachabilityHandler(new Callback(clazz, constructors, methods, fields), clazz);
    }

    public static class Callback implements Consumer<Feature.DuringAnalysisAccess> {
        final AtomicBoolean onlyOnce = new AtomicBoolean(false);
        final Class<?> clazz;
        final boolean constructors;
        final boolean methods;
        final boolean fields;

        public Callback(Class<?> clazz, boolean constructors, boolean methods, boolean fields) {
            this.clazz = clazz;
            this.constructors = constructors;
            this.methods = methods;
            this.fields = fields;
        }

        @Override
        public void accept(Feature.DuringAnalysisAccess duringAnalysisAccess) {
            if (!onlyOnce.compareAndSet(false, true)) {
                return;
            }
            log.debugf("Registering %s for reflection as it is reachable", clazz);
            RuntimeReflection.register(clazz);
            if (fields) {
                RuntimeReflection.register(clazz.getDeclaredFields());
            }
            if (constructors) {
                RuntimeReflection.register(clazz.getDeclaredConstructors());
            }
            if (methods) {
                RuntimeReflection.register(clazz.getDeclaredMethods());
            }
        }
    }

}
