/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.jwt.auth.principal;

import io.smallrye.jwt.auth.principal.DecryptionKeyLocationResolver;
import io.smallrye.jwt.auth.principal.JWTAuthContextInfo;
import io.smallrye.jwt.auth.principal.KeyLocationResolver;
import io.smallrye.jwt.auth.principal.ParseException;
import io.smallrye.jwt.auth.principal.PrincipalLogging;
import io.smallrye.jwt.auth.principal.PrincipalMessages;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.eclipse.microprofile.jwt.Claims;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwe.JsonWebEncryption;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.NumericDate;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwt.consumer.JwtContext;
import org.jose4j.keys.resolvers.DecryptionKeyResolver;
import org.jose4j.keys.resolvers.VerificationKeyResolver;
import org.jose4j.lang.JoseException;
import org.jose4j.lang.UnresolvableKeyException;

public class DefaultJWTTokenParser {
    private static final String ROLE_MAPPINGS = "roleMappings";
    private static final Pattern CLAIM_PATH_PATTERN = Pattern.compile("\\/(?=(?:(?:[^\"]*\"){2})*[^\"]*$)");
    private volatile VerificationKeyResolver keyResolver;
    private volatile DecryptionKeyResolver decryptionKeyResolver;

    public JwtContext parse(String token, JWTAuthContextInfo authContextInfo) throws ParseException {
        String tokenSequence = token;
        ProtectionLevel level = this.getProtectionLevel(authContextInfo);
        if (level == ProtectionLevel.SIGN_ENCRYPT) {
            tokenSequence = this.decryptSignedToken(tokenSequence, authContextInfo);
            level = ProtectionLevel.SIGN;
        }
        return this.parseClaims(tokenSequence, authContextInfo, level);
    }

    private String decryptSignedToken(String token, JWTAuthContextInfo authContextInfo) throws ParseException {
        try {
            JsonWebEncryption jwe = new JsonWebEncryption();
            jwe.setAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, new String[]{authContextInfo.getKeyEncryptionAlgorithm().getAlgorithm()}));
            if (authContextInfo.getPrivateDecryptionKey() != null) {
                jwe.setKey((Key)authContextInfo.getPrivateDecryptionKey());
            } else if (authContextInfo.getSecretDecryptionKey() != null) {
                jwe.setKey((Key)authContextInfo.getSecretDecryptionKey());
            } else {
                jwe.setKey(this.getDecryptionKeyResolver(authContextInfo).resolveKey(jwe, null));
            }
            jwe.setCompactSerialization(token);
            return jwe.getPlaintextString();
        }
        catch (UnresolvableKeyException e) {
            PrincipalLogging.log.decryptionKeyUnresolvable();
            throw PrincipalMessages.msg.decryptionKeyUnresolvable();
        }
        catch (JoseException e) {
            PrincipalLogging.log.encryptedTokenSequenceInvalid();
            throw PrincipalMessages.msg.encryptedTokenSequenceInvalid();
        }
    }

    private JwtContext parseClaims(String token, JWTAuthContextInfo authContextInfo, ProtectionLevel level) throws ParseException {
        try {
            Object groupsClaim;
            JwtConsumerBuilder builder = new JwtConsumerBuilder();
            if (level == ProtectionLevel.SIGN) {
                if (authContextInfo.getPublicVerificationKey() != null) {
                    builder.setVerificationKey((Key)authContextInfo.getPublicVerificationKey());
                } else if (authContextInfo.getSecretVerificationKey() != null) {
                    builder.setVerificationKey((Key)authContextInfo.getSecretVerificationKey());
                } else {
                    builder.setVerificationKeyResolver(this.getVerificationKeyResolver(authContextInfo));
                }
                builder.setJwsAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, new String[]{authContextInfo.getSignatureAlgorithm().getAlgorithm()}));
            } else {
                builder.setEnableRequireEncryption();
                builder.setDisableRequireSignature();
                if (authContextInfo.getPrivateDecryptionKey() != null) {
                    builder.setDecryptionKey((Key)authContextInfo.getPrivateDecryptionKey());
                } else if (authContextInfo.getSecretDecryptionKey() != null) {
                    builder.setDecryptionKey((Key)authContextInfo.getSecretDecryptionKey());
                } else {
                    builder.setDecryptionKeyResolver(this.getDecryptionKeyResolver(authContextInfo));
                }
                builder.setJweAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, new String[]{authContextInfo.getKeyEncryptionAlgorithm().getAlgorithm()}));
            }
            builder.setRequireExpirationTime();
            if (authContextInfo.getMaxTimeToLiveSecs() != null) {
                builder.setRequireIssuedAt();
            }
            if (authContextInfo.getIssuedBy() != null) {
                builder.setExpectedIssuer(authContextInfo.getIssuedBy());
            }
            if (authContextInfo.getExpGracePeriodSecs() > 0) {
                builder.setAllowedClockSkewInSeconds(authContextInfo.getExpGracePeriodSecs());
            } else {
                builder.setEvaluationTime(NumericDate.fromSeconds((long)0L));
            }
            this.setExpectedAudience(builder, authContextInfo);
            if (authContextInfo.isRelaxVerificationKeyValidation()) {
                builder.setRelaxVerificationKeyValidation();
            }
            JwtConsumer jwtConsumer = builder.build();
            JwtContext jwtContext = jwtConsumer.process(token);
            JwtClaims claimsSet = jwtContext.getJwtClaims();
            this.verifyTimeToLive(authContextInfo, claimsSet);
            this.verifyRequiredClaims(authContextInfo, jwtContext);
            claimsSet.setClaim(Claims.raw_token.name(), (Object)token);
            if (!claimsSet.hasClaim(Claims.sub.name())) {
                String sub = this.findSubject(authContextInfo, claimsSet);
                claimsSet.setClaim(Claims.sub.name(), (Object)sub);
            }
            if (authContextInfo.isRequireNamedPrincipal()) {
                this.checkNameClaims(jwtContext);
            }
            if ((groupsClaim = claimsSet.getClaimValue(Claims.groups.name())) == null || groupsClaim instanceof Map) {
                List<String> groups = this.findGroups(authContextInfo, claimsSet);
                claimsSet.setClaim(Claims.groups.name(), groups);
            } else if (groupsClaim instanceof String) {
                claimsSet.setClaim(Claims.groups.name(), this.splitStringClaimValue(groupsClaim.toString(), authContextInfo));
            }
            if (claimsSet.hasClaim(ROLE_MAPPINGS)) {
                this.mapRoles(claimsSet);
            }
            return jwtContext;
        }
        catch (InvalidJwtException e) {
            PrincipalLogging.log.tokenInvalid();
            throw PrincipalMessages.msg.failedToVerifyToken(e);
        }
        catch (UnresolvableKeyException e) {
            PrincipalLogging.log.verificationKeyUnresolvable();
            throw PrincipalMessages.msg.failedToVerifyToken(e);
        }
    }

    void setExpectedAudience(JwtConsumerBuilder builder, JWTAuthContextInfo authContextInfo) {
        Set<String> expectedAudience = authContextInfo.getExpectedAudience();
        if (expectedAudience != null) {
            builder.setExpectedAudience(expectedAudience.toArray(new String[0]));
        } else {
            builder.setSkipDefaultAudienceValidation();
        }
    }

    private void checkNameClaims(JwtContext jwtContext) throws InvalidJwtException {
        boolean hasPrincipalClaim;
        JwtClaims claimsSet = jwtContext.getJwtClaims();
        boolean bl = hasPrincipalClaim = claimsSet.getClaimValue(Claims.sub.name()) != null || claimsSet.getClaimValue(Claims.upn.name()) != null || claimsSet.getClaimValue(Claims.preferred_username.name()) != null;
        if (!hasPrincipalClaim) {
            throw PrincipalMessages.msg.claimNotFound(s -> new InvalidJwtException(s, Collections.emptyList(), jwtContext));
        }
    }

    private String findSubject(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) {
        if (authContextInfo.getSubjectPath() != null) {
            String[] pathSegments = DefaultJWTTokenParser.splitClaimPath(authContextInfo.getSubjectPath());
            Object claimValue = this.findClaimValue(authContextInfo.getSubjectPath(), claimsSet.getClaimsMap(), pathSegments, 0);
            if (claimValue instanceof String) {
                return (String)claimValue;
            }
            PrincipalLogging.log.claimAtPathIsNotAString(authContextInfo.getSubjectPath());
        }
        if (authContextInfo.getDefaultSubjectClaim() != null) {
            return authContextInfo.getDefaultSubjectClaim();
        }
        return null;
    }

    private List<String> findGroups(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) {
        if (authContextInfo.getGroupsPath() != null) {
            String[] pathSegments = DefaultJWTTokenParser.splitClaimPath(authContextInfo.getGroupsPath());
            Object claimValue = this.findClaimValue(authContextInfo.getGroupsPath(), claimsSet.getClaimsMap(), pathSegments, 0);
            if (claimValue instanceof List) {
                List groups = (List)List.class.cast(claimValue);
                try {
                    return Arrays.asList(groups.toArray(new String[0]));
                }
                catch (ArrayStoreException ex) {
                    PrincipalLogging.log.claimAtPathIsNotAnArrayOfStrings(authContextInfo.getGroupsPath());
                }
            } else {
                if (claimValue instanceof String) {
                    return this.splitStringClaimValue(claimValue.toString(), authContextInfo);
                }
                PrincipalLogging.log.claimAtPathIsNeitherAnArrayOfStringsNorString(authContextInfo.getGroupsPath());
            }
        }
        if (authContextInfo.getDefaultGroupsClaim() != null) {
            return Collections.singletonList(authContextInfo.getDefaultGroupsClaim());
        }
        return null;
    }

    private List<String> splitStringClaimValue(String claimValue, JWTAuthContextInfo authContextInfo) {
        return Arrays.asList(claimValue.split(authContextInfo.getGroupsSeparator()));
    }

    private static String[] splitClaimPath(String claimPath) {
        String[] stringArray;
        if (claimPath.indexOf(47) > 0) {
            stringArray = CLAIM_PATH_PATTERN.split(claimPath);
        } else {
            String[] stringArray2 = new String[1];
            stringArray = stringArray2;
            stringArray2[0] = claimPath;
        }
        return stringArray;
    }

    private void mapRoles(JwtClaims claimsSet) {
        try {
            Map rolesMapping = (Map)claimsSet.getClaimValue(ROLE_MAPPINGS, Map.class);
            List groups = claimsSet.getStringListClaimValue(Claims.groups.name());
            ArrayList<String> allGroups = new ArrayList<String>(groups);
            for (Map.Entry mapping : rolesMapping.entrySet()) {
                if (!groups.contains(mapping.getKey())) continue;
                allGroups.add((String)mapping.getValue());
            }
            claimsSet.setStringListClaim(Claims.groups.name(), allGroups);
            PrincipalLogging.log.updatedGroups(allGroups);
        }
        catch (Exception e) {
            PrincipalLogging.log.failedToAccessRolesMappingClaim(e);
        }
    }

    private Object findClaimValue(String claimPath, Map<String, Object> claimsMap, String[] pathArray, int step) {
        Object claimValue = claimsMap.get(pathArray[step].replace("\"", ""));
        if (claimValue == null) {
            PrincipalLogging.log.claimNotFoundAtPathAtSegment(claimPath, pathArray[step]);
        } else if (step + 1 < pathArray.length) {
            if (claimValue instanceof Map) {
                Map nextMap = (Map)claimValue;
                int nextStep = step + 1;
                return this.findClaimValue(claimPath, nextMap, pathArray, nextStep);
            }
            PrincipalLogging.log.claimValueIsNotAJson(claimPath);
            return null;
        }
        return claimValue;
    }

    private void verifyTimeToLive(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) throws ParseException {
        Long maxTimeToLiveSecs = authContextInfo.getMaxTimeToLiveSecs();
        if (maxTimeToLiveSecs != null) {
            NumericDate exp;
            NumericDate iat;
            try {
                iat = claimsSet.getIssuedAt();
                exp = claimsSet.getExpirationTime();
            }
            catch (Exception e) {
                throw PrincipalMessages.msg.failedToVerifyMaxTTL(e);
            }
            if (exp.getValue() - iat.getValue() > maxTimeToLiveSecs) {
                throw PrincipalMessages.msg.expExceeded(exp, maxTimeToLiveSecs, iat);
            }
        } else {
            PrincipalLogging.log.noMaxTTLSpecified();
        }
    }

    private void verifyRequiredClaims(JWTAuthContextInfo authContextInfo, JwtContext jwtContext) throws InvalidJwtException {
        Set<String> requiredClaims = authContextInfo.getRequiredClaims();
        if (requiredClaims != null && !jwtContext.getJwtClaims().getClaimsMap().keySet().containsAll(requiredClaims)) {
            if (PrincipalLogging.log.isDebugEnabled()) {
                String missingClaims = requiredClaims.stream().filter(claim -> !jwtContext.getJwtClaims().getClaimsMap().containsKey(claim)).collect(Collectors.joining(","));
                PrincipalLogging.log.missingClaims(missingClaims);
            }
            throw PrincipalMessages.msg.missingClaims(s -> new InvalidJwtException(s, Collections.emptyList(), jwtContext));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected VerificationKeyResolver getVerificationKeyResolver(JWTAuthContextInfo authContextInfo) throws UnresolvableKeyException {
        if (this.keyResolver == null) {
            DefaultJWTTokenParser defaultJWTTokenParser = this;
            synchronized (defaultJWTTokenParser) {
                if (this.keyResolver == null) {
                    this.keyResolver = new KeyLocationResolver(authContextInfo);
                }
            }
        }
        return this.keyResolver;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected DecryptionKeyResolver getDecryptionKeyResolver(JWTAuthContextInfo authContextInfo) throws UnresolvableKeyException {
        if (this.decryptionKeyResolver == null) {
            DefaultJWTTokenParser defaultJWTTokenParser = this;
            synchronized (defaultJWTTokenParser) {
                if (this.decryptionKeyResolver == null) {
                    this.decryptionKeyResolver = new DecryptionKeyLocationResolver(authContextInfo);
                }
            }
        }
        return this.decryptionKeyResolver;
    }

    protected ProtectionLevel getProtectionLevel(JWTAuthContextInfo authContextInfo) {
        if (authContextInfo.getDecryptionKeyLocation() != null || authContextInfo.getDecryptionKeyContent() != null || authContextInfo.getPrivateDecryptionKey() != null || authContextInfo.getSecretDecryptionKey() != null) {
            boolean sign = authContextInfo.getPublicVerificationKey() != null || authContextInfo.getSecretVerificationKey() != null || authContextInfo.getPublicKeyContent() != null || authContextInfo.getPublicKeyLocation() != null;
            return sign ? ProtectionLevel.SIGN_ENCRYPT : ProtectionLevel.ENCRYPT;
        }
        return ProtectionLevel.SIGN;
    }

    protected static enum ProtectionLevel {
        SIGN,
        ENCRYPT,
        SIGN_ENCRYPT;

    }
}

