package javax.slee.management;

import javax.management.ObjectName;

/**
 * This class identifies a notification such as an {@link AlarmNotification alarm} or
 * {@link TraceNotification trace} notification as being generated in response to
 * some action performed by a resource adaptor entity.  For example, if a resource
 * adaptor entity raises an alarm using the {@link javax.slee.facilities.AlarmFacility
 * alarm facility}, an alarm notification will be generated containing a
 * <code>ResourceAdaptorEntityNotification</code> object that identifies the resource
 * adaptor entity that raised the alarm.
 * @since SLEE 1.1
 */
public final class ResourceAdaptorEntityNotification extends AbstractNotificationSource implements NotificationSource {
    /**
     * The JMX notification type of alarm notifications that are generated in response
     * to a resource adaptor entity interacting with the {@link javax.slee.facilities.AlarmFacility}.
     * <p>
     * The notification type is equal to the string "javax.slee.management.alarm.raentity".
     */
    public static final String ALARM_NOTIFICATION_TYPE = "javax.slee.management.alarm.raentity";

    /**
     * The JMX notification type of trace notifications that are generated in response
     * to a resource adaptor entity interacting with a {@link javax.slee.facilities.Tracer}
     * object.
     * <p>
     * The notification type is equal to the string "javax.slee.management.trace.raentity".
     */
    public static final String TRACE_NOTIFICATION_TYPE = "javax.slee.management.trace.raentity";

    /**
     * The JMX notification type of usage notifications that are generated by a
     * {@link javax.slee.usage.UsageMBean} containing a <code>ResourceAdaptorEntityNotification</code>
     * as a notification source.
     * <p>
     * The notification type is equal to the string "javax.slee.management.usage.raentity".
     */
    public static final String USAGE_NOTIFICATION_TYPE = "javax.slee.management.usage.raentity";

    /**
     * The JMX Object Name property key that identifies the name of the resource adaptor
     * entity in a Usage MBean whose {@link javax.slee.usage.UsageMBean#NOTIFICATION_SOURCE_KEY}
     * property has a value equal to {@link #USAGE_NOTIFICATION_TYPE}.  This key is
     * equal to the string "raEntityName".
     * @see javax.slee.usage.UsageMBean#BASE_OBJECT_NAME
     * @see javax.slee.usage.UsageMBean#NOTIFICATION_SOURCE_KEY
     * @since SLEE 1.1
     */
    public static final String RESOURCE_ADAPTOR_ENTITY_NAME_KEY = "raEntityName";


    /**
     * Create a new <code>ResourceAdaptorEntityNotification</code> object that uniquely
     * identifies a resource adaptor entity.
     * @param entityName the name of the resource adaptor entity.
     * @throws NullPointerException if <code>entityName</code> is <code>null</code>.
     */
    public ResourceAdaptorEntityNotification(String entityName) {
        if (entityName == null) throw new NullPointerException("entityName is null");
        this.entityName = entityName;
    }

    /**
     * Get the name of the resource adaptor entity of this notification source.
     * @return the name of the resource adaptor entity.
     */
    public String getEntityName() {
        return entityName;
    }

    /**
     * Get the JMX notification type of alarm notifications generated in response
     * to a resource adaptor entity interacting with the Alarm Facility.
     * @return the string defined by {@link #ALARM_NOTIFICATION_TYPE}.
     */
    public String getAlarmNotificationType() {
        return ALARM_NOTIFICATION_TYPE;
    }

    /**
     * Get the JMX notification type of trace notifications generated in response
     * to a resource adaptor entity interacting with the Trace Facility.
     * @return the string defined by {@link #TRACE_NOTIFICATION_TYPE}.
     */
    public String getTraceNotificationType() {
        return TRACE_NOTIFICATION_TYPE;
    }

    /**
     * Get the JMX notification type of usage notifications generated in response
     * to a resource adaptor entity interacting with its usage parameters.
     * @return the string defined by {@link #USAGE_NOTIFICATION_TYPE}.
     */
    public String getUsageNotificationType() {
        return USAGE_NOTIFICATION_TYPE;
    }

    /**
     * Get a JMX Object Name property string that uniquely identifies the specified
     * resource adaptor entity, suitable for inclusion in the Object Name of a Usage
     * MBean.  This method makes use of the {@link ObjectName#quote}</code> method
     * to ensure that the resource adaptor entity name is valid for inclusion as property
     * values in an Object Name.
     * <p>
     * This method can be used as follows to manually construct a complete Object Name
     * for a Usage MBean:
     * <br><ul><code>
     *     ObjectName name = new ObjectName(<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link javax.slee.usage.UsageMBean#BASE_OBJECT_NAME} + "," +<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link javax.slee.usage.UsageMBean#USAGE_PARAMETER_SET_NAME_KEY} + "=" + ObjectName.quote(paramSetName) + "," &nbsp; // optional<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link javax.slee.usage.UsageMBean#NOTIFICATION_SOURCE_KEY} + "=" + {@link #USAGE_NOTIFICATION_TYPE ResourceAdaptorEntityNotification.USAGE_NOTIFICATION_TYPE} + "," +<br>
     *     &nbsp;&nbsp;&nbsp;&nbsp;{@link #getUsageMBeanProperties(String) ResourceAdaptorEntityNotification.getUsageMBeanProperties(entityName)}<br>
     *     );
     * </code></ul>
     * @param entityName the name of the resource adaptor entity.
     * @return an Object Name property string that uniquely identifies the resource
     *        adaptor entity.
     * @throws NullPointerException if <code>entityName</code> is <code>null</code>.
     */
    public static String getUsageMBeanProperties(String entityName) {
        if (entityName == null) throw new NullPointerException("entityName is null");
        return RESOURCE_ADAPTOR_ENTITY_NAME_KEY + '=' + ObjectName.quote(entityName);
    }

    /**
     * Get a JMX Object Name property string that uniquely identifies the resource
     * adaptor entity of this notification source, suitable for inclusion in the Object Name
     * of a Usage MBean.
     * <p>
     * This method is equivalent to {@link #getUsageMBeanProperties(String) getUsageMBeanProperties(getEntityName())}.
     * @return an Object Name property string that uniquely identifies the resource 
     *        adaptor entity of this notification source.
     */
    public String getUsageMBeanProperties() {
        return getUsageMBeanProperties(entityName);
    }

    /**
     * Compare this notification source for equality with another object.
     * @param obj the object to compare this with.
     * @return <code>true</code> if <code>obj</code> is an instance of this class and
     *        contains the same resource adaptor entity name as this, <code>false</code>
     *        otherwise.
     */
    public boolean equals(Object obj) {
        if (obj == this) return true;
        if (!(obj instanceof ResourceAdaptorEntityNotification)) return false;

        return this.entityName.equals(((ResourceAdaptorEntityNotification)obj).entityName);
    }

    /**
     * Get a hash code value for this notification source.
     * @return a hash code value for this notification source.
     */
    public int hashCode() {
        return entityName.hashCode();
    }

    /**
     * Get a string representation for this notification source.
     * @return a string representation for this notification source.
     * @see Object#toString()
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("RAEntityNotification[entity=").append(entityName).append(']');
        return buf.toString();
    }

    /**
     * Compare this notification source with the specified object for order.
     * Returns a negative integer, zero, or a positive integer if this object
     * is less than, equal to, or greater than the specified object.
     * <p>
     * If <code>obj</code> is a <code>ResourceAdaptorEntityNotification</code>,
     * order is determined by comparing the encapsulated resource adaptor entity
     * name.  Otherwise, if <code>obj</code> is a <code>NotificationSource</code>,
     * ordering is determined by comparing the class name of this class with the
     * class name of <code>obj</code>.
     * @param obj the object to compare this with.
     * @return a negative integer, zero, or a positive integer if this notification
     *        source is considered less than, equal to, or greater than the
     *        specified object.
     * @throws ClassCastException if <code>obj</code> does not implement the
     *        {@link NotificationSource} interface.
     * @see Comparable#compareTo(Object)
     */
    public int compareTo(Object obj) {
        // can't compare with null
        if (obj == null) throw new NullPointerException("obj is null");
        if (obj == this) return 0;

        if (obj instanceof ResourceAdaptorEntityNotification) {
            // compare the entity name
            ResourceAdaptorEntityNotification that = (ResourceAdaptorEntityNotification)obj;
            return this.entityName.compareTo(that.entityName);
        }
        else {
            return super.compareTo(TYPE, obj);
        }
    }


    // protected

    protected String getClassName() {
        return TYPE;
    }


    private final String entityName;

    // constant to avoid expensive getClass() invocations at runtime
    private static final String TYPE = ResourceAdaptorEntityNotification.class.getName();
}
