/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Date;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.snowflake.client.core.Constants;
import net.snowflake.client.core.FileUtil;
import net.snowflake.client.core.ObjectMapperFactory;
import net.snowflake.client.jdbc.SnowflakeUtil;
import net.snowflake.client.jdbc.internal.fasterxml.jackson.databind.JsonNode;
import net.snowflake.client.jdbc.internal.fasterxml.jackson.databind.ObjectMapper;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

class FileCacheManager {
    private static final SFLogger logger = SFLoggerFactory.getLogger(FileCacheManager.class);
    private static final ObjectMapper OBJECT_MAPPER = ObjectMapperFactory.getObjectMapper();
    private static final Charset DEFAULT_FILE_ENCODING = StandardCharsets.UTF_8;
    private String cacheDirectorySystemProperty;
    private String cacheDirectoryEnvironmentVariable;
    private String baseCacheFileName;
    private long cacheFileLockExpirationInMilliseconds;
    private File cacheFile;
    private File cacheLockFile;
    private File cacheDir;
    private boolean onlyOwnerPermissions = true;

    private FileCacheManager() {
    }

    static FileCacheManager builder() {
        return new FileCacheManager();
    }

    FileCacheManager setCacheDirectorySystemProperty(String cacheDirectorySystemProperty) {
        this.cacheDirectorySystemProperty = cacheDirectorySystemProperty;
        return this;
    }

    FileCacheManager setCacheDirectoryEnvironmentVariable(String cacheDirectoryEnvironmentVariable) {
        this.cacheDirectoryEnvironmentVariable = cacheDirectoryEnvironmentVariable;
        return this;
    }

    FileCacheManager setBaseCacheFileName(String baseCacheFileName) {
        this.baseCacheFileName = baseCacheFileName;
        return this;
    }

    FileCacheManager setCacheFileLockExpirationInSeconds(long cacheFileLockExpirationInSeconds) {
        this.cacheFileLockExpirationInMilliseconds = cacheFileLockExpirationInSeconds * 1000L;
        return this;
    }

    FileCacheManager setOnlyOwnerPermissions(boolean onlyOwnerPermissions) {
        this.onlyOwnerPermissions = onlyOwnerPermissions;
        return this;
    }

    synchronized String getCacheFilePath() {
        return this.cacheFile.getAbsolutePath();
    }

    synchronized void overrideCacheFile(File newCacheFile) {
        if (!FileUtil.exists(newCacheFile)) {
            logger.debug("Cache file doesn't exist. File: {}", newCacheFile);
        }
        if (this.onlyOwnerPermissions) {
            FileUtil.handleWhenFilePermissionsWiderThanUserOnly(newCacheFile, "Override cache file");
            FileUtil.handleWhenParentDirectoryPermissionsWiderThanUserOnly(newCacheFile, "Override cache file");
        } else {
            FileUtil.logFileUsage(this.cacheFile, "Override cache file", false);
        }
        this.cacheFile = newCacheFile;
        this.cacheDir = newCacheFile.getParentFile();
        this.baseCacheFileName = newCacheFile.getName();
    }

    synchronized FileCacheManager build() {
        String cacheDirPath;
        String string = cacheDirPath = this.cacheDirectorySystemProperty != null ? SnowflakeUtil.systemGetProperty(this.cacheDirectorySystemProperty) : null;
        if (cacheDirPath == null) {
            try {
                cacheDirPath = this.cacheDirectoryEnvironmentVariable != null ? SnowflakeUtil.systemGetEnv(this.cacheDirectoryEnvironmentVariable) : null;
            }
            catch (Throwable ex) {
                logger.debug("Cannot get environment variable for cache directory, skip using cache", false);
                return this;
            }
        }
        this.cacheDir = cacheDirPath != null ? new File(cacheDirPath) : FileCacheManager.getDefaultCacheDir();
        if (this.cacheDir == null) {
            return this;
        }
        if (!this.cacheDir.exists()) {
            try {
                if (!SnowflakeUtil.isWindows() && this.onlyOwnerPermissions) {
                    Files.createDirectories(this.cacheDir.toPath(), PosixFilePermissions.asFileAttribute(Stream.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE).collect(Collectors.toSet())));
                } else {
                    Files.createDirectories(this.cacheDir.toPath(), new FileAttribute[0]);
                }
            }
            catch (IOException e) {
                logger.info("Failed to create the cache directory: {}. Ignored. {}", e.getMessage(), this.cacheDir.getAbsoluteFile());
                return this;
            }
        }
        if (!this.cacheDir.exists()) {
            logger.debug("Cannot create the cache directory {}. Giving up.", this.cacheDir.getAbsolutePath());
            return this;
        }
        logger.debug("Verified Directory {}", this.cacheDir.getAbsolutePath());
        File cacheFileTmp = new File(this.cacheDir, this.baseCacheFileName).getAbsoluteFile();
        try {
            if (!cacheFileTmp.exists()) {
                if (!SnowflakeUtil.isWindows() && this.onlyOwnerPermissions) {
                    Files.createFile(cacheFileTmp.toPath(), PosixFilePermissions.asFileAttribute(Stream.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE).collect(Collectors.toSet())));
                } else {
                    Files.createFile(cacheFileTmp.toPath(), new FileAttribute[0]);
                }
                logger.debug("Successfully created a cache file {}", cacheFileTmp);
            } else {
                logger.debug("Cache file already exists {}", cacheFileTmp);
            }
            FileUtil.logFileUsage(cacheFileTmp, "Cache file creation", false);
            this.cacheFile = cacheFileTmp.getCanonicalFile();
            this.cacheLockFile = new File(this.cacheFile.getParentFile(), this.baseCacheFileName + ".lck");
        }
        catch (IOException | SecurityException ex) {
            logger.info("Failed to touch the cache file: {}. Ignored. {}", ex.getMessage(), cacheFileTmp.getAbsoluteFile());
        }
        return this;
    }

    static File getDefaultCacheDir() {
        String xdgCacheHome;
        if (Constants.getOS() == Constants.OS.LINUX && (xdgCacheHome = FileCacheManager.getXdgCacheHome()) != null) {
            return new File(xdgCacheHome, "snowflake");
        }
        String homeDir = FileCacheManager.getHomeDirProperty();
        if (homeDir == null) {
            return null;
        }
        if (Constants.getOS() == Constants.OS.WINDOWS) {
            return new File(new File(new File(new File(homeDir, "AppData"), "Local"), "Snowflake"), "Caches");
        }
        if (Constants.getOS() == Constants.OS.MAC) {
            return new File(new File(new File(homeDir, "Library"), "Caches"), "Snowflake");
        }
        return new File(new File(homeDir, ".cache"), "snowflake");
    }

    private static String getXdgCacheHome() {
        String xdgCacheHome = SnowflakeUtil.systemGetEnv("XDG_CACHE_HOME");
        if (xdgCacheHome != null && FileUtil.isWritable(xdgCacheHome)) {
            return xdgCacheHome;
        }
        return null;
    }

    private static String getHomeDirProperty() {
        String homeDir = SnowflakeUtil.systemGetProperty("user.home");
        if (homeDir != null && FileUtil.isWritable(homeDir)) {
            return homeDir;
        }
        return null;
    }

    synchronized <T> T withLock(Supplier<T> supplier) {
        if (this.cacheFile == null) {
            logger.error("No cache file assigned", false);
            return null;
        }
        if (this.cacheLockFile == null) {
            logger.error("No cache lock file assigned", false);
            return null;
        }
        if (this.cacheLockFile.exists()) {
            this.deleteCacheLockIfExpired();
        }
        if (!this.tryToLockCacheFile()) {
            logger.debug("Failed to lock the file. Skipping cache operation", false);
            return null;
        }
        try {
            T t2 = supplier.get();
            return t2;
        }
        finally {
            if (!this.unlockCacheFile()) {
                logger.debug("Failed to unlock cache file", false);
            }
        }
    }

    synchronized JsonNode readCacheFile() {
        JsonNode jsonNode;
        if (!FileUtil.exists(this.cacheFile)) {
            logger.debug("Cache file doesn't exist. File: {}", this.cacheFile);
            return null;
        }
        InputStreamReader reader = new InputStreamReader((InputStream)new FileInputStream(this.cacheFile), DEFAULT_FILE_ENCODING);
        try {
            if (this.onlyOwnerPermissions) {
                FileUtil.handleWhenFilePermissionsWiderThanUserOnly(this.cacheFile, "Read cache");
                FileUtil.handleWhenParentDirectoryPermissionsWiderThanUserOnly(this.cacheFile, "Read cache");
                FileUtil.throwWhenOwnerDifferentThanCurrentUser(this.cacheFile, "Read cache");
            } else {
                FileUtil.logFileUsage(this.cacheFile, "Read cache", false);
            }
            jsonNode = OBJECT_MAPPER.readTree(reader);
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((Reader)reader).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ex) {
                logger.debug("Failed to read the cache file. No worry. File: {}, Err: {}", this.cacheFile, ex);
                return null;
            }
        }
        ((Reader)reader).close();
        return jsonNode;
    }

    synchronized void writeCacheFile(JsonNode input) {
        logger.debug("Writing cache file. File: {}", this.cacheFile);
        try {
            if (input == null) {
                return;
            }
            try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(this.cacheFile), DEFAULT_FILE_ENCODING);){
                if (this.onlyOwnerPermissions) {
                    FileUtil.handleWhenFilePermissionsWiderThanUserOnly(this.cacheFile, "Write to cache");
                    FileUtil.handleWhenParentDirectoryPermissionsWiderThanUserOnly(this.cacheFile, "Write to cache");
                } else {
                    FileUtil.logFileUsage(this.cacheFile, "Write to cache", false);
                }
                writer.write(input.toString());
            }
        }
        catch (IOException ex) {
            logger.debug("Failed to write the cache file. File: {}", this.cacheFile);
        }
    }

    synchronized void deleteCacheFile() {
        logger.debug("Deleting cache file. File: {}, lock file: {}", this.cacheFile, this.cacheLockFile);
        if (this.cacheFile == null) {
            return;
        }
        this.unlockCacheFile();
        if (!this.cacheFile.delete()) {
            logger.debug("Failed to delete the file: {}", this.cacheFile);
        }
    }

    private synchronized boolean tryToLockCacheFile() {
        boolean locked = false;
        for (int cnt = 0; cnt < 5 && !(locked = this.lockCacheFile()); ++cnt) {
            try {
                Thread.sleep(10L);
                continue;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
        if (!locked) {
            this.deleteCacheLockIfExpired();
            if (!this.lockCacheFile()) {
                logger.debug("Failed to lock the cache file.", false);
            }
        }
        return locked;
    }

    private synchronized void deleteCacheLockIfExpired() {
        long currentTime = new Date().getTime();
        long lockFileTs = FileCacheManager.fileCreationTime(this.cacheLockFile);
        if (lockFileTs < 0L) {
            logger.debug("Failed to get the timestamp of lock directory", new Object[0]);
        } else if (lockFileTs < currentTime - this.cacheFileLockExpirationInMilliseconds) {
            try {
                if (!this.cacheLockFile.delete()) {
                    logger.debug("Failed to delete the directory. Dir: {}", this.cacheLockFile);
                } else {
                    logger.debug("Deleted expired cache lock directory.", false);
                }
            }
            catch (Exception e) {
                logger.debug("Failed to delete the directory. Dir: {}, Error: {}", this.cacheLockFile, e.getMessage());
            }
        }
    }

    private static synchronized long fileCreationTime(File targetFile) {
        if (!FileUtil.exists(targetFile)) {
            logger.debug("File does not exist. File: {}", targetFile);
            return -1L;
        }
        try {
            Path cacheFileLockPath = Paths.get(targetFile.getAbsolutePath(), new String[0]);
            BasicFileAttributes attr = Files.readAttributes(cacheFileLockPath, BasicFileAttributes.class, new LinkOption[0]);
            return attr.creationTime().toMillis();
        }
        catch (IOException ex) {
            logger.debug("Failed to get creation time. File/Dir: {}, Err: {}", targetFile, ex);
            return -1L;
        }
    }

    private synchronized boolean lockCacheFile() {
        return this.cacheLockFile.mkdirs();
    }

    private synchronized boolean unlockCacheFile() {
        return this.cacheLockFile.delete();
    }
}

