
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.expr.fn.impl;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.exception.OutOfMemoryException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






import org.apache.drill.exec.expr.DrillSimpleFunc;
import org.apache.drill.exec.expr.annotations.FunctionTemplate;
import org.apache.drill.exec.expr.annotations.FunctionTemplate.NullHandling;
import org.apache.drill.exec.expr.annotations.Output;
import org.apache.drill.exec.expr.annotations.Param;
import org.apache.drill.exec.expr.annotations.Workspace;
import org.apache.drill.exec.expr.fn.FunctionGenerationHelper;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.exec.record.RecordBatch;

import io.netty.buffer.ByteBuf;

import java.nio.ByteBuffer;

@SuppressWarnings("unused")
public class Decimal18Functions {

    @FunctionTemplate(name = "add", scope = FunctionTemplate.FunctionScope.DECIMAL_ADD_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18AddFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Output Decimal18Holder result;

        public void setup() {
            outputPrecision = Integer.MIN_VALUE;
        }


        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionAddFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionAddFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }

            result.value = left.value + right.value;
            result.precision = outputPrecision;
            result.scale = outputScale;
        }
    }

    @FunctionTemplate(name = "subtract", scope = FunctionTemplate.FunctionScope.DECIMAL_ADD_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18SubtractFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Output Decimal18Holder result;

        public void setup() {
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionAddFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionAddFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }

            result.value = left.value - right.value;
            result.precision = outputPrecision;
            result.scale = outputScale;
        }
    }
    @FunctionTemplate(name = "multiply", scope = FunctionTemplate.FunctionScope.DECIMAL_MUL_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18MultiplyFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Output Decimal18Holder result;

        public void setup() {
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionMulFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionMulFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.value = left.value * right.value;
            result.precision = outputPrecision;
            result.scale = outputScale;
        }
    }

    @FunctionTemplate(name = "abs", scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18AbsFunction implements DrillSimpleFunc {

        @Param Decimal18Holder in;
        @Output Decimal18Holder out;

        public void setup() {}

        public void eval() {
            out.precision = out.maxPrecision;
            out.scale = in.scale;

            out.value = in.value;

            if (out.value < 0){
                out.value *= -1;
            }
        }
    }

    @FunctionTemplate(name = "exact_divide", scope = FunctionTemplate.FunctionScope.DECIMAL_DIV_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18DivideFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output Decimal18Holder result;
        @Workspace int outputScale;
        @Workspace int outputPrecision;

        public void setup() {
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {

            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionDivideFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionDivideFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.scale = outputScale;
            result.precision = outputPrecision;

            java.math.BigDecimal numerator = new java.math.BigDecimal(java.math.BigInteger.valueOf(left.value), left.scale);
            java.math.BigDecimal denominator = new java.math.BigDecimal(java.math.BigInteger.valueOf(right.value), right.scale);

            java.math.BigDecimal output = numerator.divide(denominator, (int) result.scale, java.math.BigDecimal.ROUND_HALF_UP);

            result.value = output.unscaledValue().longValue();
        }
    }

    @FunctionTemplate(name = "mod", scope = FunctionTemplate.FunctionScope.DECIMAL_MOD_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18ModFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Workspace int outputScale;
        @Workspace int outputPrecision;
        @Output Decimal18Holder result;

        public void setup() {
            outputPrecision = Integer.MIN_VALUE;
        }

        public void eval() {
            if (outputPrecision == Integer.MIN_VALUE) {
                org.apache.drill.common.util.DecimalScalePrecisionModFunction resultScalePrec =
                new org.apache.drill.common.util.DecimalScalePrecisionModFunction((int) left.precision, (int) left.scale, (int) right.precision, (int) right.scale);
                outputScale = resultScalePrec.getOutputScale();
                outputPrecision = resultScalePrec.getOutputPrecision();
            }
            result.precision = outputPrecision;
            result.scale = outputScale;
            java.math.BigDecimal numerator = new java.math.BigDecimal(java.math.BigInteger.valueOf(left.value), left.scale);
            java.math.BigDecimal denominator = new java.math.BigDecimal(java.math.BigInteger.valueOf(right.value), right.scale);

            java.math.BigDecimal output = numerator.remainder(denominator);
            output.setScale(result.scale, java.math.BigDecimal.ROUND_HALF_UP);

            result.value = output.unscaledValue().longValue();
        }
    }

    @FunctionTemplate(name = "sign", scope = FunctionTemplate.FunctionScope.SIMPLE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18SignFunction implements DrillSimpleFunc {

        @Param Decimal18Holder in;
        @Output IntHolder out;

        public void setup() {}

        public void eval() {

            out.value = (in.value < 0) ? -1 : ((in.value > 0) ? 1 : 0);
        }
    }

    @FunctionTemplate(names = {"trunc", "truncate"}, scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18TruncFunction implements DrillSimpleFunc {

        @Param Decimal18Holder in;
        @Output Decimal18Holder out;

        public void setup() {}

        public void eval() {

            out.value =(long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleDivide(in.value, (int) in.scale));
            out.precision = out.maxPrecision;
            out.scale = 0;
        }
    }

    @FunctionTemplate(names = {"trunc", "truncate"}, scope = FunctionTemplate.FunctionScope.DECIMAL_SET_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18TruncateScaleFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param IntHolder right;
        @Output Decimal18Holder out;

        public void setup() {}

        public void eval() {

            out.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleDivide(left.value, (int) (left.scale - right.value)));
            out.precision = out.maxPrecision;
            out.scale = right.value;
        }
    }

    @FunctionTemplate(names = {"ceil", "ceiling"}, scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18CeilFunction implements DrillSimpleFunc {

        @Param Decimal18Holder in;
        @Output Decimal18Holder out;

        public void setup() {

        }

        public void eval() {
          long scaleFactor = (long) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen((int) in.scale));

          // Get the integer part
          long integerPart = in.value / scaleFactor;

          // Get the fractional part, if its non-zero increment the integer part
          long fractionalPart = (long) (in.value % scaleFactor);
          if (fractionalPart != 0 && in.value >= 0) {
            integerPart++;
          }

          out.scale = 0;
          out.value = integerPart;
        }
    }

    @FunctionTemplate(name = "floor", scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18FloorFunction implements DrillSimpleFunc {

        @Param Decimal18Holder in;
        @Output Decimal18Holder out;

        public void setup() {
        }

        public void eval() {

          long scaleFactor = (long) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen((int) in.scale));
          out.scale = 0;
          out.value = (in.value / scaleFactor);

          // For negative values we have to decrement by 1
          if (in.value < 0) {
            long fractionalPart = (long) (in.value % scaleFactor);
            if (fractionalPart != 0) {
              out.value--;
            }
          }
        }
    }

    @FunctionTemplate(name = "round", scope = FunctionTemplate.FunctionScope.DECIMAL_ZERO_SCALE, nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18RoundFunction implements DrillSimpleFunc {

        @Param Decimal18Holder in;
        @Output Decimal18Holder out;

        public void setup() {
        }

        public void eval() {

          long scaleFactor = (long) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen((int) in.scale));
          long extractDigit = scaleFactor / 10;

          out.scale = 0;
          // Assign the integer part to the output
          out.value = in.value / scaleFactor;

          // Get the fractional part
          long fractionalPart = in.value % scaleFactor;
          // Get the first digit to check for rounding
          int digit = Math.abs((int) (fractionalPart / extractDigit));

          if (digit > 4) {
            if (in.value > 0) {
              out.value++;
            } else if (in.value < 0) {
              out.value--;
            }
          }
        }
    }

    @FunctionTemplate(name = "round",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_SET_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18RoundScaleFunction implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param IntHolder right;
        @Output Decimal18Holder out;


        public void setup() {
        }

        public void eval() {

          long scaleFactor = (long) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen((int) left.scale));
          long newScaleFactor = (long) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen((int) right.value));
          long truncScaleFactor = (long) (org.apache.drill.exec.util.DecimalUtility.getPowerOfTen( Math.abs(left.scale - right.value)));
          int truncFactor = (int) (left.scale - right.value);

          // If rounding scale is >= current scale
          if (right.value >= left.scale) {
            out.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.value - left.scale)));
          }
          else {
            out.scale = right.value;
            // Assign the integer part to the output
            out.value = left.value / scaleFactor;

            // Get the fractional part
            long fractionalPart = left.value % scaleFactor;

            // From the entire fractional part extract the digits upto which rounding is needed
            long newFractionalPart = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleDivide(fractionalPart, truncFactor));
            long truncatedFraction = fractionalPart % truncScaleFactor;


            // Get the truncated fractional part and extract the first digit to see if we need to add 1
            int digit = Math.abs((int) org.apache.drill.exec.util.DecimalUtility.adjustScaleDivide(truncatedFraction, truncFactor - 1));

            if (digit > 4) {
              if (left.value > 0) {
                newFractionalPart++;
              } else if (left.value < 0) {
                newFractionalPart--;
              }
            }

            out.value = (out.value * newScaleFactor) + newFractionalPart;
          }
        }
    }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal18VsNullableDecimal18NullHigh implements DrillSimpleFunc {

    @Param NullableDecimal18Holder left;
    @Param NullableDecimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {

        if ( left.isSet == 0 ) {
          if ( right.isSet == 0 ) {
            out.value = 0;
            break outside;
          } else {
            out.value = 1;
            break outside;
          }
        } else if ( right.isSet == 0 ) {
          out.value = -1;
          break outside;
        }



            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }

  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal18VsNullableDecimal18NullLow implements DrillSimpleFunc {

    @Param NullableDecimal18Holder left;
    @Param NullableDecimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {

        if ( left.isSet == 0 ) {
          if ( right.isSet == 0 ) {
            out.value = 0;
            break outside;
          } else {
            out.value = -1;
            break outside;
          }
        } else if ( right.isSet == 0 ) {
          out.value = 1;
          break outside;
        }


            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal18VsDecimal18NullHigh implements DrillSimpleFunc {

    @Param NullableDecimal18Holder left;
    @Param Decimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {

        if ( left.isSet == 0 ) {
          out.value = 1;
          break outside;
        }



            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }

  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareNullableDecimal18VsDecimal18NullLow implements DrillSimpleFunc {

    @Param NullableDecimal18Holder left;
    @Param Decimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {

        if ( left.isSet == 0 ) {
          out.value = -1;
          break outside;
        }


            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal18VsNullableDecimal18NullHigh implements DrillSimpleFunc {

    @Param Decimal18Holder left;
    @Param NullableDecimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {

      if ( right.isSet == 0 ) {
        out.value = -1;
        break outside;
      }



            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }

  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal18VsNullableDecimal18NullLow implements DrillSimpleFunc {

    @Param Decimal18Holder left;
    @Param NullableDecimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {

      if ( right.isSet == 0 ) {
        out.value = 1;
        break outside;
      }


            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }


  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_HIGH,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal18VsDecimal18NullHigh implements DrillSimpleFunc {

    @Param Decimal18Holder left;
    @Param Decimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {




            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }

  @FunctionTemplate(name = FunctionGenerationHelper.COMPARE_TO_NULLS_LOW,
                    scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                    nulls = NullHandling.INTERNAL)
  public static class GCompareDecimal18VsDecimal18NullLow implements DrillSimpleFunc {

    @Param Decimal18Holder left;
    @Param Decimal18Holder right;
    @Output IntHolder out;

    public void setup() {}

    public void eval() {
     outside:
      {



            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
        out.value = (left.value < right.value) ? -1 : (left.value > right.value) ? 1 : 0;
      } // outside
    }
  }


    @FunctionTemplate(name = "less than",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18LessThan implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
            out.value = (left.value < right.value) ? 1 : 0;
        }
    }

    @FunctionTemplate(name = "less than or equal to",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18LessThanEq implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
            out.value = (left.value <= right.value) ? 1 : 0;
        }
    }

    @FunctionTemplate(name = "greater than",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18GreaterThan implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
            out.value = (left.value > right.value) ? 1 : 0;
        }
    }

    @FunctionTemplate(name = "greater than or equal to",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18GreaterThanEq implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
            out.value = (left.value >= right.value) ? 1 : 0;
        }
    }

    @FunctionTemplate(name = "Equal",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18Equal implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
            out.value = (left.value == right.value) ? 1 : 0;
        }
    }


    @FunctionTemplate(name = "not equal",
                      scope = FunctionTemplate.FunctionScope.DECIMAL_MAX_SCALE,
                      nulls = NullHandling.NULL_IF_NULL)
    public static class Decimal18NotEqual implements DrillSimpleFunc {

        @Param Decimal18Holder left;
        @Param Decimal18Holder right;
        @Output BitHolder out;
        public void setup() {}

        public void eval() {

            // Adjust the scale of the two inputs to be the same

            if (left.scale < right.scale) {
                left.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(left.value, (int) (right.scale - left.scale)));
                left.scale = right.scale;
            } else if (right.scale < left.scale) {
                right.value = (long) (org.apache.drill.exec.util.DecimalUtility.adjustScaleMultiply(right.value, (int) (left.scale - right.scale)));
                right.scale = left.scale;
            }
            out.value = (left.value != right.value) ? 1 : 0;
        }
    }
}


