/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.core.fs;

import java.io.IOException;
import java.util.Random;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.FileSystemKind;
import org.apache.flink.core.fs.Path;
import org.apache.flink.util.StringUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public abstract class FileSystemBehaviorTestSuite {
    private static final Random RND = new Random();
    private FileSystem fs;
    private Path basePath;

    public abstract FileSystem getFileSystem() throws Exception;

    public abstract Path getBasePath() throws Exception;

    public abstract FileSystemKind getFileSystemKind();

    @Before
    public void prepare() throws Exception {
        this.fs = this.getFileSystem();
        this.basePath = new Path(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(this.basePath);
    }

    @After
    public void cleanup() throws Exception {
        this.fs.delete(this.basePath, true);
    }

    @Test
    public void testFileSystemKind() {
        Assert.assertEquals((Object)this.getFileSystemKind(), (Object)this.fs.getKind());
    }

    @Test
    public void testPathAndScheme() throws Exception {
        Assert.assertEquals((Object)this.fs.getUri(), (Object)this.getBasePath().getFileSystem().getUri());
        Assert.assertEquals((Object)this.fs.getUri().getScheme(), (Object)this.getBasePath().toUri().getScheme());
    }

    @Test
    public void testHomeAndWorkDir() {
        Assert.assertEquals((Object)this.fs.getUri().getScheme(), (Object)this.fs.getWorkingDirectory().toUri().getScheme());
        Assert.assertEquals((Object)this.fs.getUri().getScheme(), (Object)this.fs.getHomeDirectory().toUri().getScheme());
    }

    @Test
    public void testFileExists() throws IOException {
        Path filePath = this.createRandomFileInDirectory(this.basePath);
        Assert.assertTrue((boolean)this.fs.exists(filePath));
    }

    @Test
    public void testFileDoesNotExist() throws IOException {
        Assert.assertFalse((boolean)this.fs.exists(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName())));
    }

    @Test
    public void testExistingFileDeletion() throws IOException {
        this.testSuccessfulDeletion(this.createRandomFileInDirectory(this.basePath), false);
    }

    @Test
    public void testExistingFileRecursiveDeletion() throws IOException {
        this.testSuccessfulDeletion(this.createRandomFileInDirectory(this.basePath), true);
    }

    @Test
    public void testNotExistingFileDeletion() throws IOException {
        this.testSuccessfulDeletion(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()), false);
    }

    @Test
    public void testNotExistingFileRecursiveDeletion() throws IOException {
        this.testSuccessfulDeletion(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()), true);
    }

    @Test
    public void testExistingEmptyDirectoryDeletion() throws IOException {
        Path path = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(path);
        this.testSuccessfulDeletion(path, false);
    }

    @Test
    public void testExistingEmptyDirectoryRecursiveDeletion() throws IOException {
        Path path = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.fs.mkdirs(path);
        this.testSuccessfulDeletion(path, true);
    }

    private void testSuccessfulDeletion(Path path, boolean recursionEnabled) throws IOException {
        this.fs.delete(path, recursionEnabled);
        Assert.assertFalse((boolean)this.fs.exists(path));
    }

    @Test
    public void testExistingNonEmptyDirectoryDeletion() throws IOException {
        Path directoryPath = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Path filePath = this.createRandomFileInDirectory(directoryPath);
        Assert.assertThrows(IOException.class, () -> this.fs.delete(directoryPath, false));
        Assert.assertTrue((boolean)this.fs.exists(directoryPath));
        Assert.assertTrue((boolean)this.fs.exists(filePath));
    }

    @Test
    public void testExistingNonEmptyDirectoryRecursiveDeletion() throws IOException {
        Path directoryPath = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Path filePath = this.createRandomFileInDirectory(directoryPath);
        this.fs.delete(directoryPath, true);
        Assert.assertFalse((boolean)this.fs.exists(directoryPath));
        Assert.assertFalse((boolean)this.fs.exists(filePath));
    }

    @Test
    public void testExistingNonEmptyDirectoryWithSubDirRecursiveDeletion() throws IOException {
        Path level1SubDirWithFile = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Path fileInLevel1Subdir = this.createRandomFileInDirectory(level1SubDirWithFile);
        Path level2SubDirWithFile = new Path(level1SubDirWithFile, FileSystemBehaviorTestSuite.randomName());
        Path fileInLevel2Subdir = this.createRandomFileInDirectory(level2SubDirWithFile);
        this.testSuccessfulDeletion(level1SubDirWithFile, true);
        Assert.assertFalse((boolean)this.fs.exists(fileInLevel1Subdir));
        Assert.assertFalse((boolean)this.fs.exists(level2SubDirWithFile));
        Assert.assertFalse((boolean)this.fs.exists(fileInLevel2Subdir));
    }

    @Test
    public void testMkdirsReturnsTrueWhenCreatingDirectory() throws Exception {
        Path directory = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        Assert.assertTrue((boolean)this.fs.mkdirs(directory));
        if (this.getFileSystemKind() != FileSystemKind.OBJECT_STORE) {
            Assert.assertTrue((boolean)this.fs.exists(directory));
        }
    }

    @Test
    public void testMkdirsCreatesParentDirectories() throws Exception {
        Path directory = new Path(new Path(new Path(this.basePath, FileSystemBehaviorTestSuite.randomName()), FileSystemBehaviorTestSuite.randomName()), FileSystemBehaviorTestSuite.randomName());
        Assert.assertTrue((boolean)this.fs.mkdirs(directory));
        if (this.getFileSystemKind() != FileSystemKind.OBJECT_STORE) {
            Assert.assertTrue((boolean)this.fs.exists(directory));
        }
    }

    @Test
    public void testMkdirsReturnsTrueForExistingDirectory() throws Exception {
        Path directory = new Path(this.basePath, FileSystemBehaviorTestSuite.randomName());
        this.createRandomFileInDirectory(directory);
        Assert.assertTrue((boolean)this.fs.mkdirs(directory));
    }

    @Test
    public void testMkdirsFailsForExistingFile() throws Exception {
        this.assumeNotObjectStore();
        Path file = new Path(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.createFile(file);
        try {
            this.fs.mkdirs(file);
            Assert.fail((String)"should fail with an IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    public void testMkdirsFailsWithExistingParentFile() throws Exception {
        this.assumeNotObjectStore();
        Path file = new Path(this.getBasePath(), FileSystemBehaviorTestSuite.randomName());
        this.createFile(file);
        Path dirUnderFile = new Path(file, FileSystemBehaviorTestSuite.randomName());
        try {
            this.fs.mkdirs(dirUnderFile);
            Assert.fail((String)"should fail with an IOException");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static String randomName() {
        return StringUtils.getRandomString((Random)RND, (int)16, (int)16, (char)'a', (char)'z');
    }

    private void createFile(Path file) throws IOException {
        try (FSDataOutputStream out = this.fs.create(file, FileSystem.WriteMode.NO_OVERWRITE);){
            out.write(new byte[]{1, 2, 3, 4, 5, 6, 7, 8});
        }
    }

    private Path createRandomFileInDirectory(Path directory) throws IOException {
        this.fs.mkdirs(directory);
        Path filePath = new Path(directory, FileSystemBehaviorTestSuite.randomName());
        this.createFile(filePath);
        return filePath;
    }

    private void assumeNotObjectStore() {
        Assume.assumeTrue((String)"Test does not apply to object stores", (this.getFileSystemKind() != FileSystemKind.OBJECT_STORE ? 1 : 0) != 0);
    }
}

