/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.api.common.io;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.io.DelimitedInputFormat;
import org.apache.flink.api.common.io.FileInputFormat;
import org.apache.flink.api.common.io.statistics.BaseStatistics;
import org.apache.flink.configuration.ConfigConstants;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.core.fs.FileInputSplit;
import org.apache.flink.core.fs.Path;
import org.apache.flink.util.function.FunctionWithException;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class DelimitedInputFormatTest {
    private DelimitedInputFormat<String> format;

    @Before
    public void setup() {
        this.format = new MyTextInputFormat();
        this.format.setFilePath(new Path("file:///some/file/that/will/not/be/read"));
    }

    @After
    public void shutdown() throws Exception {
        if (this.format != null) {
            this.format.close();
        }
    }

    @Test
    public void testConfigure() {
        Configuration cfg = new Configuration();
        cfg.setString("delimited-format.delimiter", "\n");
        this.format.configure(cfg);
        Assert.assertEquals((Object)"\n", (Object)new String(this.format.getDelimiter(), this.format.getCharset()));
        cfg.setString("delimited-format.delimiter", "&-&");
        this.format.configure(cfg);
        Assert.assertEquals((Object)"&-&", (Object)new String(this.format.getDelimiter(), this.format.getCharset()));
    }

    @Test
    public void testSerialization() throws Exception {
        byte[] DELIMITER = new byte[]{1, 2, 3, 4};
        int NUM_LINE_SAMPLES = 7;
        int LINE_LENGTH_LIMIT = 12345;
        int BUFFER_SIZE = 178;
        MyTextInputFormat format = new MyTextInputFormat();
        format.setDelimiter(DELIMITER);
        format.setNumLineSamples(7);
        format.setLineLengthLimit(12345);
        format.setBufferSize(178);
        ByteArrayOutputStream baos = new ByteArrayOutputStream(4096);
        ObjectOutputStream oos = new ObjectOutputStream(baos);
        oos.writeObject((Object)format);
        oos.flush();
        oos.close();
        ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(baos.toByteArray()));
        DelimitedInputFormat deserialized = (DelimitedInputFormat)ois.readObject();
        Assert.assertEquals((long)7L, (long)deserialized.getNumLineSamples());
        Assert.assertEquals((long)12345L, (long)deserialized.getLineLengthLimit());
        Assert.assertEquals((long)178L, (long)deserialized.getBufferSize());
        Assert.assertArrayEquals((byte[])DELIMITER, (byte[])deserialized.getDelimiter());
    }

    @Test
    public void testOpen() throws IOException {
        String myString = "my mocked line 1\nmy mocked line 2\n";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("my mocked line 1\nmy mocked line 2\n");
        int bufferSize = 5;
        this.format.setBufferSize(bufferSize);
        this.format.open(split);
        Assert.assertEquals((long)0L, (long)this.format.splitStart);
        Assert.assertEquals((long)("my mocked line 1\nmy mocked line 2\n".length() - bufferSize), (long)this.format.splitLength);
        Assert.assertEquals((long)bufferSize, (long)this.format.getBufferSize());
    }

    @Test
    public void testReadWithoutTrailingDelimiter() throws IOException {
        String myString = "my key|my val$$$my key2\n$$ctd.$$|my value2";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("my key|my val$$$my key2\n$$ctd.$$|my value2");
        Configuration parameters = new Configuration();
        this.format.configure(parameters);
        this.format.open(split);
        String first = (String)this.format.nextRecord(null);
        String second = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)first);
        Assert.assertNotNull((Object)second);
        Assert.assertEquals((Object)"my key|my val$$$my key2", (Object)first);
        Assert.assertEquals((Object)"$$ctd.$$|my value2", (Object)second);
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
    }

    @Test
    public void testReadWithTrailingDelimiter() throws IOException {
        String myString = "my key|my val$$$my key2\n$$ctd.$$|my value2\n";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("my key|my val$$$my key2\n$$ctd.$$|my value2\n");
        Configuration parameters = new Configuration();
        this.format.configure(parameters);
        this.format.open(split);
        String first = (String)this.format.nextRecord(null);
        String second = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)first);
        Assert.assertNotNull((Object)second);
        Assert.assertEquals((Object)"my key|my val$$$my key2", (Object)first);
        Assert.assertEquals((Object)"$$ctd.$$|my value2", (Object)second);
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
    }

    @Test
    public void testReadCustomDelimiter() throws IOException {
        String myString = "my key|my val$$$my key2\n$$ctd.$$|my value2";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("my key|my val$$$my key2\n$$ctd.$$|my value2");
        Configuration parameters = new Configuration();
        this.format.setDelimiter("$$$");
        this.format.configure(parameters);
        this.format.open(split);
        String first = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)first);
        Assert.assertEquals((Object)"my key|my val", (Object)first);
        String second = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)second);
        Assert.assertEquals((Object)"my key2\n$$ctd.$$|my value2", (Object)second);
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
    }

    @Test
    public void testMultiCharDelimiter() throws IOException {
        String myString = "www112xx1123yyy11123zzzzz1123";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("www112xx1123yyy11123zzzzz1123");
        Configuration parameters = new Configuration();
        this.format.setDelimiter("1123");
        this.format.configure(parameters);
        this.format.open(split);
        String first = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)first);
        Assert.assertEquals((Object)"www112xx", (Object)first);
        String second = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)second);
        Assert.assertEquals((Object)"yyy1", (Object)second);
        String third = (String)this.format.nextRecord(null);
        Assert.assertNotNull((Object)third);
        Assert.assertEquals((Object)"zzzzz", (Object)third);
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
    }

    @Test
    public void testReadCustomDelimiterWithCharset() throws IOException {
        Object[] records = new String[]{"\u020e\u021f\u05c0\u020b\u020f", "Apache", "\nFlink", "\u0000", "\u05c0"};
        String delimiter = "\u05c0\u05c0";
        String fileContent = StringUtils.join((Object[])records, (String)delimiter);
        for (final String charset : new String[]{"UTF-8", "UTF-16BE", "UTF-16LE"}) {
            DelimitedInputFormat<String> format = new DelimitedInputFormat<String>(){

                public String readRecord(String reuse, byte[] bytes, int offset, int numBytes) throws IOException {
                    return new String(bytes, offset, numBytes, charset);
                }
            };
            format.setFilePath("file:///some/file/that/will/not/be/read");
            FileInputSplit split = DelimitedInputFormatTest.createTempFile(fileContent, charset);
            format.setDelimiter(delimiter);
            format.setCharset(charset);
            format.configure(new Configuration());
            format.open(split);
            for (Object record : records) {
                String value = (String)format.nextRecord(null);
                Assert.assertEquals((Object)record, (Object)value);
            }
            Assert.assertNull((Object)format.nextRecord(null));
            Assert.assertTrue((boolean)format.reachedEnd());
        }
    }

    @Test
    public void testReadOverSplitBoundariesUnaligned() throws IOException {
        String myString = "value1\nvalue2\nvalue3";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("value1\nvalue2\nvalue3");
        FileInputSplit split1 = new FileInputSplit(0, split.getPath(), 0L, split.getLength() / 2L, split.getHostnames());
        FileInputSplit split2 = new FileInputSplit(1, split.getPath(), split1.getLength(), split.getLength(), split.getHostnames());
        Configuration parameters = new Configuration();
        this.format.configure(parameters);
        this.format.open(split1);
        Assert.assertEquals((Object)"value1", (Object)this.format.nextRecord(null));
        Assert.assertEquals((Object)"value2", (Object)this.format.nextRecord(null));
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
        this.format.open(split2);
        Assert.assertEquals((Object)"value3", (Object)this.format.nextRecord(null));
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
    }

    @Test
    public void testReadWithBufferSizeIsMultiple() throws IOException {
        String next;
        String myString = "aaaaaaa\nbbbbbbb\nccccccc\nddddddd\n";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("aaaaaaa\nbbbbbbb\nccccccc\nddddddd\n");
        FileInputSplit split1 = new FileInputSplit(0, split.getPath(), 0L, split.getLength() / 2L, split.getHostnames());
        FileInputSplit split2 = new FileInputSplit(1, split.getPath(), split1.getLength(), split.getLength(), split.getHostnames());
        Configuration parameters = new Configuration();
        this.format.setBufferSize(2 * (int)split1.getLength());
        this.format.configure(parameters);
        int count = 0;
        this.format.open(split1);
        while ((next = (String)this.format.nextRecord(null)) != null) {
            Assert.assertEquals((long)7L, (long)next.length());
            ++count;
        }
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
        Assert.assertEquals((long)3L, (long)count);
        this.format.open(split2);
        while ((next = (String)this.format.nextRecord(null)) != null) {
            Assert.assertEquals((long)7L, (long)next.length());
            ++count;
        }
        this.format.close();
        Assert.assertEquals((long)4L, (long)count);
    }

    @Test
    public void testReadExactlyBufferSize() throws IOException {
        String next;
        String myString = "aaaaaaa\nbbbbbbb\nccccccc\nddddddd\n";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("aaaaaaa\nbbbbbbb\nccccccc\nddddddd\n");
        Configuration parameters = new Configuration();
        this.format.setBufferSize((int)split.getLength());
        this.format.configure(parameters);
        this.format.open(split);
        int count = 0;
        while ((next = (String)this.format.nextRecord(null)) != null) {
            Assert.assertEquals((long)7L, (long)next.length());
            ++count;
        }
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
        Assert.assertEquals((long)4L, (long)count);
    }

    @Test
    public void testReadRecordsLargerThanBuffer() throws IOException {
        String next;
        String myString = "aaaaaaaaaaaaaaaaaaaaa\nbbbbbbbbbbbbbbbbbbbbbbbbb\nccccccccccccccccccc\nddddddddddddddddddddddddddddddddddd\n";
        FileInputSplit split = DelimitedInputFormatTest.createTempFile("aaaaaaaaaaaaaaaaaaaaa\nbbbbbbbbbbbbbbbbbbbbbbbbb\nccccccccccccccccccc\nddddddddddddddddddddddddddddddddddd\n");
        FileInputSplit split1 = new FileInputSplit(0, split.getPath(), 0L, split.getLength() / 2L, split.getHostnames());
        FileInputSplit split2 = new FileInputSplit(1, split.getPath(), split1.getLength(), split.getLength(), split.getHostnames());
        Configuration parameters = new Configuration();
        this.format.setBufferSize(8);
        this.format.configure(parameters);
        ArrayList<String> result = new ArrayList<String>();
        this.format.open(split1);
        while ((next = (String)this.format.nextRecord(null)) != null) {
            result.add(next);
        }
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
        this.format.open(split2);
        while ((next = (String)this.format.nextRecord(null)) != null) {
            result.add(next);
        }
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
        Assert.assertEquals((long)4L, (long)result.size());
        Assert.assertEquals(Arrays.asList("aaaaaaaaaaaaaaaaaaaaa\nbbbbbbbbbbbbbbbbbbbbbbbbb\nccccccccccccccccccc\nddddddddddddddddddddddddddddddddddd\n".split("\n")), result);
    }

    @Test
    public void testDelimiterOnBufferBoundary() throws IOException {
        this.testDelimiterOnBufferBoundary((FunctionWithException<String, FileInputSplit, IOException>)((FunctionWithException)fileContent -> DelimitedInputFormatTest.createTempFile(fileContent)));
    }

    @Test
    public void testDelimiterOnBufferBoundaryWithWholeFileSplit() throws IOException {
        this.testDelimiterOnBufferBoundary((FunctionWithException<String, FileInputSplit, IOException>)((FunctionWithException)fileContent -> {
            FileInputSplit split = DelimitedInputFormatTest.createTempFile(fileContent);
            return new FileInputSplit(0, split.getPath(), 0L, -1L, split.getHostnames());
        }));
    }

    private void testDelimiterOnBufferBoundary(FunctionWithException<String, FileInputSplit, IOException> splitCreator) throws IOException {
        Object[] records = new String[]{"1234567890<DEL?NO!>1234567890", "1234567890<DEL?NO!>1234567890", "<DEL?NO!>"};
        String delimiter = "<DELIM>";
        String fileContent = StringUtils.join((Object[])records, (String)delimiter);
        FileInputSplit split = (FileInputSplit)splitCreator.apply((Object)fileContent);
        Configuration parameters = new Configuration();
        this.format.setBufferSize(12);
        this.format.setDelimiter(delimiter);
        this.format.configure(parameters);
        this.format.open(split);
        for (Object record : records) {
            String value = (String)this.format.nextRecord(null);
            Assert.assertEquals((Object)record, (Object)value);
        }
        Assert.assertNull((Object)this.format.nextRecord(null));
        Assert.assertTrue((boolean)this.format.reachedEnd());
        this.format.close();
    }

    @Test
    public void testGetStatistics() throws IOException {
        String myString = "my mocked line 1\nmy mocked line 2\n";
        long size = "my mocked line 1\nmy mocked line 2\n".length();
        Path filePath = DelimitedInputFormatTest.createTempFilePath("my mocked line 1\nmy mocked line 2\n");
        String myString2 = "my mocked line 1\nmy mocked line 2\nanother mocked line3\n";
        long size2 = "my mocked line 1\nmy mocked line 2\nanother mocked line3\n".length();
        Path filePath2 = DelimitedInputFormatTest.createTempFilePath("my mocked line 1\nmy mocked line 2\nanother mocked line3\n");
        long totalSize = size + size2;
        MyTextInputFormat format = new MyTextInputFormat();
        format.setFilePaths(new String[]{filePath.toUri().toString(), filePath2.toUri().toString()});
        FileInputFormat.FileBaseStatistics stats = format.getStatistics(null);
        Assert.assertNotNull((Object)stats);
        Assert.assertEquals((String)"The file size from the statistics is wrong.", (long)totalSize, (long)stats.getTotalInputSize());
    }

    @Test
    public void testGetStatisticsFileDoesNotExist() throws IOException {
        MyTextInputFormat format = new MyTextInputFormat();
        format.setFilePaths(new String[]{"file:///path/does/not/really/exist", "file:///another/path/that/does/not/exist"});
        FileInputFormat.FileBaseStatistics stats = format.getStatistics(null);
        Assert.assertNull((String)"The file statistics should be null.", (Object)stats);
    }

    @Test
    public void testGetStatisticsSingleFileWithCachedVersion() throws IOException {
        String myString = "my mocked line 1\nmy mocked line 2\n";
        Path tempFile = DelimitedInputFormatTest.createTempFilePath("my mocked line 1\nmy mocked line 2\n");
        long size = "my mocked line 1\nmy mocked line 2\n".length();
        long fakeSize = 10065L;
        MyTextInputFormat format = new MyTextInputFormat();
        format.setFilePath(tempFile);
        format.configure(new Configuration());
        FileInputFormat.FileBaseStatistics stats = format.getStatistics(null);
        Assert.assertNotNull((Object)stats);
        Assert.assertEquals((String)"The file size from the statistics is wrong.", (long)size, (long)stats.getTotalInputSize());
        format = new MyTextInputFormat();
        format.setFilePath(tempFile);
        format.configure(new Configuration());
        FileInputFormat.FileBaseStatistics newStats = format.getStatistics((BaseStatistics)stats);
        Assert.assertEquals((String)"Statistics object was changed.", (Object)newStats, (Object)stats);
        format = new MyTextInputFormat();
        format.setFilePath(tempFile);
        format.configure(new Configuration());
        FileInputFormat.FileBaseStatistics fakeStats = new FileInputFormat.FileBaseStatistics(stats.getLastModificationTime(), 10065L, -1.0f);
        FileInputFormat.FileBaseStatistics latest = format.getStatistics((BaseStatistics)fakeStats);
        Assert.assertEquals((String)"The file size from the statistics is wrong.", (long)10065L, (long)latest.getTotalInputSize());
        format = new MyTextInputFormat();
        format.setFilePath(tempFile);
        format.configure(new Configuration());
        FileInputFormat.FileBaseStatistics outDatedFakeStats = new FileInputFormat.FileBaseStatistics(stats.getLastModificationTime() - 1L, 10065L, -1.0f);
        FileInputFormat.FileBaseStatistics reGathered = format.getStatistics((BaseStatistics)outDatedFakeStats);
        Assert.assertEquals((String)"The file size from the statistics is wrong.", (long)size, (long)reGathered.getTotalInputSize());
    }

    static FileInputSplit createTempFile(String contents) throws IOException {
        File tempFile = File.createTempFile("test_contents", "tmp");
        tempFile.deleteOnExit();
        try (OutputStreamWriter out = new OutputStreamWriter(new FileOutputStream(tempFile));){
            out.write(contents);
        }
        return new FileInputSplit(0, new Path(tempFile.toURI().toString()), 0L, tempFile.length(), new String[]{"localhost"});
    }

    static FileInputSplit createTempFile(String contents, String charset) throws IOException {
        File tempFile = File.createTempFile("test_contents", "tmp");
        tempFile.deleteOnExit();
        try (OutputStreamWriter out = new OutputStreamWriter((OutputStream)new FileOutputStream(tempFile), charset);){
            out.write(contents);
        }
        return new FileInputSplit(0, new Path(tempFile.toURI().toString()), 0L, tempFile.length(), new String[]{"localhost"});
    }

    private static Path createTempFilePath(String contents) throws IOException {
        File tempFile = File.createTempFile("test_contents", "tmp");
        tempFile.deleteOnExit();
        try (OutputStreamWriter wrt = new OutputStreamWriter(new FileOutputStream(tempFile));){
            wrt.write(contents);
        }
        return new Path(tempFile.toURI().toString());
    }

    protected static final class MyTextInputFormat
    extends DelimitedInputFormat<String> {
        private static final long serialVersionUID = 1L;

        protected MyTextInputFormat() {
        }

        public String readRecord(String reuse, byte[] bytes, int offset, int numBytes) {
            return new String(bytes, offset, numBytes, ConfigConstants.DEFAULT_CHARSET);
        }

        public boolean supportsMultiPaths() {
            return true;
        }
    }
}

