/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.configuration;

import java.io.File;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.ConfigOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.ConfigurationUtils;
import org.apache.flink.configuration.CoreOptions;
import org.apache.flink.testutils.junit.extensions.parameterized.Parameter;
import org.apache.flink.testutils.junit.extensions.parameterized.ParameterizedTestExtension;
import org.apache.flink.testutils.junit.extensions.parameterized.Parameters;
import org.apache.flink.util.TimeUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestTemplate;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(value={ParameterizedTestExtension.class})
public class ConfigurationUtilsTest {
    @Parameter
    public boolean standardYaml;

    @Parameters(name="standardYaml: {0}")
    public static Collection<Boolean> parameters() {
        return Arrays.asList(true, false);
    }

    @TestTemplate
    void testPropertiesToConfiguration() {
        Properties properties = new Properties();
        int entries = 10;
        for (int i = 0; i < 10; ++i) {
            properties.setProperty("key" + i, "value" + i);
        }
        Configuration configuration = ConfigurationUtils.createConfiguration((Properties)properties);
        for (String key : properties.stringPropertyNames()) {
            Assertions.assertThat((String)configuration.getString(key, "")).isEqualTo(properties.getProperty(key));
        }
        Assertions.assertThat((Map)configuration.toMap()).hasSize(properties.size());
    }

    @Test
    void testStandardYamlSupportLegacyPattern() {
        List<String> expectedList = Arrays.asList("a", "b", "c");
        String legacyListPattern = "a;b;c";
        HashMap<String, String> expectedMap = new HashMap<String, String>();
        expectedMap.put("k1", "v1");
        expectedMap.put("k2", "v2");
        String legacyMapPattern = "k1:v1,k2:v2";
        Configuration configuration = new Configuration(true);
        configuration.setString("listKey", legacyListPattern);
        configuration.setString("mapKey", legacyMapPattern);
        Assertions.assertThat((List)((List)configuration.get(ConfigOptions.key((String)"listKey").stringType().asList().noDefaultValue()))).isEqualTo(expectedList);
        Assertions.assertThat((Map)((Map)configuration.get(ConfigOptions.key((String)"mapKey").mapType().noDefaultValue()))).isEqualTo(expectedMap);
    }

    @TestTemplate
    void testConvertConfigToWritableLinesAndFlattenYaml() {
        this.testConvertConfigToWritableLines(true);
    }

    @TestTemplate
    void testConvertConfigToWritableLinesAndNoFlattenYaml() {
        this.testConvertConfigToWritableLines(false);
    }

    private void testConvertConfigToWritableLines(boolean flattenYaml) {
        Configuration configuration = new Configuration(this.standardYaml);
        ConfigOption nestedListOption = ConfigOptions.key((String)"nested.test-list-key").stringType().asList().noDefaultValue();
        String listValues = "value1;value2;value3";
        String yamlListValues = "[value1, value2, value3]";
        configuration.set(nestedListOption, Arrays.asList("value1;value2;value3".split(";")));
        ConfigOption nestedMapOption = ConfigOptions.key((String)"nested.test-map-key").mapType().noDefaultValue();
        String mapValues = "key1:value1,key2:value2";
        String yamlMapValues = "{key1: value1, key2: value2}";
        configuration.set(nestedMapOption, Arrays.stream("key1:value1,key2:value2".split(",")).collect(Collectors.toMap(e -> e.split(":")[0], e -> e.split(":")[1])));
        ConfigOption nestedDurationOption = ConfigOptions.key((String)"nested.test-duration-key").durationType().noDefaultValue();
        Duration duration = Duration.ofMillis(3000L);
        configuration.set(nestedDurationOption, (Object)duration);
        ConfigOption nestedStringOption = ConfigOptions.key((String)"nested.test-string-key").stringType().noDefaultValue();
        String strValues = "*";
        String yamlStrValues = "'*'";
        configuration.set(nestedStringOption, (Object)"*");
        ConfigOption intOption = ConfigOptions.key((String)"test-int-key").intType().noDefaultValue();
        boolean intValue = true;
        configuration.set(intOption, (Object)1);
        List actualData = ConfigurationUtils.convertConfigToWritableLines((Configuration)configuration, (boolean)flattenYaml);
        List<String> expected = this.standardYaml ? (flattenYaml ? Arrays.asList(nestedListOption.key() + ": " + "[value1, value2, value3]", nestedMapOption.key() + ": " + "{key1: value1, key2: value2}", nestedDurationOption.key() + ": " + TimeUtils.formatWithHighestUnit((Duration)duration), nestedStringOption.key() + ": " + "'*'", intOption.key() + ": " + 1) : Arrays.asList("nested:", "  test-list-key:", "  - value1", "  - value2", "  - value3", "  test-map-key:", "    key1: value1", "    key2: value2", "  test-duration-key: 3 s", "  test-string-key: '*'", "test-int-key: 1")) : Arrays.asList(nestedListOption.key() + ": " + "value1;value2;value3", nestedMapOption.key() + ": " + "key1:value1,key2:value2", nestedDurationOption.key() + ": " + TimeUtils.formatWithHighestUnit((Duration)duration), nestedStringOption.key() + ": " + "*", intOption.key() + ": " + 1);
        Assertions.assertThat(expected).containsExactlyInAnyOrderElementsOf((Iterable)actualData);
    }

    @TestTemplate
    void testHideSensitiveValues() {
        HashMap<String, String> keyValuePairs = new HashMap<String, String>();
        keyValuePairs.put("foobar", "barfoo");
        String secretKey1 = "secret.key";
        keyValuePairs.put("secret.key", "12345");
        String secretKey2 = "my.password";
        keyValuePairs.put("my.password", "12345");
        HashMap<String, String> expectedKeyValuePairs = new HashMap<String, String>(keyValuePairs);
        for (String secretKey : Arrays.asList("secret.key", "my.password")) {
            expectedKeyValuePairs.put(secretKey, "******");
        }
        Map hiddenSensitiveValues = ConfigurationUtils.hideSensitiveValues(keyValuePairs);
        Assertions.assertThat((Map)hiddenSensitiveValues).isEqualTo(expectedKeyValuePairs);
    }

    @TestTemplate
    void testGetPrefixedKeyValuePairs() {
        String prefix = "test.prefix.";
        HashMap<String, String> expectedKeyValuePairs = new HashMap<String, String>(){
            {
                this.put("k1", "v1");
                this.put("k2", "v2");
            }
        };
        Configuration configuration = new Configuration();
        expectedKeyValuePairs.forEach((k, v) -> configuration.setString("test.prefix." + k, v));
        Map resultKeyValuePairs = ConfigurationUtils.getPrefixedKeyValuePairs((String)"test.prefix.", (Configuration)configuration);
        Assertions.assertThat((Map)resultKeyValuePairs).isEqualTo((Object)expectedKeyValuePairs);
    }

    @TestTemplate
    void testConvertToString() {
        Assertions.assertThat((String)ConfigurationUtils.convertToString((Object)"Simple String", (boolean)this.standardYaml)).isEqualTo("Simple String");
        Assertions.assertThat((String)ConfigurationUtils.convertToString((Object)Duration.ZERO, (boolean)this.standardYaml)).isEqualTo("0 ms");
        Assertions.assertThat((String)ConfigurationUtils.convertToString((Object)Duration.ofMillis(123L), (boolean)this.standardYaml)).isEqualTo("123 ms");
        Assertions.assertThat((String)ConfigurationUtils.convertToString((Object)Duration.ofMillis(1234000L), (boolean)this.standardYaml)).isEqualTo("1234 s");
        Assertions.assertThat((String)ConfigurationUtils.convertToString((Object)Duration.ofHours(25L), (boolean)this.standardYaml)).isEqualTo("25 h");
        ArrayList<Object> listElements = new ArrayList<Object>();
        listElements.add("Test;String");
        listElements.add(Duration.ZERO);
        listElements.add(42);
        if (this.standardYaml) {
            Assertions.assertThat((String)"[Test;String, 0 ms, 42]").isEqualTo(ConfigurationUtils.convertToString(listElements, (boolean)true));
        } else {
            Assertions.assertThat((String)"'Test;String';0 ms;42").isEqualTo(ConfigurationUtils.convertToString(listElements, (boolean)false));
        }
        HashMap<Object, Object> mapElements = new HashMap<Object, Object>();
        mapElements.put("A:,B", "C:,D");
        mapElements.put(10, 20);
        if (this.standardYaml) {
            Assertions.assertThat((String)"{'A:,B': 'C:,D', 10: 20}").isEqualTo(ConfigurationUtils.convertToString(mapElements, (boolean)true));
        } else {
            Assertions.assertThat((String)"'''A:,B'':''C:,D''',10:20").isEqualTo(ConfigurationUtils.convertToString(mapElements, (boolean)false));
        }
    }

    @TestTemplate
    void testRandomTempDirectorySelection() {
        Configuration configuration = new Configuration();
        StringBuilder tempDirectories = new StringBuilder();
        int numberTempDirectories = 20;
        for (int i = 0; i < 20; ++i) {
            tempDirectories.append(UUID.randomUUID()).append(',');
        }
        configuration.set(CoreOptions.TMP_DIRS, (Object)tempDirectories.toString());
        Set allTempDirectories = Arrays.stream(ConfigurationUtils.parseTempDirectories((Configuration)configuration)).map(File::new).collect(Collectors.toSet());
        HashSet<File> drawnTempDirectories = new HashSet<File>();
        int numberDraws = 100;
        for (int i = 0; i < 100; ++i) {
            drawnTempDirectories.add(ConfigurationUtils.getRandomTempDirectory((Configuration)configuration));
        }
        Assertions.assertThat(drawnTempDirectories).hasSizeGreaterThan(1);
        Assertions.assertThat(drawnTempDirectories).isSubsetOf(allTempDirectories);
    }
}

