/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.HashSet;
import java.util.Iterator;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.util.NetUtils;
import org.apache.flink.util.TestLogger;
import org.assertj.core.api.Assertions;
import org.hamcrest.Matcher;
import org.hamcrest.core.IsCollectionContaining;
import org.hamcrest.core.IsNot;
import org.junit.Assert;
import org.junit.Test;

public class NetUtilsTest
extends TestLogger {
    @Test
    public void testCorrectHostnamePort() throws Exception {
        URL url = new URL("http", "foo.com", 8080, "/index.html");
        Assert.assertEquals((Object)url, (Object)NetUtils.getCorrectHostnamePort((String)"foo.com:8080/index.html"));
    }

    @Test
    public void testCorrectHostnamePortWithHttpsScheme() throws Exception {
        URL url = new URL("https", "foo.com", 8080, "/some/other/path/index.html");
        Assert.assertEquals((Object)url, (Object)NetUtils.getCorrectHostnamePort((String)"https://foo.com:8080/some/other/path/index.html"));
    }

    @Test
    public void testParseHostPortAddress() {
        InetSocketAddress socketAddress = new InetSocketAddress("foo.com", 8080);
        Assert.assertEquals((Object)socketAddress, (Object)NetUtils.parseHostPortAddress((String)"foo.com:8080"));
    }

    @Test
    public void testAcceptWithoutTimeoutSuppressesTimeoutException() throws IOException {
        final Socket expected = new Socket();
        ServerSocket serverSocket = new ServerSocket(){
            private int count = 0;

            @Override
            public Socket accept() throws IOException {
                if (this.count < 2) {
                    ++this.count;
                    throw new SocketTimeoutException();
                }
                return expected;
            }
        };
        Assert.assertEquals((Object)expected, (Object)NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket));
    }

    @Test
    public void testAcceptWithoutTimeoutDefaultTimeout() throws IOException {
        final Socket expected = new Socket();
        try (ServerSocket serverSocket = new ServerSocket(0){

            @Override
            public Socket accept() {
                return expected;
            }
        };){
            Assert.assertEquals((Object)expected, (Object)NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket));
        }
    }

    @Test
    public void testAcceptWithoutTimeoutZeroTimeout() throws IOException {
        final Socket expected = new Socket();
        try (ServerSocket serverSocket = new ServerSocket(0){

            @Override
            public Socket accept() {
                return expected;
            }
        };){
            serverSocket.setSoTimeout(0);
            Assert.assertEquals((Object)expected, (Object)NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket));
        }
    }

    @Test(expected=IllegalArgumentException.class)
    public void testAcceptWithoutTimeoutRejectsSocketWithSoTimeout() throws IOException {
        try (ServerSocket serverSocket = new ServerSocket(0);){
            serverSocket.setSoTimeout(5);
            NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket);
        }
    }

    @Test
    public void testIPv4toURL() {
        try {
            String addressString = "192.168.0.1";
            InetAddress address = InetAddress.getByName("192.168.0.1");
            Assert.assertEquals((Object)"192.168.0.1", (Object)NetUtils.ipAddressToUrlString((InetAddress)address));
        }
        catch (Exception e) {
            e.printStackTrace();
            Assert.fail((String)e.getMessage());
        }
    }

    @Test
    public void testIPv6toURL() {
        try {
            String addressString = "2001:01db8:00:0:00:ff00:42:8329";
            String normalizedAddress = "[2001:1db8::ff00:42:8329]";
            InetAddress address = InetAddress.getByName("2001:01db8:00:0:00:ff00:42:8329");
            Assert.assertEquals((Object)"[2001:1db8::ff00:42:8329]", (Object)NetUtils.ipAddressToUrlString((InetAddress)address));
        }
        catch (Exception e) {
            e.printStackTrace();
            Assert.fail((String)e.getMessage());
        }
    }

    @Test
    public void testIPv4URLEncoding() {
        try {
            String addressString = "10.244.243.12";
            int port = 23453;
            InetAddress address = InetAddress.getByName("10.244.243.12");
            InetSocketAddress socketAddress = new InetSocketAddress(address, 23453);
            Assert.assertEquals((Object)"10.244.243.12", (Object)NetUtils.ipAddressToUrlString((InetAddress)address));
            Assert.assertEquals((Object)"10.244.243.12:23453", (Object)NetUtils.ipAddressAndPortToUrlString((InetAddress)address, (int)23453));
            Assert.assertEquals((Object)"10.244.243.12:23453", (Object)NetUtils.socketAddressToUrlString((InetSocketAddress)socketAddress));
        }
        catch (Exception e) {
            e.printStackTrace();
            Assert.fail((String)e.getMessage());
        }
    }

    @Test
    public void testIPv6URLEncoding() {
        try {
            String addressString = "2001:db8:10:11:12:ff00:42:8329";
            String bracketedAddressString = "[2001:db8:10:11:12:ff00:42:8329]";
            int port = 23453;
            InetAddress address = InetAddress.getByName("2001:db8:10:11:12:ff00:42:8329");
            InetSocketAddress socketAddress = new InetSocketAddress(address, 23453);
            Assert.assertEquals((Object)"[2001:db8:10:11:12:ff00:42:8329]", (Object)NetUtils.ipAddressToUrlString((InetAddress)address));
            Assert.assertEquals((Object)"[2001:db8:10:11:12:ff00:42:8329]:23453", (Object)NetUtils.ipAddressAndPortToUrlString((InetAddress)address, (int)23453));
            Assert.assertEquals((Object)"[2001:db8:10:11:12:ff00:42:8329]:23453", (Object)NetUtils.socketAddressToUrlString((InetSocketAddress)socketAddress));
        }
        catch (Exception e) {
            e.printStackTrace();
            Assert.fail((String)e.getMessage());
        }
    }

    @Test
    public void testFreePortRangeUtility() {
        String rangeDefinition = "50000-50050, 50100-50200,51234 ";
        Iterator portsIter = NetUtils.getPortRangeFromString((String)rangeDefinition);
        HashSet ports = new HashSet();
        while (portsIter.hasNext()) {
            Assert.assertTrue((String)"Duplicate element", (boolean)ports.add(portsIter.next()));
        }
        Assert.assertEquals((long)153L, (long)ports.size());
        Assert.assertThat(ports, (Matcher)IsCollectionContaining.hasItems((Object[])new Integer[]{50000, 50001, 50002, 50050}));
        Assert.assertThat(ports, (Matcher)IsCollectionContaining.hasItems((Object[])new Integer[]{50100, 50101, 50110, 50200, 51234}));
        Assert.assertThat(ports, (Matcher)IsNot.not((Matcher)IsCollectionContaining.hasItems((Object[])new Integer[]{50051, 50052, 1337, 50201, 49999, 50099})));
        portsIter = NetUtils.getPortRangeFromString((String)" 51234");
        Assert.assertTrue((boolean)portsIter.hasNext());
        Assert.assertEquals((long)51234L, (long)((Integer)portsIter.next()).intValue());
        Assert.assertFalse((boolean)portsIter.hasNext());
        portsIter = NetUtils.getPortRangeFromString((String)"5,1,2,3,4");
        Assert.assertTrue((boolean)portsIter.hasNext());
        Assert.assertEquals((long)5L, (long)((Integer)portsIter.next()).intValue());
        Assert.assertEquals((long)1L, (long)((Integer)portsIter.next()).intValue());
        Assert.assertEquals((long)2L, (long)((Integer)portsIter.next()).intValue());
        Assert.assertEquals((long)3L, (long)((Integer)portsIter.next()).intValue());
        Assert.assertEquals((long)4L, (long)((Integer)portsIter.next()).intValue());
        Assert.assertFalse((boolean)portsIter.hasNext());
        Throwable error = null;
        try {
            NetUtils.getPortRangeFromString((String)"localhost");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof NumberFormatException));
        error = null;
        try {
            NetUtils.getPortRangeFromString((String)"5-");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof NumberFormatException));
        error = null;
        try {
            NetUtils.getPortRangeFromString((String)"-5");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof NumberFormatException));
        error = null;
        try {
            NetUtils.getPortRangeFromString((String)",5");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof NumberFormatException));
        try {
            NetUtils.getPortRangeFromString((String)"70000");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof IllegalConfigurationException));
        try {
            NetUtils.getPortRangeFromString((String)"70000-70001");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof IllegalConfigurationException));
        try {
            NetUtils.getPortRangeFromString((String)"0-70000");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof IllegalConfigurationException));
        try {
            NetUtils.getPortRangeFromString((String)"5-5");
        }
        catch (Throwable t) {
            error = t;
        }
        Assert.assertTrue((boolean)(error instanceof IllegalConfigurationException));
    }

    @Test
    public void testFormatAddress() {
        String host = null;
        int port = 42;
        Assert.assertEquals((Object)("127.0.0.1:" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString(host, (int)port));
        host = "1.2.3.4";
        port = 42;
        Assert.assertEquals((Object)(host + ":" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port));
        host = "2001:0db8:85a3:0000:0000:8a2e:0370:7334";
        port = 42;
        Assert.assertEquals((Object)("[2001:db8:85a3::8a2e:370:7334]:" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port));
        host = "[2001:0db8:85a3:0000:0000:8a2e:0370:7334]";
        port = 42;
        Assert.assertEquals((Object)("[2001:db8:85a3::8a2e:370:7334]:" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port));
        host = "somerandomhostname";
        port = 99;
        Assert.assertEquals((Object)(host + ":" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port));
        host = "  somerandomhostname  ";
        port = 99;
        Assert.assertEquals((Object)(host.trim() + ":" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port));
        host = "illegalhost.";
        port = 42;
        try {
            NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port);
            Assert.fail();
        }
        catch (Exception exception) {
            // empty catch block
        }
        host = "illegalhost:fasf";
        try {
            NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port);
            Assert.fail();
        }
        catch (Exception exception) {
            // empty catch block
        }
        host = "1.2.3.4";
        port = -1;
        try {
            NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port);
            Assert.fail();
        }
        catch (Exception exception) {
            // empty catch block
        }
        host = "CamelCaseHostName";
        port = 99;
        Assert.assertEquals((Object)(host.toLowerCase() + ":" + port), (Object)NetUtils.unresolvedHostAndPortToNormalizedString((String)host, (int)port));
    }

    @Test
    public void testSocketToUrl() throws MalformedURLException {
        InetSocketAddress socketAddress = new InetSocketAddress("foo.com", 8080);
        URL expectedResult = new URL("http://foo.com:8080");
        Assertions.assertThat((URL)NetUtils.socketToUrl((InetSocketAddress)socketAddress)).isEqualTo((Object)expectedResult);
    }

    @Test
    public void testIpv6SocketToUrl() throws MalformedURLException {
        InetSocketAddress socketAddress = new InetSocketAddress("[2001:1db8::ff00:42:8329]", 8080);
        URL expectedResult = new URL("http://[2001:1db8::ff00:42:8329]:8080");
        Assertions.assertThat((URL)NetUtils.socketToUrl((InetSocketAddress)socketAddress)).isEqualTo((Object)expectedResult);
    }

    @Test
    public void testIpv4SocketToUrl() throws MalformedURLException {
        InetSocketAddress socketAddress = new InetSocketAddress("192.168.0.1", 8080);
        URL expectedResult = new URL("http://192.168.0.1:8080");
        Assertions.assertThat((URL)NetUtils.socketToUrl((InetSocketAddress)socketAddress)).isEqualTo((Object)expectedResult);
    }
}

