/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.util;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Iterator;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.util.NetUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

class NetUtilsTest {
    NetUtilsTest() {
    }

    @Test
    void testCorrectHostnamePort() throws Exception {
        URL url = new URL("http", "foo.com", 8080, "/index.html");
        Assertions.assertThat((URL)NetUtils.getCorrectHostnamePort((String)"foo.com:8080/index.html")).isEqualTo((Object)url);
    }

    @Test
    void testCorrectHostnamePortWithHttpsScheme() throws Exception {
        URL url = new URL("https", "foo.com", 8080, "/some/other/path/index.html");
        Assertions.assertThat((URL)NetUtils.getCorrectHostnamePort((String)"https://foo.com:8080/some/other/path/index.html")).isEqualTo((Object)url);
    }

    @Test
    void testParseHostPortAddress() {
        InetSocketAddress socketAddress = new InetSocketAddress("foo.com", 8080);
        Assertions.assertThat((Object)NetUtils.parseHostPortAddress((String)"foo.com:8080")).isEqualTo((Object)socketAddress);
    }

    @Test
    void testAcceptWithoutTimeoutSuppressesTimeoutException() throws IOException {
        final Socket expected = new Socket();
        ServerSocket serverSocket = new ServerSocket(){
            private int count = 0;

            @Override
            public Socket accept() throws IOException {
                if (this.count < 2) {
                    ++this.count;
                    throw new SocketTimeoutException();
                }
                return expected;
            }
        };
        Assertions.assertThat((Object)NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket)).isEqualTo((Object)expected);
    }

    @Test
    void testAcceptWithoutTimeoutDefaultTimeout() throws IOException {
        final Socket expected = new Socket();
        try (ServerSocket serverSocket = new ServerSocket(0){

            @Override
            public Socket accept() {
                return expected;
            }
        };){
            Assertions.assertThat((Object)NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket)).isEqualTo((Object)expected);
        }
    }

    @Test
    void testAcceptWithoutTimeoutZeroTimeout() throws IOException {
        final Socket expected = new Socket();
        try (ServerSocket serverSocket = new ServerSocket(0){

            @Override
            public Socket accept() {
                return expected;
            }
        };){
            serverSocket.setSoTimeout(0);
            Assertions.assertThat((Object)NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket)).isEqualTo((Object)expected);
        }
    }

    @Test
    void testAcceptWithoutTimeoutRejectsSocketWithSoTimeout() {
        Assertions.assertThatThrownBy(() -> {
            try (ServerSocket serverSocket = new ServerSocket(0);){
                serverSocket.setSoTimeout(5);
                NetUtils.acceptWithoutTimeout((ServerSocket)serverSocket);
            }
        }).isInstanceOf(IllegalArgumentException.class);
    }

    @Test
    void testIPv4toURL() throws UnknownHostException {
        String addressString = "192.168.0.1";
        InetAddress address = InetAddress.getByName("192.168.0.1");
        Assertions.assertThat((String)NetUtils.ipAddressToUrlString((InetAddress)address)).isEqualTo("192.168.0.1");
    }

    @Test
    void testIPv6toURL() throws UnknownHostException {
        String addressString = "2001:01db8:00:0:00:ff00:42:8329";
        String normalizedAddress = "[2001:1db8::ff00:42:8329]";
        InetAddress address = InetAddress.getByName("2001:01db8:00:0:00:ff00:42:8329");
        Assertions.assertThat((String)NetUtils.ipAddressToUrlString((InetAddress)address)).isEqualTo("[2001:1db8::ff00:42:8329]");
    }

    @Test
    void testIPv4URLEncoding() throws UnknownHostException {
        String addressString = "10.244.243.12";
        int port = 23453;
        InetAddress address = InetAddress.getByName("10.244.243.12");
        InetSocketAddress socketAddress = new InetSocketAddress(address, 23453);
        Assertions.assertThat((String)NetUtils.ipAddressToUrlString((InetAddress)address)).isEqualTo("10.244.243.12");
        Assertions.assertThat((String)NetUtils.ipAddressAndPortToUrlString((InetAddress)address, (int)23453)).isEqualTo("10.244.243.12:23453");
        Assertions.assertThat((String)NetUtils.socketAddressToUrlString((InetSocketAddress)socketAddress)).isEqualTo("10.244.243.12:23453");
    }

    @Test
    void testIPv6URLEncoding() throws UnknownHostException {
        String addressString = "2001:db8:10:11:12:ff00:42:8329";
        String bracketedAddressString = "[2001:db8:10:11:12:ff00:42:8329]";
        int port = 23453;
        InetAddress address = InetAddress.getByName("2001:db8:10:11:12:ff00:42:8329");
        InetSocketAddress socketAddress = new InetSocketAddress(address, 23453);
        Assertions.assertThat((String)NetUtils.ipAddressToUrlString((InetAddress)address)).isEqualTo("[2001:db8:10:11:12:ff00:42:8329]");
        Assertions.assertThat((String)NetUtils.ipAddressAndPortToUrlString((InetAddress)address, (int)23453)).isEqualTo("[2001:db8:10:11:12:ff00:42:8329]:23453");
        Assertions.assertThat((String)NetUtils.socketAddressToUrlString((InetSocketAddress)socketAddress)).isEqualTo("[2001:db8:10:11:12:ff00:42:8329]:23453");
    }

    @Test
    void testFreePortRangeUtility() {
        String rangeDefinition = "50000-50050, 50100-50200,51234 ";
        Iterator portsIter = NetUtils.getPortRangeFromString((String)rangeDefinition);
        HashSet<Integer> ports = new HashSet<Integer>();
        while (portsIter.hasNext()) {
            Assertions.assertThat((boolean)ports.add((Integer)portsIter.next())).isTrue();
        }
        Assertions.assertThat(ports).hasSize(153);
        Assertions.assertThat(ports).contains((Object[])new Integer[]{50000, 50001, 50002, 50050});
        Assertions.assertThat(ports).contains((Object[])new Integer[]{50100, 50101, 50110, 50200, 51234});
        Assertions.assertThat(ports).doesNotContain((Object[])new Integer[]{50051, 50052, 1337, 50201, 49999, 50099});
        portsIter = NetUtils.getPortRangeFromString((String)" 51234");
        Assertions.assertThat((boolean)portsIter.hasNext()).isTrue();
        Assertions.assertThat((int)((Integer)portsIter.next())).isEqualTo(51234);
        Assertions.assertThat((Iterator)portsIter).isExhausted();
        portsIter = NetUtils.getPortRangeFromString((String)"5,1,2,3,4");
        Assertions.assertThat((boolean)portsIter.hasNext()).isTrue();
        Assertions.assertThat((int)((Integer)portsIter.next())).isEqualTo(5);
        Assertions.assertThat((int)((Integer)portsIter.next())).isOne();
        Assertions.assertThat((int)((Integer)portsIter.next())).isEqualTo(2);
        Assertions.assertThat((int)((Integer)portsIter.next())).isEqualTo(3);
        Assertions.assertThat((int)((Integer)portsIter.next())).isEqualTo(4);
        Assertions.assertThat((Iterator)portsIter).isExhausted();
        Throwable error = null;
        try {
            NetUtils.getPortRangeFromString((String)"localhost");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(NumberFormatException.class);
        error = null;
        try {
            NetUtils.getPortRangeFromString((String)"5-");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(NumberFormatException.class);
        error = null;
        try {
            NetUtils.getPortRangeFromString((String)"-5");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(NumberFormatException.class);
        error = null;
        try {
            NetUtils.getPortRangeFromString((String)",5");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(NumberFormatException.class);
        try {
            NetUtils.getPortRangeFromString((String)"70000");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(IllegalConfigurationException.class);
        try {
            NetUtils.getPortRangeFromString((String)"70000-70001");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(IllegalConfigurationException.class);
        try {
            NetUtils.getPortRangeFromString((String)"0-70000");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(IllegalConfigurationException.class);
        try {
            NetUtils.getPortRangeFromString((String)"5-5");
        }
        catch (Throwable t) {
            error = t;
        }
        Assertions.assertThat((Throwable)error).isInstanceOf(IllegalConfigurationException.class);
    }

    @Test
    void testFormatAddress() {
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString(null, (int)42)).isEqualTo("127.0.0.1:42");
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString((String)"1.2.3.4", (int)42)).isEqualTo("1.2.3.4:42");
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString((String)"2001:0db8:85a3:0000:0000:8a2e:0370:7334", (int)42)).isEqualTo("[2001:db8:85a3::8a2e:370:7334]:42");
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString((String)"[2001:0db8:85a3:0000:0000:8a2e:0370:7334]", (int)42)).isEqualTo("[2001:db8:85a3::8a2e:370:7334]:42");
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString((String)"somerandomhostname", (int)99)).isEqualTo("somerandomhostname:99");
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString((String)"  somerandomhostname  ", (int)99)).isEqualTo("somerandomhostname:99");
        Assertions.assertThatThrownBy(() -> NetUtils.unresolvedHostAndPortToNormalizedString((String)"illegalhost.", (int)42)).isInstanceOf(IllegalConfigurationException.class);
        Assertions.assertThatThrownBy(() -> NetUtils.unresolvedHostAndPortToNormalizedString((String)"illegalhost:fasf", (int)42)).isInstanceOf(IllegalConfigurationException.class);
        Assertions.assertThatThrownBy(() -> NetUtils.unresolvedHostAndPortToNormalizedString((String)"1.2.3.4", (int)-1)).isInstanceOf(Exception.class);
        Assertions.assertThat((String)NetUtils.unresolvedHostAndPortToNormalizedString((String)"CamelCaseHostName", (int)99)).isEqualTo("camelcasehostname:99");
    }

    @Test
    void testSocketToUrl() throws MalformedURLException {
        InetSocketAddress socketAddress = new InetSocketAddress("foo.com", 8080);
        URL expectedResult = new URL("http://foo.com:8080");
        Assertions.assertThat((URL)NetUtils.socketToUrl((InetSocketAddress)socketAddress)).isEqualTo((Object)expectedResult);
    }

    @Test
    void testIpv6SocketToUrl() throws MalformedURLException {
        InetSocketAddress socketAddress = new InetSocketAddress("[2001:1db8::ff00:42:8329]", 8080);
        URL expectedResult = new URL("http://[2001:1db8::ff00:42:8329]:8080");
        Assertions.assertThat((URL)NetUtils.socketToUrl((InetSocketAddress)socketAddress)).isEqualTo((Object)expectedResult);
    }

    @Test
    void testIpv4SocketToUrl() throws MalformedURLException {
        InetSocketAddress socketAddress = new InetSocketAddress("192.168.0.1", 8080);
        URL expectedResult = new URL("http://192.168.0.1:8080");
        Assertions.assertThat((URL)NetUtils.socketToUrl((InetSocketAddress)socketAddress)).isEqualTo((Object)expectedResult);
    }
}

