/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.streaming.runtime.tasks.mailbox;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.flink.core.testutils.CheckedThread;
import org.apache.flink.streaming.runtime.tasks.mailbox.Mail;
import org.apache.flink.streaming.runtime.tasks.mailbox.TaskMailbox;
import org.apache.flink.streaming.runtime.tasks.mailbox.TaskMailboxImpl;
import org.apache.flink.util.function.FunctionWithException;
import org.apache.flink.util.function.RunnableWithException;
import org.apache.flink.util.function.ThrowingRunnable;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class TaskMailboxImplTest {
    private static final RunnableWithException NO_OP = () -> {};
    private static final int DEFAULT_PRIORITY = 0;
    private TaskMailbox taskMailbox;

    TaskMailboxImplTest() {
    }

    @BeforeEach
    void setUp() {
        this.taskMailbox = new TaskMailboxImpl();
    }

    @AfterEach
    void tearDown() {
        this.taskMailbox.close();
    }

    @Test
    void testPutAsHead() throws InterruptedException {
        Mail mailA = new Mail(() -> {}, Integer.MAX_VALUE, "mailA", new Object[0]);
        Mail mailB = new Mail(() -> {}, Integer.MAX_VALUE, "mailB", new Object[0]);
        Mail mailC = new Mail(() -> {}, 0, "mailC, DEFAULT_PRIORITY", new Object[0]);
        Mail mailD = new Mail(() -> {}, 0, "mailD, DEFAULT_PRIORITY", new Object[0]);
        this.taskMailbox.put(mailC);
        this.taskMailbox.putFirst(mailB);
        this.taskMailbox.put(mailD);
        this.taskMailbox.putFirst(mailA);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailA);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailB);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailC);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailD);
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(0)).isNotPresent();
    }

    @Test
    void testContracts() throws InterruptedException {
        LinkedList<Mail> testObjects = new LinkedList<Mail>();
        Assertions.assertThat((boolean)this.taskMailbox.hasMail()).isFalse();
        for (int i = 0; i < 10; ++i) {
            Mail mail = new Mail((ThrowingRunnable)NO_OP, 0, "mail, DEFAULT_PRIORITY", new Object[0]);
            testObjects.add(mail);
            this.taskMailbox.put(mail);
            Assertions.assertThat((boolean)this.taskMailbox.hasMail()).isTrue();
        }
        while (!testObjects.isEmpty()) {
            Assertions.assertThat((Object)this.taskMailbox.take(0)).isEqualTo(testObjects.remove());
            Assertions.assertThat((boolean)this.taskMailbox.hasMail()).isEqualTo(!testObjects.isEmpty());
        }
    }

    @Test
    void testConcurrentPutTakeBlocking() throws Exception {
        this.testPutTake((FunctionWithException<TaskMailbox, Mail, InterruptedException>)((FunctionWithException)mailbox -> mailbox.take(0)));
    }

    @Test
    void testConcurrentPutTakeNonBlockingAndWait() throws Exception {
        this.testPutTake((FunctionWithException<TaskMailbox, Mail, InterruptedException>)((FunctionWithException)mailbox -> {
            Optional optionalMail = mailbox.tryTake(0);
            while (!optionalMail.isPresent()) {
                optionalMail = mailbox.tryTake(0);
            }
            return (Mail)optionalMail.get();
        }));
    }

    @Test
    void testCloseUnblocks() throws InterruptedException {
        this.testAllPuttingUnblocksInternal(TaskMailbox::close);
    }

    @Test
    void testQuiesceUnblocks() throws InterruptedException {
        this.testAllPuttingUnblocksInternal(TaskMailbox::quiesce);
    }

    @Test
    void testLifeCycleQuiesce() throws InterruptedException {
        this.taskMailbox.put(new Mail((ThrowingRunnable)NO_OP, 0, "NO_OP, DEFAULT_PRIORITY", new Object[0]));
        this.taskMailbox.put(new Mail((ThrowingRunnable)NO_OP, 0, "NO_OP, DEFAULT_PRIORITY", new Object[0]));
        this.taskMailbox.quiesce();
        this.testLifecyclePuttingInternal();
        this.taskMailbox.take(0);
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(0)).isPresent();
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(0)).isNotPresent();
    }

    @Test
    void testLifeCycleClose() {
        this.taskMailbox.close();
        this.testLifecyclePuttingInternal();
        Assertions.assertThatThrownBy(() -> this.taskMailbox.take(0)).isInstanceOf(TaskMailbox.MailboxClosedException.class);
        Assertions.assertThatThrownBy(() -> this.taskMailbox.tryTake(0)).isInstanceOf(TaskMailbox.MailboxClosedException.class);
    }

    private void testLifecyclePuttingInternal() {
        Assertions.assertThatThrownBy(() -> this.taskMailbox.put(new Mail((ThrowingRunnable)NO_OP, 0, "NO_OP, DEFAULT_PRIORITY", new Object[0]))).isInstanceOf(TaskMailbox.MailboxClosedException.class);
        Assertions.assertThatThrownBy(() -> this.taskMailbox.putFirst(new Mail((ThrowingRunnable)NO_OP, Integer.MAX_VALUE, "NO_OP", new Object[0]))).isInstanceOf(TaskMailbox.MailboxClosedException.class);
    }

    private void testAllPuttingUnblocksInternal(Consumer<TaskMailbox> unblockMethod) throws InterruptedException {
        this.testUnblocksInternal(() -> this.taskMailbox.put(new Mail((ThrowingRunnable)NO_OP, 0, "NO_OP, DEFAULT_PRIORITY", new Object[0])), unblockMethod);
        this.setUp();
        this.testUnblocksInternal(() -> this.taskMailbox.putFirst(new Mail((ThrowingRunnable)NO_OP, Integer.MAX_VALUE, "NO_OP", new Object[0])), unblockMethod);
    }

    private void testUnblocksInternal(final RunnableWithException testMethod, Consumer<TaskMailbox> unblockMethod) throws InterruptedException {
        CheckedThread[] blockedThreads = new CheckedThread[8];
        final CountDownLatch countDownLatch = new CountDownLatch(blockedThreads.length);
        for (int i = 0; i < blockedThreads.length; ++i) {
            CheckedThread blocked;
            blockedThreads[i] = blocked = new CheckedThread(){

                public void go() throws Exception {
                    countDownLatch.countDown();
                    while (true) {
                        testMethod.run();
                    }
                }
            };
            blocked.start();
        }
        countDownLatch.await();
        unblockMethod.accept(this.taskMailbox);
        for (CheckedThread blockedThread : blockedThreads) {
            Assertions.assertThatThrownBy(() -> ((CheckedThread)blockedThread).sync()).isInstanceOf(TaskMailbox.MailboxClosedException.class);
        }
    }

    private void testPutTake(FunctionWithException<TaskMailbox, Mail, InterruptedException> takeMethod) throws Exception {
        int numThreads = 10;
        int numMailsPerThread = 1000;
        int[] results = new int[10];
        Thread[] writerThreads = new Thread[10];
        for (int i = 0; i < writerThreads.length; ++i) {
            int threadId = i;
            writerThreads[i] = new Thread(ThrowingRunnable.unchecked(() -> {
                for (int k = 0; k < 1000; ++k) {
                    this.taskMailbox.put(new Mail(() -> {
                        int n = threadId;
                        results[n] = results[n] + 1;
                    }, 0, "result " + k, new Object[0]));
                }
            }));
        }
        for (Thread writerThread : writerThreads) {
            writerThread.start();
        }
        for (Thread writerThread : writerThreads) {
            writerThread.join();
        }
        AtomicBoolean isRunning = new AtomicBoolean(true);
        this.taskMailbox.put(new Mail(() -> isRunning.set(false), 0, "POISON_MAIL, DEFAULT_PRIORITY", new Object[0]));
        while (isRunning.get()) {
            ((Mail)takeMethod.apply((Object)this.taskMailbox)).run();
        }
        for (int perThreadResult : results) {
            Assertions.assertThat((int)perThreadResult).isEqualTo(1000);
        }
    }

    @Test
    void testPutAsHeadWithPriority() throws InterruptedException {
        Mail mailA = new Mail(() -> {}, 2, "mailA", new Object[0]);
        Mail mailB = new Mail(() -> {}, 2, "mailB", new Object[0]);
        Mail mailC = new Mail(() -> {}, 1, "mailC", new Object[0]);
        Mail mailD = new Mail(() -> {}, 1, "mailD", new Object[0]);
        this.taskMailbox.put(mailC);
        this.taskMailbox.put(mailB);
        this.taskMailbox.put(mailD);
        this.taskMailbox.putFirst(mailA);
        Assertions.assertThat((Object)this.taskMailbox.take(2)).isSameAs((Object)mailA);
        Assertions.assertThat((Object)this.taskMailbox.take(2)).isSameAs((Object)mailB);
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(2)).isNotPresent();
        Assertions.assertThat((Object)this.taskMailbox.take(1)).isSameAs((Object)mailC);
        Assertions.assertThat((Object)this.taskMailbox.take(1)).isSameAs((Object)mailD);
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(1)).isNotPresent();
    }

    @Test
    void testPutWithPriorityAndReadingFromMainMailbox() throws InterruptedException {
        Mail mailA = new Mail(() -> {}, 2, "mailA", new Object[0]);
        Mail mailB = new Mail(() -> {}, 2, "mailB", new Object[0]);
        Mail mailC = new Mail(() -> {}, 1, "mailC", new Object[0]);
        Mail mailD = new Mail(() -> {}, 1, "mailD", new Object[0]);
        this.taskMailbox.put(mailC);
        this.taskMailbox.put(mailB);
        this.taskMailbox.put(mailD);
        this.taskMailbox.putFirst(mailA);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailA);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailC);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailB);
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isSameAs((Object)mailD);
    }

    @Test
    void testBatchAndNonBatchTake() throws InterruptedException {
        List mails = IntStream.range(0, 6).mapToObj(i -> new Mail((ThrowingRunnable)NO_OP, 0, String.valueOf(i), new Object[0])).collect(Collectors.toList());
        mails.subList(0, 3).forEach(arg_0 -> ((TaskMailbox)this.taskMailbox).put(arg_0));
        Assertions.assertThat((boolean)this.taskMailbox.createBatch()).isTrue();
        mails.subList(3, 6).forEach(arg_0 -> ((TaskMailbox)this.taskMailbox).put(arg_0));
        Assertions.assertThat((Optional)this.taskMailbox.tryTakeFromBatch()).hasValue(mails.get(0));
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(0)).hasValue(mails.get(1));
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isEqualTo(mails.get(2));
        Assertions.assertThat((Optional)this.taskMailbox.tryTakeFromBatch()).isEmpty();
        Assertions.assertThat((Optional)this.taskMailbox.tryTake(0)).hasValue(mails.get(3));
        Assertions.assertThat((Object)this.taskMailbox.take(0)).isEqualTo(mails.get(4));
        Assertions.assertThat((List)this.taskMailbox.close()).containsExactly((Object[])new Mail[]{(Mail)mails.get(5)});
    }

    @Test
    void testBatchDrain() {
        Mail mailA = new Mail(() -> {}, Integer.MAX_VALUE, "mailA", new Object[0]);
        Mail mailB = new Mail(() -> {}, Integer.MAX_VALUE, "mailB", new Object[0]);
        this.taskMailbox.put(mailA);
        Assertions.assertThat((boolean)this.taskMailbox.createBatch()).isTrue();
        this.taskMailbox.put(mailB);
        Assertions.assertThat((List)this.taskMailbox.drain()).containsExactly((Object[])new Mail[]{mailA, mailB});
    }

    @Test
    void testBatchPriority() throws Exception {
        Mail mailA = new Mail(() -> {}, 1, "mailA", new Object[0]);
        Mail mailB = new Mail(() -> {}, 2, "mailB", new Object[0]);
        this.taskMailbox.put(mailA);
        Assertions.assertThat((boolean)this.taskMailbox.createBatch()).isTrue();
        this.taskMailbox.put(mailB);
        Assertions.assertThat((Object)this.taskMailbox.take(2)).isEqualTo((Object)mailB);
        Assertions.assertThat((Optional)this.taskMailbox.tryTakeFromBatch()).hasValue((Object)mailA);
    }

    @Test
    void testRunExclusively() throws InterruptedException {
        CountDownLatch exclusiveCodeStarted = new CountDownLatch(1);
        int numMails = 10;
        new Thread(() -> this.taskMailbox.runExclusively(() -> {
            exclusiveCodeStarted.countDown();
            for (int index = 0; index < 10; ++index) {
                try {
                    this.taskMailbox.put(new Mail(() -> {}, 1, "mailD", new Object[0]));
                    Thread.sleep(1L);
                    continue;
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        })).start();
        exclusiveCodeStarted.await();
        Assertions.assertThat((List)this.taskMailbox.close()).hasSize(10);
    }
}

