/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import java.io.IOException;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.master.cleaner.BaseHFileCleanerDelegate;
import org.apache.hadoop.hbase.master.cleaner.CleanerChore;
import org.apache.hadoop.hbase.testclassification.SmallTests;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.StoppableImplementation;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;
import org.mockito.verification.VerificationMode;

@Category(value={SmallTests.class})
public class TestCleanerChore {
    private static final Log LOG = LogFactory.getLog(TestCleanerChore.class);
    private static final HBaseTestingUtility UTIL = new HBaseTestingUtility();

    @BeforeClass
    public static void setup() {
        CleanerChore.initChorePool((Configuration)UTIL.getConfiguration());
    }

    @AfterClass
    public static void cleanup() throws Exception {
        UTIL.cleanupTestDir();
        CleanerChore.shutDownChorePool();
    }

    @Test
    public void testSavesFilesOnRequest() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, NeverDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        Path parent = new Path(testDir, "parent");
        Path file = new Path(parent, "someFile");
        fs.mkdirs(parent);
        fs.create(file).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        chore.chore();
        Assert.assertTrue((String)"File shouldn't have been deleted", (boolean)fs.exists(file));
        Assert.assertTrue((String)"directory shouldn't have been deleted", (boolean)fs.exists(parent));
    }

    @Test
    public void retriesIOExceptionInStatus() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        Path child = new Path(testDir, "child");
        Path file = new Path(child, "file");
        fs.mkdirs(child);
        fs.create(file).close();
        Assert.assertTrue((String)"test file didn't get created.", (boolean)fs.exists(file));
        final AtomicBoolean fails = new AtomicBoolean(true);
        FilterFileSystem filtered = new FilterFileSystem(fs){

            public FileStatus[] listStatus(Path f) throws IOException {
                if (fails.get()) {
                    throw new IOException("whomp whomp.");
                }
                return this.fs.listStatus(f);
            }
        };
        AllValidPaths chore = new AllValidPaths("test-retry-ioe", stop, conf, (FileSystem)filtered, testDir, confKey);
        Boolean result = chore.runCleaner();
        Assert.assertTrue((String)"test rig failed to inject failure.", (boolean)fs.exists(file));
        Assert.assertTrue((String)"test rig failed to inject failure.", (boolean)fs.exists(child));
        Assert.assertFalse((String)"chore should report that it failed.", (boolean)result);
        fails.set(false);
        result = chore.runCleaner();
        Assert.assertFalse((String)"file should have been destroyed.", (boolean)fs.exists(file));
        Assert.assertFalse((String)"directory should have been destroyed.", (boolean)fs.exists(child));
        Assert.assertTrue((String)"chore should claim it succeeded.", (boolean)result);
    }

    @Test
    public void testDeletesEmptyDirectories() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        Path parent = new Path(testDir, "parent");
        Path child = new Path(parent, "child");
        Path emptyChild = new Path(parent, "emptyChild");
        Path file = new Path(child, "someFile");
        fs.mkdirs(child);
        fs.mkdirs(emptyChild);
        fs.create(file).close();
        Path topFile = new Path(testDir, "topFile");
        fs.create(topFile).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(topFile));
        chore.chore();
        Assert.assertFalse((String)"File didn't get deleted", (boolean)fs.exists(topFile));
        Assert.assertFalse((String)"File didn't get deleted", (boolean)fs.exists(file));
        Assert.assertFalse((String)"Empty directory didn't get deleted", (boolean)fs.exists(child));
        Assert.assertFalse((String)"Empty directory didn't get deleted", (boolean)fs.exists(parent));
    }

    @Test
    public void testDoesNotCheckDirectories() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        AlwaysDelete delegate = (AlwaysDelete)((Object)chore.cleanersChain.get(0));
        AlwaysDelete spy = (AlwaysDelete)((Object)Mockito.spy((Object)((Object)delegate)));
        chore.cleanersChain.set(0, spy);
        Path parent = new Path(testDir, "parent");
        Path file = new Path(parent, "someFile");
        fs.mkdirs(parent);
        Assert.assertTrue((String)"Test parent didn't get created.", (boolean)fs.exists(parent));
        fs.create(file).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        FileStatus fStat = fs.getFileStatus(parent);
        chore.chore();
        ((AlwaysDelete)((Object)Mockito.verify((Object)((Object)spy), (VerificationMode)Mockito.never()))).isFileDeletable(fStat);
        Mockito.reset((Object[])new AlwaysDelete[]{spy});
    }

    @Test
    public void testStoppedCleanerDoesNotDeleteFiles() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        Path topFile = new Path(testDir, "topFile");
        fs.create(topFile).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(topFile));
        stop.stop("testing stop");
        chore.chore();
        Assert.assertTrue((String)"File got deleted while chore was stopped", (boolean)fs.exists(topFile));
    }

    @Test
    public void testCleanerDoesNotDeleteDirectoryWithLateAddedFiles() throws IOException {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        final Path testDir = UTIL.getDataTestDir();
        final FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        AlwaysDelete delegate = (AlwaysDelete)((Object)chore.cleanersChain.get(0));
        AlwaysDelete spy = (AlwaysDelete)((Object)Mockito.spy((Object)((Object)delegate)));
        chore.cleanersChain.set(0, spy);
        Path parent = new Path(testDir, "parent");
        Path file = new Path(parent, "someFile");
        fs.mkdirs(parent);
        fs.create(file).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        final Path addedFile = new Path(parent, "addedFile");
        ((AlwaysDelete)((Object)Mockito.doAnswer((Answer)new Answer<Boolean>(){

            public Boolean answer(InvocationOnMock invocation) throws Throwable {
                fs.create(addedFile).close();
                FSUtils.logFileSystemState((FileSystem)fs, (Path)testDir, (Log)LOG);
                return (Boolean)invocation.callRealMethod();
            }
        }).when((Object)spy))).isFileDeletable((FileStatus)Mockito.any(FileStatus.class));
        chore.chore();
        Assert.assertTrue((String)"Added file unexpectedly deleted", (boolean)fs.exists(addedFile));
        Assert.assertTrue((String)"Parent directory deleted unexpectedly", (boolean)fs.exists(parent));
        Assert.assertFalse((String)"Original file unexpectedly retained", (boolean)fs.exists(file));
        ((AlwaysDelete)((Object)Mockito.verify((Object)((Object)spy), (VerificationMode)Mockito.times((int)1)))).isFileDeletable((FileStatus)Mockito.any(FileStatus.class));
        Mockito.reset((Object[])new AlwaysDelete[]{spy});
    }

    @Test
    public void testNoExceptionFromDirectoryWithRacyChildren() throws Exception {
        UTIL.cleanupTestDir();
        StoppableImplementation stop = new StoppableImplementation();
        HBaseTestingUtility localUtil = new HBaseTestingUtility();
        Configuration conf = localUtil.getConfiguration();
        final Path testDir = UTIL.getDataTestDir();
        final FileSystem fs = UTIL.getTestFileSystem();
        LOG.debug((Object)("Writing test data to: " + testDir));
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        AlwaysDelete delegate = (AlwaysDelete)((Object)chore.cleanersChain.get(0));
        AlwaysDelete spy = (AlwaysDelete)((Object)Mockito.spy((Object)((Object)delegate)));
        chore.cleanersChain.set(0, spy);
        Path parent = new Path(testDir, "parent");
        Path file = new Path(parent, "someFile");
        fs.mkdirs(parent);
        fs.create(file).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        final Path racyFile = new Path(parent, "addedFile");
        ((AlwaysDelete)((Object)Mockito.doAnswer((Answer)new Answer<Boolean>(){

            public Boolean answer(InvocationOnMock invocation) throws Throwable {
                fs.create(racyFile).close();
                FSUtils.logFileSystemState((FileSystem)fs, (Path)testDir, (Log)LOG);
                return (Boolean)invocation.callRealMethod();
            }
        }).when((Object)spy))).isFileDeletable((FileStatus)Mockito.any(FileStatus.class));
        chore.chore();
        Assert.assertTrue((String)"Added file unexpectedly deleted", (boolean)fs.exists(racyFile));
        Assert.assertTrue((String)"Parent directory deleted unexpectedly", (boolean)fs.exists(parent));
        Assert.assertFalse((String)"Original file unexpectedly retained", (boolean)fs.exists(file));
        ((AlwaysDelete)((Object)Mockito.verify((Object)((Object)spy), (VerificationMode)Mockito.times((int)1)))).isFileDeletable((FileStatus)Mockito.any(FileStatus.class));
    }

    @Test
    public void testDeleteFileWithCleanerEnabled() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        chore.setEnabled(true);
        Path parent = new Path(testDir, "parent");
        Path child = new Path(parent, "child");
        Path file = new Path(child, "someFile");
        fs.mkdirs(child);
        fs.create(file).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        chore.chore();
        Assert.assertFalse((String)"File didn't get deleted", (boolean)fs.exists(file));
        Assert.assertFalse((String)"Empty directory didn't get deleted", (boolean)fs.exists(child));
        Assert.assertFalse((String)"Empty directory didn't get deleted", (boolean)fs.exists(parent));
    }

    @Test
    public void testDeleteFileWithCleanerDisabled() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        chore.setEnabled(false);
        Path parent = new Path(testDir, "parent");
        Path child = new Path(parent, "child");
        Path file = new Path(child, "someFile");
        fs.mkdirs(child);
        fs.create(file).close();
        Assert.assertTrue((String)"Test file didn't get created.", (boolean)fs.exists(file));
        chore.chore();
        Assert.assertTrue((String)"File got deleted with cleaner disabled", (boolean)fs.exists(file));
        Assert.assertTrue((String)"Directory got deleted", (boolean)fs.exists(child));
        Assert.assertTrue((String)"Directory got deleted", (boolean)fs.exists(parent));
    }

    @Test
    public void testOnConfigurationChange() throws Exception {
        int availableProcessorNum = Runtime.getRuntime().availableProcessors();
        if (availableProcessorNum == 1) {
            return;
        }
        int initPoolSize = availableProcessorNum / 2;
        int changedPoolSize = availableProcessorNum;
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        conf.set("hbase.cleaner.scan.dir.concurrent.size", String.valueOf(initPoolSize));
        final AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        chore.setEnabled(true);
        int dirNums = 6;
        Path[] subdirs = new Path[dirNums];
        for (int i = 0; i < dirNums; ++i) {
            subdirs[i] = new Path(testDir, "subdir-" + i);
            fs.mkdirs(subdirs[i]);
        }
        for (Path subdir : subdirs) {
            this.createFiles(fs, subdir, 6);
        }
        Thread t = new Thread(new Runnable(){

            @Override
            public void run() {
                chore.chore();
            }
        });
        t.setDaemon(true);
        t.start();
        conf.set("hbase.cleaner.scan.dir.concurrent.size", String.valueOf(changedPoolSize));
        chore.onConfigurationChange(conf);
        Assert.assertEquals((long)changedPoolSize, (long)chore.getChorePoolSize());
        t.join();
    }

    @Test
    public void testMinimumNumberOfThreads() throws Exception {
        StoppableImplementation stop = new StoppableImplementation();
        Configuration conf = UTIL.getConfiguration();
        Path testDir = UTIL.getDataTestDir();
        FileSystem fs = UTIL.getTestFileSystem();
        String confKey = "hbase.test.cleaner.delegates";
        conf.set(confKey, AlwaysDelete.class.getName());
        conf.set("hbase.cleaner.scan.dir.concurrent.size", "2");
        AllValidPaths chore = new AllValidPaths("test-file-cleaner", stop, conf, fs, testDir, confKey);
        int numProcs = Runtime.getRuntime().availableProcessors();
        Assert.assertEquals((long)numProcs, (long)AllValidPaths.calculatePoolSize((String)Integer.toString(numProcs)));
        Assert.assertEquals((long)numProcs, (long)AllValidPaths.calculatePoolSize((String)Integer.toString(numProcs + 2)));
        Assert.assertEquals((long)1L, (long)AllValidPaths.calculatePoolSize((String)"0.0"));
    }

    private void createFiles(FileSystem fs, Path parentDir, int numOfFiles) throws IOException {
        Random random = new Random();
        for (int i = 0; i < numOfFiles; ++i) {
            int xMega = 1 + random.nextInt(3);
            try (FSDataOutputStream fsdos = fs.create(new Path(parentDir, "file-" + i));){
                for (int m = 0; m < xMega; ++m) {
                    byte[] M = new byte[0x100000];
                    random.nextBytes(M);
                    fsdos.write(M);
                }
                continue;
            }
        }
    }

    public static class NeverDelete
    extends BaseHFileCleanerDelegate {
        public boolean isFileDeletable(FileStatus fStat) {
            return false;
        }
    }

    public static class AlwaysDelete
    extends BaseHFileCleanerDelegate {
        public boolean isFileDeletable(FileStatus fStat) {
            return true;
        }
    }

    private static class AllValidPaths
    extends CleanerChore<BaseHFileCleanerDelegate> {
        public AllValidPaths(String name, Stoppable s, Configuration conf, FileSystem fs, Path oldFileDir, String confkey) {
            super(name, Integer.MAX_VALUE, s, conf, fs, oldFileDir, confkey);
        }

        protected boolean validate(Path file) {
            return true;
        }
    }
}

