/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.core.security.authentication.token;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.security.Principal;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.jcr.Credentials;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.jackrabbit.api.security.principal.ItemBasedPrincipal;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.jackrabbit.core.NodeImpl;
import org.apache.jackrabbit.core.SessionImpl;
import org.apache.jackrabbit.core.id.NodeId;
import org.apache.jackrabbit.core.security.authentication.Authentication;
import org.apache.jackrabbit.util.ISO8601;
import org.apache.jackrabbit.util.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TokenBasedAuthentication
implements Authentication {
    private static final Logger log = LoggerFactory.getLogger(TokenBasedAuthentication.class);
    public static final long TOKEN_EXPIRATION = 0x6DDD00L;
    public static final String TOKEN_ATTRIBUTE = ".token";
    private static final String TOKEN_ATTRIBUTE_EXPIRY = ".token.exp";
    private static final String TOKEN_ATTRIBUTE_KEY = ".token.key";
    private static final String TOKENS_NODE_NAME = ".tokens";
    private static final String TOKENS_NT_NAME = "nt:unstructured";
    private static final char DELIM = '_';
    private final String token;
    private final long tokenExpiration;
    private final Session session;
    private final Map<String, String> attributes;
    private final Map<String, String> info;
    private final long expiry;
    private final String key;

    public TokenBasedAuthentication(String token, long tokenExpiration, Session session) throws RepositoryException {
        this.session = session;
        this.tokenExpiration = tokenExpiration;
        this.token = token;
        long expTime = Long.MAX_VALUE;
        String keyV = null;
        if (token != null) {
            this.attributes = new HashMap<String, String>();
            this.info = new HashMap<String, String>();
            Node n = TokenBasedAuthentication.getTokenNode(token, session);
            PropertyIterator it = n.getProperties();
            while (it.hasNext()) {
                Property p = it.nextProperty();
                String name = p.getName();
                if (TOKEN_ATTRIBUTE_EXPIRY.equals(name)) {
                    expTime = p.getLong();
                    continue;
                }
                if (TOKEN_ATTRIBUTE_KEY.equals(name)) {
                    keyV = p.getString();
                    continue;
                }
                if (TokenBasedAuthentication.isMandatoryAttribute(name)) {
                    this.attributes.put(name, p.getString());
                    continue;
                }
                if (!TokenBasedAuthentication.isInfoAttribute(name)) continue;
                this.info.put(name, p.getString());
            }
        } else {
            this.attributes = Collections.emptyMap();
            this.info = Collections.emptyMap();
        }
        this.expiry = expTime;
        this.key = keyV;
    }

    @Override
    public boolean canHandle(Credentials credentials) {
        return this.token != null && TokenBasedAuthentication.isTokenBasedLogin(credentials);
    }

    @Override
    public boolean authenticate(Credentials credentials) throws RepositoryException {
        if (!(credentials instanceof TokenCredentials)) {
            throw new RepositoryException("TokenCredentials expected. Cannot handle " + credentials.getClass().getName());
        }
        TokenCredentials tokenCredentials = (TokenCredentials)credentials;
        if (this.token.equals(tokenCredentials.getToken())) {
            long loginTime = new Date().getTime();
            if (this.expiry < loginTime) {
                this.removeToken();
                return false;
            }
            if (this.key != null && !this.key.equals(TokenBasedAuthentication.getDigestedKey(tokenCredentials))) {
                return false;
            }
            for (String name : this.attributes.keySet()) {
                if (this.attributes.get(name).equals(tokenCredentials.getAttribute(name))) continue;
                return false;
            }
            List<String> attrNames = Arrays.asList(tokenCredentials.getAttributeNames());
            for (String key : this.info.keySet()) {
                if (attrNames.contains(key)) continue;
                tokenCredentials.setAttribute(key, this.info.get(key));
            }
            this.updateTokenNode(this.expiry, loginTime);
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateTokenNode(long tokenExpiry, long loginTime) {
        Session s = null;
        try {
            if (tokenExpiry - loginTime <= this.tokenExpiration / 2L) {
                long expirationTime = loginTime + this.tokenExpiration;
                Calendar cal = GregorianCalendar.getInstance();
                cal.setTimeInMillis(expirationTime);
                s = ((SessionImpl)this.session).createSession(this.session.getWorkspace().getName());
                Node tokenNode = TokenBasedAuthentication.getTokenNode(this.token, s);
                tokenNode.setProperty(TOKEN_ATTRIBUTE_EXPIRY, s.getValueFactory().createValue(cal));
                s.save();
            }
        }
        catch (RepositoryException e) {
            log.warn("Failed to update expiry or informative attributes of token node.", (Throwable)e);
        }
        finally {
            if (s != null) {
                s.logout();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeToken() {
        Session s = null;
        try {
            s = ((SessionImpl)this.session).createSession(this.session.getWorkspace().getName());
            Node tokenNode = TokenBasedAuthentication.getTokenNode(this.token, s);
            tokenNode.remove();
            s.save();
        }
        catch (RepositoryException e) {
            log.warn("Internal error while removing token node.", (Throwable)e);
        }
        finally {
            if (s != null) {
                s.logout();
            }
        }
    }

    public static boolean isTokenBasedLogin(Credentials credentials) {
        return credentials instanceof TokenCredentials;
    }

    public static boolean isMandatoryAttribute(String attributeName) {
        return attributeName != null && attributeName.startsWith(TOKEN_ATTRIBUTE);
    }

    private static boolean isInfoAttribute(String propertyName) {
        String prefix = Text.getNamespacePrefix((String)propertyName);
        return !"jcr".equals(prefix) && !"rep".equals(prefix);
    }

    public static boolean doCreateToken(Credentials credentials) {
        if (credentials instanceof SimpleCredentials) {
            Object attr = ((SimpleCredentials)credentials).getAttribute(TOKEN_ATTRIBUTE);
            return attr != null && "".equals(attr.toString());
        }
        return false;
    }

    public static Credentials createToken(User user, SimpleCredentials credentials, long tokenExpiration, Session session) throws RepositoryException {
        String workspaceName = session.getWorkspace().getName();
        if (user == null) {
            throw new RepositoryException("Cannot create login token: No corresponding node for 'null' user in workspace '" + workspaceName + "'.");
        }
        String userPath = null;
        Principal pr = user.getPrincipal();
        if (pr instanceof ItemBasedPrincipal) {
            userPath = ((ItemBasedPrincipal)pr).getPath();
        }
        if (userPath != null && session.nodeExists(userPath)) {
            Node userNode = session.getNode(userPath);
            if (!userNode.hasNode(TOKENS_NODE_NAME)) {
                userNode.addNode(TOKENS_NODE_NAME, TOKENS_NT_NAME);
                try {
                    session.save();
                }
                catch (RepositoryException e) {
                    session.refresh(false);
                }
            }
            Node tokenParent = userNode.getNode(TOKENS_NODE_NAME);
            long creationTime = new Date().getTime();
            long expirationTime = creationTime + tokenExpiration;
            Calendar cal = GregorianCalendar.getInstance();
            cal.setTimeInMillis(creationTime);
            String key = TokenBasedAuthentication.generateKey(8);
            String tokenName = Text.replace((String)ISO8601.format((Calendar)cal), (String)":", (String)".");
            Node tokenNode = System.getProperty("jackrabbit.sequentialNodeId") == null ? tokenParent.addNode(tokenName) : ((NodeImpl)tokenParent).addNodeWithUuid(tokenName, NodeId.randomId().toString());
            StringBuilder sb = new StringBuilder(tokenNode.getIdentifier());
            sb.append('_').append(key);
            String token = sb.toString();
            TokenCredentials tokenCredentials = new TokenCredentials(token);
            credentials.setAttribute(TOKEN_ATTRIBUTE, (Object)token);
            tokenNode.setProperty(TOKEN_ATTRIBUTE_KEY, TokenBasedAuthentication.getDigestedKey(key));
            cal.setTimeInMillis(expirationTime);
            tokenNode.setProperty(TOKEN_ATTRIBUTE_EXPIRY, session.getValueFactory().createValue(cal));
            for (String name : credentials.getAttributeNames()) {
                if (TOKEN_ATTRIBUTE.equals(name)) continue;
                String value = credentials.getAttribute(name).toString();
                tokenNode.setProperty(name, value);
                tokenCredentials.setAttribute(name, value);
            }
            session.save();
            return tokenCredentials;
        }
        throw new RepositoryException("Cannot create login token: No corresponding node for User " + user.getID() + " in workspace '" + workspaceName + "'.");
    }

    public static Node getTokenNode(TokenCredentials credentials, Session session) throws RepositoryException {
        return TokenBasedAuthentication.getTokenNode(credentials.getToken(), session);
    }

    private static Node getTokenNode(String token, Session session) throws RepositoryException {
        int pos = token.indexOf(95);
        String id = pos == -1 ? token : token.substring(0, pos);
        return session.getNodeByIdentifier(id);
    }

    private static String generateKey(int size) {
        SecureRandom random = new SecureRandom();
        byte[] key = new byte[size];
        random.nextBytes(key);
        StringBuffer res = new StringBuffer(key.length * 2);
        for (byte b : key) {
            res.append(Text.hexTable[b >> 4 & 0xF]);
            res.append(Text.hexTable[b & 0xF]);
        }
        return res.toString();
    }

    private static String getDigestedKey(TokenCredentials tc) throws RepositoryException {
        String tk = tc.getToken();
        int pos = tk.indexOf(95);
        if (pos > -1) {
            return TokenBasedAuthentication.getDigestedKey(tk.substring(pos + 1));
        }
        return null;
    }

    private static String getDigestedKey(String key) throws RepositoryException {
        try {
            StringBuilder sb = new StringBuilder();
            sb.append("{").append("sha1").append("}");
            sb.append(Text.digest((String)"sha1", (String)key, (String)"UTF-8"));
            return sb.toString();
        }
        catch (NoSuchAlgorithmException e) {
            throw new RepositoryException("Failed to generate login token.");
        }
        catch (UnsupportedEncodingException e) {
            throw new RepositoryException("Failed to generate login token.");
        }
    }
}

