/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.openstack.keystone.v2_0.features;

import java.util.Set;
import org.jclouds.collect.PagedIterable;
import org.jclouds.openstack.keystone.v2_0.domain.PaginatedCollection;
import org.jclouds.openstack.keystone.v2_0.domain.Role;
import org.jclouds.openstack.keystone.v2_0.domain.User;
import org.jclouds.openstack.v2_0.options.PaginationOptions;

import com.google.common.annotations.Beta;

/**
 * Provides synchronous access to the KeyStone User API.
 * <p/>
 * 
 * @author Adam Lowe
 * @see UserAsyncApi
 * @see <a href=
 *      "http://docs.openstack.org/api/openstack-identity-service/2.0/content/User_Operations.html"
 *      />
 */
@Beta
public interface UserApi {

   /**
    * Retrieve the list of users
    * <p/>
    * NOTE: this method is not in API documentation for keystone, but does work
    * 
    * @return the list of users
    */
   PagedIterable<? extends User> list();

   PaginatedCollection<? extends User> list(PaginationOptions options);

   /**
    * Retrieve information about a user, by user ID
    * 
    * @return the information about the user
    */
   User get(String userId);

   /**
    * Retrieve information about a user, by user name
    * <p/>
    * NOTE: currently not working in openstack ( https://bugs.launchpad.net/keystone/+bug/956687 )
    * 
    * @return the information about the user
    */
   User getByName(String userName);

   /**
    * Retrieves the list of global roles associated with a specific user (excludes tenant roles).
    * <p/>
    * NOTE: Broken in openstack ( https://bugs.launchpad.net/keystone/+bug/933565 )
    * 
    * @return the set of Roles granted to the user
    */
   Set<? extends Role> listRolesOfUser(String userId);

   /**
    * List the roles a user has been granted on a specific tenant
    * 
    * @return the set of roles
    */
   Set<? extends Role> listRolesOfUserOnTenant(String userId, String tenantId);

}
