/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jclouds.trmk.vcloud_0_8.config;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static org.jclouds.trmk.vcloud_0_8.reference.VCloudConstants.PROPERTY_VCLOUD_DEFAULT_CATALOG;

import java.util.Map;

import javax.inject.Inject;
import javax.inject.Singleton;

import org.jclouds.config.ValueOfConfigurationKeyOrNull;
import org.jclouds.trmk.vcloud_0_8.domain.Org;
import org.jclouds.trmk.vcloud_0_8.domain.ReferenceType;
import org.jclouds.trmk.vcloud_0_8.endpoints.Catalog;
import org.jclouds.trmk.vcloud_0_8.suppliers.OnlyReferenceTypeFirstWithNameMatchingConfigurationKeyOrDefault;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;

/**
 * 
 * @author Adrian Cole
 */
@Singleton
public class DefaultCatalogForOrg implements Function<ReferenceType, ReferenceType> {
   private final OnlyReferenceTypeFirstWithNameMatchingConfigurationKeyOrDefault selector;
   private final Supplier<Map<String, ? extends Org>> nameToOrg;

   @Inject
   public DefaultCatalogForOrg(ValueOfConfigurationKeyOrNull valueOfConfigurationKeyOrNull,
         @Catalog Predicate<ReferenceType> defaultSelector, Supplier<Map<String, ? extends Org>> nameToOrg) {
      this.selector = new OnlyReferenceTypeFirstWithNameMatchingConfigurationKeyOrDefault(checkNotNull(
            valueOfConfigurationKeyOrNull, "valueOfConfigurationKeyOrNull"), PROPERTY_VCLOUD_DEFAULT_CATALOG,
            checkNotNull(defaultSelector, "defaultSelector"));
      this.nameToOrg = checkNotNull(nameToOrg, "nameToOrg");
   }

   @Override
   public ReferenceType apply(ReferenceType defaultOrg) {
      org.jclouds.trmk.vcloud_0_8.domain.Org org = nameToOrg.get().get(defaultOrg.getName());
      checkState(org != null, "could not retrieve Org at %s", defaultOrg);
      return selector.apply(org.getCatalogs().values());
   }

}
