/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.johnzon.jaxrs;

import javax.json.Json;
import javax.json.JsonReader;
import javax.json.JsonReaderFactory;
import javax.json.JsonStructure;
import javax.ws.rs.Consumes;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyReader;
import javax.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collections;

import static javax.ws.rs.core.MediaType.WILDCARD;

@Provider
@Consumes(WILDCARD)
public class JsrMessageBodyReader implements MessageBodyReader<JsonStructure> {
    private final JsonReaderFactory factory;
    private final boolean closeStream;

    public JsrMessageBodyReader() {
        this(Json.createReaderFactory(Collections.<String, Object>emptyMap()), false);
    }

    public JsrMessageBodyReader(final JsonReaderFactory factory, final boolean closeStream) {
        this.factory = factory;
        this.closeStream = closeStream;
    }

    @Override
    public boolean isReadable(final Class<?> aClass, final Type type,
                              final Annotation[] annotations, final MediaType mediaType) {
        return JsonStructure.class.isAssignableFrom(aClass);
    }

    @Override
    public JsonStructure readFrom(final Class<JsonStructure> jsonStructureClass, final Type type,
                                  final Annotation[] annotations, final MediaType mediaType,
                                  final MultivaluedMap<String, String> stringStringMultivaluedMap,
                                  final InputStream inputStream) throws IOException, WebApplicationException {
        JsonReader reader = null;
        try {
            reader = factory.createReader(inputStream);
            return reader.read();
        } finally {
            if (closeStream && reader != null) {
                reader.close();
            }
        }
    }
}
