/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.oauth;

import com.google.common.collect.Lists;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.logging.Handler;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import net.oauth.OAuth;
import org.apache.commons.codec.binary.Base64;
import org.apache.shindig.auth.BasicSecurityToken;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.cache.CacheProvider;
import org.apache.shindig.common.cache.LruCacheProvider;
import org.apache.shindig.common.crypto.BasicBlobCrypter;
import org.apache.shindig.common.crypto.BlobCrypter;
import org.apache.shindig.common.util.CharsetUtil;
import org.apache.shindig.common.util.FakeTimeSource;
import org.apache.shindig.common.util.TimeSource;
import org.apache.shindig.gadgets.FakeGadgetSpecFactory;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.GadgetSpecFactory;
import org.apache.shindig.gadgets.RequestSigningException;
import org.apache.shindig.gadgets.http.DefaultHttpCache;
import org.apache.shindig.gadgets.http.HttpCache;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.oauth.AccessorInfo;
import org.apache.shindig.gadgets.oauth.BasicOAuthStore;
import org.apache.shindig.gadgets.oauth.BasicOAuthStoreConsumerIndex;
import org.apache.shindig.gadgets.oauth.BasicOAuthStoreConsumerKeyAndSecret;
import org.apache.shindig.gadgets.oauth.GadgetOAuthTokenStore;
import org.apache.shindig.gadgets.oauth.OAuthArguments;
import org.apache.shindig.gadgets.oauth.OAuthError;
import org.apache.shindig.gadgets.oauth.OAuthFetcher;
import org.apache.shindig.gadgets.oauth.OAuthFetcherConfig;
import org.apache.shindig.gadgets.oauth.OAuthStore;
import org.apache.shindig.gadgets.oauth.testing.FakeOAuthServiceProvider;
import org.apache.shindig.gadgets.oauth.testing.MakeRequestClient;
import org.json.JSONObject;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class OAuthFetcherTest {
    private OAuthFetcherConfig fetcherConfig;
    private FakeOAuthServiceProvider serviceProvider;
    private BasicOAuthStore base;
    private final List<LogRecord> logRecords = Lists.newArrayList();
    private FakeTimeSource clock = new FakeTimeSource();
    public static final String GADGET_URL = "http://www.example.com/gadget.xml";
    public static final String GADGET_URL_NO_KEY = "http://www.example.com/nokey.xml";
    public static final String GADGET_URL_HEADER = "http://www.example.com/header.xml";
    public static final String GADGET_URL_BODY = "http://www.example.com/body.xml";

    @Before
    public void setUp() throws Exception {
        this.base = new BasicOAuthStore();
        this.serviceProvider = new FakeOAuthServiceProvider((TimeSource)this.clock);
        this.fetcherConfig = new OAuthFetcherConfig((BlobCrypter)new BasicBlobCrypter("abcdefghijklmnop".getBytes()), OAuthFetcherTest.getOAuthStore(this.base), (HttpCache)new DefaultHttpCache((CacheProvider)new LruCacheProvider(10)), (TimeSource)this.clock);
        Logger logger = Logger.getLogger(OAuthFetcher.class.getName());
        logger.addHandler(new Handler(){

            public void close() throws SecurityException {
            }

            public void flush() {
            }

            public void publish(LogRecord arg0) {
                OAuthFetcherTest.this.logRecords.add(arg0);
            }
        });
    }

    public static GadgetOAuthTokenStore getOAuthStore(BasicOAuthStore base) throws GadgetException {
        if (base == null) {
            base = new BasicOAuthStore();
        }
        OAuthFetcherTest.addValidConsumer(base);
        OAuthFetcherTest.addInvalidConsumer(base);
        OAuthFetcherTest.addAuthHeaderConsumer(base);
        OAuthFetcherTest.addBodyConsumer(base);
        OAuthFetcherTest.addDefaultKey(base);
        GadgetOAuthTokenStore store = new GadgetOAuthTokenStore((OAuthStore)base, (GadgetSpecFactory)new FakeGadgetSpecFactory());
        base.initFromConfigString("{}");
        return store;
    }

    private static void addValidConsumer(BasicOAuthStore base) {
        OAuthFetcherTest.addConsumer(base, GADGET_URL, "testservice", "consumer", "secret");
    }

    private static void addInvalidConsumer(BasicOAuthStore base) {
        OAuthFetcherTest.addConsumer(base, GADGET_URL_NO_KEY, "nokey", "garbage_key", "garbage_secret");
    }

    private static void addAuthHeaderConsumer(BasicOAuthStore base) {
        OAuthFetcherTest.addConsumer(base, GADGET_URL_HEADER, "testservice", "consumer", "secret");
    }

    private static void addBodyConsumer(BasicOAuthStore base) {
        OAuthFetcherTest.addConsumer(base, GADGET_URL_BODY, "testservice", "consumer", "secret");
    }

    private static void addConsumer(BasicOAuthStore base, String gadgetUrl, String serviceName, String consumerKey, String consumerSecret) {
        BasicOAuthStoreConsumerIndex providerKey = new BasicOAuthStoreConsumerIndex();
        providerKey.setGadgetUri(gadgetUrl);
        providerKey.setServiceName(serviceName);
        BasicOAuthStoreConsumerKeyAndSecret kas = new BasicOAuthStoreConsumerKeyAndSecret(consumerKey, consumerSecret, BasicOAuthStoreConsumerKeyAndSecret.KeyType.HMAC_SYMMETRIC, null);
        base.setConsumerKeyAndSecret(providerKey, kas);
    }

    private static void addDefaultKey(BasicOAuthStore base) {
        BasicOAuthStoreConsumerKeyAndSecret defaultKey = new BasicOAuthStoreConsumerKeyAndSecret("signedfetch", "MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALRiMLAh9iimur8VA7qVvdqxevEuUkW4K+2KdMXmnQbG9Aa7k7eBjK1S+0LYmVjPKlJGNXHDGuy5Fw/d7rjVJ0BLB+ubPK8iA/Tw3hLQgXMRRGRXXCn8ikfuQfjUS1uZSatdLB81mydBETlJhI6GH4twrbDJCR2Bwy/XWXgqgGRzAgMBAAECgYBYWVtleUzavkbrPjy0T5FMou8HX9u2AC2ry8vD/l7cqedtwMPp9k7TubgNFo+NGvKsl2ynyprOZR1xjQ7WgrgVB+mmuScOM/5HVceFuGRDhYTCObE+y1kxRloNYXnx3ei1zbeYLPCHdhxRYW7T0qcynNmwrn05/KO2RLjgQNalsQJBANeA3Q4Nugqy4QBUCEC09SqylT2K9FrrItqL2QKc9v0ZzO2uwllCbg0dwpVuYPYXYvikNHHg+aCWF+VXsb9rpPsCQQDWR9TT4ORdzoj+NccnqkMsDmzt0EfNaAOwHOmVJ2RVBspPcxt5iN4HI7HNeG6U5YsFBb+/GZbgfBT3kpNGWPTpAkBI+gFhjfJvRw38n3g/+UeAkwMI2TJQS4n8+hid0uus3/zOjDySH3XHCUnocn1xOJAyZODBo47E+67R4jV1/gzbAkEAklJaspRPXP877NssM5nAZMU0/O/NGCZ+3jPgDUno6WbJn5cqm8MqWhW1xGkImgRk+fkDBquiq4gPiT898jusgQJAd5Zrr6Q8AO/0isr/3aa6O6NLQxISLKcPDk2NOccAfS/xOtfOz4sJYM3+Bs4Io9+dZGSDCA54Lw03eHTNQghS0A==", BasicOAuthStoreConsumerKeyAndSecret.KeyType.RSA_PRIVATE, "foo");
        base.setDefaultKey(defaultKey);
    }

    public static SecurityToken getNormalSecurityToken(String owner, String viewer) throws Exception {
        return OAuthFetcherTest.getSecurityToken(owner, viewer, GADGET_URL);
    }

    public static SecurityToken getNokeySecurityToken(String owner, String viewer) throws Exception {
        return OAuthFetcherTest.getSecurityToken(owner, viewer, GADGET_URL_NO_KEY);
    }

    public static SecurityToken getHeaderSecurityToken(String owner, String viewer) throws Exception {
        return OAuthFetcherTest.getSecurityToken(owner, viewer, GADGET_URL_HEADER);
    }

    public static SecurityToken getBodySecurityToken(String owner, String viewer) throws Exception {
        return OAuthFetcherTest.getSecurityToken(owner, viewer, GADGET_URL_BODY);
    }

    public static SecurityToken getSecurityToken(String owner, String viewer, String gadget) throws Exception {
        return new BasicSecurityToken(owner, viewer, "app", "container.com", gadget, "0");
    }

    @After
    public void tearDown() throws Exception {
    }

    private MakeRequestClient makeNonSocialClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken(owner, viewer, gadget);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
        client.getBaseArgs().setSignOwner(true);
        client.getBaseArgs().setSignViewer(true);
        return client;
    }

    private MakeRequestClient makeStrictNonSocialClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken(owner, viewer, gadget);
        return new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
    }

    private MakeRequestClient makeSocialOAuthClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken(owner, viewer, gadget);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "testservice");
        client.getBaseArgs().setUseToken(OAuthArguments.UseToken.IF_AVAILABLE);
        return client;
    }

    private MakeRequestClient makeSignedFetchClient(String owner, String viewer, String gadget) throws Exception {
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken(owner, viewer, gadget);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, null);
        client.setBaseArgs(client.makeSignedFetchArguments());
        return client;
    }

    @Test
    public void testOAuthFlow() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.checkEmptyLog();
    }

    @Test
    public void testOAuthFlow_tokenReused() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        MakeRequestClient client2 = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        response = client2.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testOAuthFlow_unauthUser() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient(null, null, GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.UNAUTHENTICATED.toString(), response.getMetadata().get("oauthError"));
    }

    @Test
    public void testAccessTokenNotUsedForSocialPage() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        MakeRequestClient friend = this.makeNonSocialClient("owner", "friend", GADGET_URL);
        response = friend.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)403L, (long)response.getHttpStatusCode());
        Assert.assertEquals((Object)OAuthError.NOT_OWNER.toString(), response.getMetadata().get("oauthError"));
    }

    @Test
    public void testParamsInHeader() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_HEADER);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String aznHeader = response.getHeader("X-Echoed-Authz");
        Assert.assertNotNull((Object)aznHeader);
        Assert.assertTrue((String)("azn header: " + aznHeader), (aznHeader.indexOf("OAuth") != -1 ? 1 : 0) != 0);
    }

    @Test
    public void testParamsInBody() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendFormPost("http://www.example.com/data", "");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String echoedBody = response.getHeader("X-Echoed-Body");
        Assert.assertNotNull((Object)echoedBody);
        Assert.assertTrue((String)("body: " + echoedBody), (echoedBody.indexOf("oauth_consumer_key=") != -1 ? 1 : 0) != 0);
    }

    @Test
    public void testParamsInBody_withExtraParams() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendFormPost("http://www.example.com/data", "foo=bar&foo=baz");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String echoedBody = response.getHeader("X-Echoed-Body");
        Assert.assertNotNull((Object)echoedBody);
        Assert.assertTrue((String)("body: " + echoedBody), (echoedBody.indexOf("oauth_consumer_key=") != -1 ? 1 : 0) != 0);
        Assert.assertTrue((String)("body: " + echoedBody), (echoedBody.indexOf("foo=bar&foo=baz") != -1 ? 1 : 0) != 0);
    }

    @Test
    public void testParamsInBody_forGetRequest() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        this.serviceProvider.addParamLocation(AccessorInfo.OAuthParamLocation.AUTH_HEADER);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        String aznHeader = response.getHeader("X-Echoed-Authz");
        Assert.assertNotNull((Object)aznHeader);
        Assert.assertTrue((String)("azn header: " + aznHeader), (aznHeader.indexOf("OAuth") != -1 ? 1 : 0) != 0);
    }

    @Test
    public void testParamsInBody_forGetRequestStrictSp() throws Exception {
        this.serviceProvider.setParamLocation(AccessorInfo.OAuthParamLocation.POST_BODY);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL_BODY);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertNotNull(response.getMetadata().get("oauthApprovalUrl"));
    }

    @Test
    public void testRevokedAccessToken() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.serviceProvider.revokeAllAccessTokens();
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertNotNull(response.getMetadata().get("oauthApprovalUrl"));
        client.approveToken("user_data=reapproved");
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is reapproved", (Object)response.getResponseAsString());
    }

    @Test
    public void testError401() throws Exception {
        this.serviceProvider.setVagueErrors(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.serviceProvider.revokeAllAccessTokens();
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        this.checkLogContains("GET /data?cachebust=2");
        this.checkLogContains("HTTP/1.1 401");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertNotNull(response.getMetadata().get("oauthApprovalUrl"));
        client.approveToken("user_data=reapproved");
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is reapproved", (Object)response.getResponseAsString());
    }

    @Test
    public void testUnknownConsumerKey() throws Exception {
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken("owner", "owner", GADGET_URL_NO_KEY);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "nokey");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"consumer_key_unknown", metadata.get("oauthError"));
        Assert.assertEquals((Object)"invalid consumer: garbage_key", metadata.get("oauthErrorText"));
    }

    @Test
    public void testError403() throws Exception {
        this.serviceProvider.setVagueErrors(true);
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken("owner", "owner", GADGET_URL_NO_KEY);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "nokey");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"403", metadata.get("oauthError"));
        Assert.assertNull(metadata.get("oauthErrorText"));
        this.checkLogContains("HTTP/1.1 403");
        this.checkLogContains("GET /request");
        this.checkLogContains("some vague error");
    }

    @Test
    public void testError404() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        response = client.sendGet("http://www.example.com/404");
        Assert.assertEquals((Object)"not found", (Object)response.getResponseAsString());
        Assert.assertEquals((long)404L, (long)response.getHttpStatusCode());
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testConsumerThrottled() throws Exception {
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        this.serviceProvider.setConsumersThrottled(true);
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Map metadata = response.getMetadata();
        Assert.assertNotNull((Object)metadata);
        Assert.assertEquals((Object)"consumer_key_refused", metadata.get("oauthError"));
        Assert.assertEquals((Object)"exceeded quota", metadata.get("oauthErrorText"));
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        this.serviceProvider.setConsumersThrottled(false);
        response = client.sendGet("http://www.example.com/data?cachebust=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testSocialOAuth_tokenRevoked() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        this.serviceProvider.revokeAllAccessTokens();
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
        client = this.makeSocialOAuthClient("owner", "owner", GADGET_URL);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.base.getAccessTokenRemoveCount());
    }

    @Test
    public void testWrongServiceName() throws Exception {
        SecurityToken securityToken = OAuthFetcherTest.getSecurityToken("owner", "owner", GADGET_URL);
        MakeRequestClient client = new MakeRequestClient(securityToken, this.fetcherConfig, this.serviceProvider, "nosuchservice");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Map metadata = response.getMetadata();
        Assert.assertNull(metadata.get("oauthApprovalUrl"));
        Assert.assertEquals((Object)"BAD_OAUTH_CONFIGURATION", metadata.get("oauthError"));
        String errorText = (String)metadata.get("oauthErrorText");
        Assert.assertTrue((String)errorText, (boolean)errorText.startsWith("Spec for gadget http://www.example.com/gadget.xml does not contain OAuth service nosuchservice.  Known services: testservice"));
    }

    @Test
    public void testPreapprovedToken() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        FakeOAuthServiceProvider.TokenPair reqToken = this.serviceProvider.getPreapprovedToken("preapproved");
        client.getBaseArgs().setRequestToken(reqToken.token);
        client.getBaseArgs().setRequestTokenSecret(reqToken.secret);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cachebust=2");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testPreapprovedToken_invalid() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        client.getBaseArgs().setRequestToken("garbage");
        client.getBaseArgs().setRequestTokenSecret("garbage");
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testPreapprovedToken_notUsedIfAccessTokenExists() throws Exception {
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        FakeOAuthServiceProvider.TokenPair reqToken = this.serviceProvider.getPreapprovedToken("preapproved");
        client.getBaseArgs().setRequestToken(reqToken.token);
        client.getBaseArgs().setRequestTokenSecret(reqToken.secret);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client2 = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        response = client2.sendGet("http://www.example.com/data?cachebust=1");
        Assert.assertEquals((Object)"User data is preapproved", (Object)response.getResponseAsString());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testCachedResponse() throws Exception {
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)0L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testSignedFetchParametersSet() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
    }

    @Test
    public void testPostBinaryData() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendRawPost("http://www.example.com/data", null, raw);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        String echoed = resp.getHeader("X-Echoed-Raw-Body");
        byte[] echoedBytes = Base64.decodeBase64((byte[])CharsetUtil.getUtf8Bytes((String)echoed));
        Assert.assertTrue((boolean)Arrays.equals(raw, echoedBytes));
    }

    @Test
    public void testPostWeirdContentType() throws Exception {
        byte[] raw = new byte[]{0, 1, 2, 3, 4, 5};
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendRawPost("http://www.example.com/data", "funky-content", raw);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        String echoed = resp.getHeader("X-Echoed-Raw-Body");
        byte[] echoedBytes = Base64.decodeBase64((byte[])CharsetUtil.getUtf8Bytes((String)echoed));
        Assert.assertTrue((boolean)Arrays.equals(raw, echoedBytes));
    }

    @Test
    public void testSignedFetch_error401() throws Exception {
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
        this.serviceProvider.setConsumersThrottled(true);
        this.serviceProvider.setVagueErrors(true);
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)0L, (long)this.base.getAccessTokenRemoveCount());
    }

    @Test
    public void testSignedFetch_unnamedConsumerKey() throws Exception {
        BasicOAuthStoreConsumerKeyAndSecret defaultKey = new BasicOAuthStoreConsumerKeyAndSecret(null, "MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALRiMLAh9iimur8VA7qVvdqxevEuUkW4K+2KdMXmnQbG9Aa7k7eBjK1S+0LYmVjPKlJGNXHDGuy5Fw/d7rjVJ0BLB+ubPK8iA/Tw3hLQgXMRRGRXXCn8ikfuQfjUS1uZSatdLB81mydBETlJhI6GH4twrbDJCR2Bwy/XWXgqgGRzAgMBAAECgYBYWVtleUzavkbrPjy0T5FMou8HX9u2AC2ry8vD/l7cqedtwMPp9k7TubgNFo+NGvKsl2ynyprOZR1xjQ7WgrgVB+mmuScOM/5HVceFuGRDhYTCObE+y1kxRloNYXnx3ei1zbeYLPCHdhxRYW7T0qcynNmwrn05/KO2RLjgQNalsQJBANeA3Q4Nugqy4QBUCEC09SqylT2K9FrrItqL2QKc9v0ZzO2uwllCbg0dwpVuYPYXYvikNHHg+aCWF+VXsb9rpPsCQQDWR9TT4ORdzoj+NccnqkMsDmzt0EfNaAOwHOmVJ2RVBspPcxt5iN4HI7HNeG6U5YsFBb+/GZbgfBT3kpNGWPTpAkBI+gFhjfJvRw38n3g/+UeAkwMI2TJQS4n8+hid0uus3/zOjDySH3XHCUnocn1xOJAyZODBo47E+67R4jV1/gzbAkEAklJaspRPXP877NssM5nAZMU0/O/NGCZ+3jPgDUno6WbJn5cqm8MqWhW1xGkImgRk+fkDBquiq4gPiT898jusgQJAd5Zrr6Q8AO/0isr/3aa6O6NLQxISLKcPDk2NOccAfS/xOtfOz4sJYM3+Bs4Io9+dZGSDCA54Lw03eHTNQghS0A==", BasicOAuthStoreConsumerKeyAndSecret.KeyType.RSA_PRIVATE, "foo");
        this.base.setDefaultKey(defaultKey);
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "container.com"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
    }

    @Test
    public void testSignedFetch_extraQueryParameters() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data?foo=bar&foo=baz");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_app_id", "app"));
        Assert.assertTrue((boolean)this.contains(queryParams, "oauth_consumer_key", "signedfetch"));
        Assert.assertTrue((boolean)this.contains(queryParams, "xoauth_signature_publickey", "foo"));
    }

    @Test
    public void testNoSignViewer() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setSignViewer(false);
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertFalse((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
    }

    @Test
    public void testNoSignOwner() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setSignOwner(false);
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertFalse((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
    }

    @Test
    public void testCacheHit() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testCacheMiss_noOwner() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.getBaseArgs().setSignOwner(false);
        client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client2 = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client2.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testCacheHit_ownerOnly() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v1", "http://www.example.com/app");
        client.getBaseArgs().setSignViewer(false);
        client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client2 = this.makeSignedFetchClient("o", "v2", "http://www.example.com/app");
        client2.getBaseArgs().setSignViewer(false);
        client2.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testCacheMiss_bypassCache() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v1", "http://www.example.com/app");
        client.getBaseArgs().setSignViewer(false);
        client.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        MakeRequestClient client2 = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client2.setIgnoreCache(true);
        client2.getBaseArgs().setSignViewer(false);
        client2.sendGet("http://www.example.com/data");
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test(expected=RequestSigningException.class)
    public void testTrickyParametersInQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        String tricky = "%6fpensocial_owner_id=gotcha";
        client.sendGet("http://www.example.com/data?" + tricky);
    }

    @Test(expected=RequestSigningException.class)
    public void testTrickyParametersInBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        String tricky = "%6fpensocial_owner_id=gotcha";
        client.sendFormPost("http://www.example.com/data", tricky);
    }

    @Test
    public void testGetNoQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_viewer_id", "v"));
    }

    @Test
    public void testGetWithQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data?a=b");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "a", "b"));
    }

    @Test
    public void testGetWithQueryMultiParam() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendGet("http://www.example.com/data?a=b&a=c");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "a", "b"));
        Assert.assertTrue((boolean)this.contains(queryParams, "a", "c"));
    }

    @Test
    public void testValidParameterCharacters() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        String weird = "~!@$*()-_[]:,./";
        HttpResponse resp = client.sendGet("http://www.example.com/data?" + weird + "=foo");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, weird, "foo"));
    }

    @Test
    public void testPostNoQueryNoData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", null);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "opensocial_owner_id", "o"));
        Assert.assertEquals((Object)"", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testPostWithQueryNoData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data?name=value", null);
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "name", "value"));
        Assert.assertEquals((Object)"", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testPostNoQueryWithData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data", "name=value");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertFalse((boolean)this.contains(queryParams, "name", "value"));
        Assert.assertEquals((Object)"name=value", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test
    public void testPostWithQueryWithData() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        HttpResponse resp = client.sendFormPost("http://www.example.com/data?queryName=queryValue", "name=value");
        List queryParams = OAuth.decodeForm((String)resp.getResponseAsString());
        Assert.assertTrue((boolean)this.contains(queryParams, "queryName", "queryValue"));
        Assert.assertEquals((Object)"name=value", (Object)resp.getHeader("X-Echoed-Body"));
    }

    @Test(expected=RequestSigningException.class)
    public void testStripOpenSocialParamsFromQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.sendFormPost("http://www.example.com/data?opensocial_foo=bar", null);
    }

    @Test(expected=RequestSigningException.class)
    public void testStripOAuthParamsFromQuery() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.sendFormPost("http://www.example.com/data?oauth_foo=bar", "name=value");
    }

    @Test(expected=RequestSigningException.class)
    public void testStripOpenSocialParamsFromBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.sendFormPost("http://www.example.com/data", "opensocial_foo=bar");
    }

    @Test(expected=RequestSigningException.class)
    public void testStripOAuthParamsFromBody() throws Exception {
        MakeRequestClient client = this.makeSignedFetchClient("o", "v", "http://www.example.com/app");
        client.sendFormPost("http://www.example.com/data", "oauth_foo=bar");
    }

    @Test
    public void testAccessTokenExpires_onClient() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cb=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=4");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testAccessTokenExpires_onClientNoPredictedExpiration() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        this.serviceProvider.setReportExpirationTimes(false);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getResourceAccessCount());
        response = client.sendGet("http://www.example.com/data?cb=3");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)4L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=4");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)6L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testAccessTokenExpires_onServer() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        this.clock.incrementSeconds(61);
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
    }

    @Test
    public void testAccessTokenExpired_andRevoked() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        this.serviceProvider.revokeAllAccessTokens();
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=renewed");
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        Assert.assertEquals((Object)"User data is renewed", (Object)response.getResponseAsString());
    }

    @Test
    public void testBadSessionHandle() throws Exception {
        this.serviceProvider.setSessionExtension(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        this.clock.incrementSeconds(61);
        this.serviceProvider.changeAllSessionHandles();
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)1L, (long)this.serviceProvider.getResourceAccessCount());
        client.approveToken("user_data=renewed");
        response = client.sendGet("http://www.example.com/data?cb=1");
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getRequestTokenCount());
        Assert.assertEquals((long)3L, (long)this.serviceProvider.getAccessTokenCount());
        Assert.assertEquals((long)2L, (long)this.serviceProvider.getResourceAccessCount());
        Assert.assertEquals((Object)"User data is renewed", (Object)response.getResponseAsString());
    }

    @Test
    public void testExtraParamsRejected() throws Exception {
        this.serviceProvider.setRejectExtraParams(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"parameter_rejected", response.getMetadata().get("oauthError"));
    }

    @Test
    public void testExtraParamsSuppressed() throws Exception {
        this.serviceProvider.setRejectExtraParams(true);
        MakeRequestClient client = this.makeStrictNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testCanRetrieveAccessTokenData() throws Exception {
        this.serviceProvider.setReturnAccessTokenData(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"application/json; charset=utf-8", (Object)response.getHeader("Content-Type"));
        JSONObject json = new JSONObject(response.getResponseAsString());
        Assert.assertEquals((Object)"userid value", (Object)json.get("userid"));
        Assert.assertEquals((Object)"xoauth_stuff value", (Object)json.get("xoauth_stuff"));
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testAccessTokenData_noOAuthParams() throws Exception {
        this.serviceProvider.setReturnAccessTokenData(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/access");
        JSONObject json = new JSONObject(response.getResponseAsString());
        Assert.assertEquals((Object)"userid value", (Object)json.get("userid"));
        Assert.assertEquals((Object)"xoauth_stuff value", (Object)json.get("xoauth_stuff"));
        Assert.assertEquals((long)2L, (long)json.length());
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
    }

    @Test
    public void testAccessTokenData_noDirectRequest() throws Exception {
        this.serviceProvider.setReturnAccessTokenData(true);
        MakeRequestClient client = this.makeNonSocialClient("owner", "owner", GADGET_URL);
        HttpResponse response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        client.approveToken("user_data=hello-oauth");
        response = client.sendGet("http://www.example.com/data");
        Assert.assertEquals((Object)"User data is hello-oauth", (Object)response.getResponseAsString());
        response = client.sendGet("http://www.example.com/access");
        Assert.assertEquals((Object)"", (Object)response.getResponseAsString());
        Assert.assertTrue((boolean)response.getMetadata().containsKey("oauthApprovalUrl"));
    }

    private boolean contains(List<OAuth.Parameter> params, String key, String value) {
        for (OAuth.Parameter p : params) {
            if (!p.getKey().equals(key) || !p.getValue().equals(value)) continue;
            return true;
        }
        return false;
    }

    private String getLogText() {
        StringBuilder logText = new StringBuilder();
        for (LogRecord record : this.logRecords) {
            logText.append(record.getMessage());
        }
        return logText.toString();
    }

    private void checkLogContains(String text) {
        String logText = this.getLogText();
        if (!logText.contains(text)) {
            Assert.fail((String)("Should have logged '" + text + "', instead got " + logText));
        }
    }

    private void checkEmptyLog() {
        Assert.assertEquals((Object)"", (Object)this.getLogText());
    }
}

