/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.apache.shindig.gadgets.parse;

import java.io.IOException;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * Handles serializing and deserializing parse trees (Lists of {@code ParsedHtmlNode} objects)
 * to and from byte[] form.
 * 
 * When deserializing, different underlying objects are actually
 * provided, but provide identical data behind the ParsedHtmlNode and Attribute interfaces.
 * This is a side effect of the fact that what's actually serialized are custom-serialized
 * implementations of these interfaces. This custom implementation provides several benefits
 * as compared to standard Java serialization, smaller data size, faster processing time,
 * and cross-platform compatibility.
 * 
 * Each serialized object is annotated with a versionID, similar to serialVersionUID used
 * by Java's serialization mechanism.
 */
public class ParseTreeSerializer {
  private static final int SERIALIZATION_VERSION_ID = 1;

  /**
   * Convert the list of {@code ParsedHtmlNode} elements into a byte blob suitable
   * for persisting in a cache.
   * @param nodes List of parsed html contents.
   * @return Blob of bytes representing contents.
   */
  public byte[] serialize(List<ParsedHtmlNode> nodes) {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    ObjectOutputStream oos = null;
    try {
      SerializableParsedNode parent = SerializableParsedNode.makeParent(nodes);
      parent.writeObject(baos);
    } catch (IOException e) {
      // Never happens.
    }
    return baos.toByteArray();
  }
  
  /**
   * Attempt to convert a blob of bytes, likely generated by {@code serialize()},
   * into a Java representation of parsed content.
   * @param buffer Byte blob of serialized data.
   * @return List of nodes.
   */
  public List<ParsedHtmlNode> deserialize(byte[] buffer) {
    ByteArrayInputStream bais = new ByteArrayInputStream(buffer);
    ObjectInputStream ois = null;
    List<ParsedHtmlNode> nodes = null;
    try {
      SerializableParsedNode spn = SerializableParsedNode.readObject(bais);
      if (spn == null) {
        return null;
      }
      nodes = spn.getChildren();
    } catch (IOException e) {
      // Strange, but no matter: invalidate by returning null.
    } catch (ClassCastException e) {
      // Invalidate entry by returning null.
    }
    return nodes;
  }
  
  // Helper method: converts input list to a list of SerializableParsedNodes
  private static List<SerializableParsedNode> getSerializableNodeList(List<ParsedHtmlNode> nodes) {
    if (nodes == null) {
      return null;
    }
    List<SerializableParsedNode> outNodes = new ArrayList<SerializableParsedNode>(nodes.size());
    for (ParsedHtmlNode node : nodes) {
      outNodes.add(new SerializableParsedNode(node));
    }
    return outNodes;
  }
  
  // Helper method: converts input list to a list of SerializableParsedAttribs
  private static List<SerializableParsedAttrib> getSerializableAttribList(
      List<ParsedHtmlAttribute> attribs) {
    if (attribs == null) {
      return null;
    }
    List<SerializableParsedAttrib> outAttribs = new ArrayList<SerializableParsedAttrib>(attribs.size());
    for (ParsedHtmlAttribute attrib : attribs) {
      outAttribs.add(new SerializableParsedAttrib(attrib));
    }
    return outAttribs;
  }
  
  // Serialization helper: writes an integer to an OutputStream
  private static void writeInt(OutputStream out, int num) throws IOException {
    out.write(num >> 0);
    out.write(num >> 8);
    out.write(num >> 16);
    out.write(num >> 24);
  }
  
  // Serialization helper: reads an integer written by writeInt from an InputStream
  private static int readInt(InputStream in) throws IOException {
    return in.read() << 0 |
           in.read() << 8 |
           in.read() << 16 |
           in.read() << 24;
  }

  // Serialization helper: write a String to OutputStream. Uses UTF8 format for all Strings
  private static void writeString(OutputStream out, String str) throws IOException {
    if (str == null) {
      writeInt(out, 0);
      return;
    }
    byte[] utf8bytes = str.getBytes("UTF8");
    writeInt(out, utf8bytes.length);
    out.write(utf8bytes, 0, utf8bytes.length);
  }
  
  // Serialization helper: reads a writeString()-written String from the given InputStream
  private static String readString(InputStream in) throws IOException {
    int len = readInt(in);
    if (len == 0) {
      return null;
    }
    byte[] stringBytes = new byte[len];
    int read = 0;
    while (read < len) {
      int thisTime = in.read(stringBytes, read, (len - read));
      if (thisTime == -1) {
        throw new IOException("Insufficient data in buffer to read");
      }
      read += thisTime;
    }
    return new String(stringBytes, "UTF8");
  }
  
  //
  // ParsedHtmlNode implementation providing custom serialization routines.
  // This class is used for both serializing and deserializing ParsedHtmlNode lists.
  //
  private static class SerializableParsedNode implements ParsedHtmlNode {
    private String tag;
    private List<SerializableParsedAttrib> sattribs;
    private List<ParsedHtmlAttribute> attribs;
    private List<SerializableParsedNode> schildren;
    private List<ParsedHtmlNode> children;
    private String text;

    /** {@inheritDoc} */
    public List<ParsedHtmlAttribute> getAttributes() {
      if (attribs == null) {
        attribs = new ArrayList<ParsedHtmlAttribute>(this.sattribs.size());
        attribs.addAll(this.sattribs);
      }
      return attribs;
    }

    /** {@inheritDoc} */
    public List<ParsedHtmlNode> getChildren() {
      if (children == null) {
        children = new ArrayList<ParsedHtmlNode>(this.schildren.size());
        children.addAll(this.schildren);
      }
      return children;
    }

    /** {@inheritDoc} */
    public String getTagName() {
      return tag;
    }

    /** {@inheritDoc} */
    public String getText() {
      return text;
    }
    
    // Key helper method, responsible for writing this object to OutputStream as bytes.
    private void writeObject(OutputStream out) throws IOException {
      // Format: version, then isTag, followed by fields.
      writeInt(out, SERIALIZATION_VERSION_ID);
      
      if (tag != null) {
        // isTag
        out.write(1);
        
        // Tag: length and bytes
        writeString(out, tag);
        
        // Attribs: length (as short) and each attrib
        writeInt(out, sattribs.size()); 
        for (SerializableParsedAttrib attrib : sattribs) {
          attrib.writeObject(out);
        }
        
        // Children: length and each child
        writeInt(out, schildren.size());
        for (SerializableParsedNode node : schildren) {
          node.writeObject(out);
        }
      } else {
        // isTag
        out.write(0);
        
        // Text: length and bytes
        writeString(out, text);
      }
    }

    // Read the object from the InputStream, ensuring its byte representation
    // is compatible with the current deserialization routine.
    private static SerializableParsedNode readObject(InputStream in) throws IOException {
      int writtenVersion = readInt(in);
      if (writtenVersion != SERIALIZATION_VERSION_ID) {
        // Now-invalid entry. Serialization format has changed.
        return null;
      }
      
      boolean isTag = in.read() != 0;
      
      if (isTag) {
        // Read tag and iteratively read each attrib and child.
        String tag = readString(in);
        
        try {
          int attribLen = readInt(in);
          List<SerializableParsedAttrib> sattribs = new ArrayList<SerializableParsedAttrib>(attribLen);
          for (int i = 0; i < attribLen; ++i) {
            sattribs.add(SerializableParsedAttrib.readObject(in));
          }
        
          int childLen = readInt(in);
          List<SerializableParsedNode> schildren = new ArrayList<SerializableParsedNode>(childLen);
          for (int i = 0; i < childLen; ++i) {
            schildren.add(SerializableParsedNode.readObject(in));
          }
          
          return new SerializableParsedNode(tag, sattribs, schildren); 
        } catch (ClassCastException e) {
          throw new IOException("Class cast exception reading object: " + e);
        }
      } else {
        // Just read the text field.
        return new SerializableParsedNode(readString(in));
      }
    }
    
    // Create a wrapper node as a convenience for serializing a list of nodes.
    private static SerializableParsedNode makeParent(List<ParsedHtmlNode> kids) {
      return new SerializableParsedNode("x",
          new ArrayList<SerializableParsedAttrib>(), getSerializableNodeList(kids));
    }
    
    // Constructors: from generic ParsedHtmlNode, and for a new Tag or Text type node.
    private SerializableParsedNode(ParsedHtmlNode source) {
      this.tag = source.getTagName();
      this.sattribs = getSerializableAttribList(source.getAttributes());
      this.schildren = getSerializableNodeList(source.getChildren());
      this.text = source.getText();
    }
    
    private SerializableParsedNode(String tag,
        List<SerializableParsedAttrib> sattribs, List<SerializableParsedNode> schildren) {
      this.tag = tag;
      this.sattribs = sattribs;
      this.schildren = schildren;
    }
    
    private SerializableParsedNode(String text) {
      this.text = text;
    }
  }
  
  // Attribute equivalent of SerializableParsedNode
  private static class SerializableParsedAttrib implements ParsedHtmlAttribute {
    private String name;
    private String value;
    
    private SerializableParsedAttrib(ParsedHtmlAttribute source) {
      this.name = source.getName();
      this.value = source.getValue();
    }
    
    private SerializableParsedAttrib(String name, String value) {
      this.name = name;
      this.value = value;
    }

    public String getName() {
      return name;
    }

    public String getValue() {
      return value;
    }
    
    public void writeObject(OutputStream out) throws IOException {
      writeString(out, name);
      writeString(out, value);
    }
    
    private static SerializableParsedAttrib readObject(InputStream in) throws IOException {
      return new SerializableParsedAttrib(readString(in), readString(in));
    }
  }
}
