/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.oauth;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import net.oauth.OAuth;
import net.oauth.OAuthAccessor;
import net.oauth.OAuthException;
import net.oauth.OAuthMessage;
import net.oauth.OAuthProblemException;
import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.common.uri.UriBuilder;
import org.apache.shindig.common.util.CharsetUtil;
import org.apache.shindig.gadgets.ChainedContentFetcher;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.RequestSigningException;
import org.apache.shindig.gadgets.http.HttpCacheKey;
import org.apache.shindig.gadgets.http.HttpFetcher;
import org.apache.shindig.gadgets.http.HttpRequest;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.http.HttpResponseBuilder;
import org.apache.shindig.gadgets.oauth.AccessorInfo;
import org.apache.shindig.gadgets.oauth.OAuthClientState;
import org.apache.shindig.gadgets.oauth.OAuthError;
import org.apache.shindig.gadgets.oauth.OAuthFetcherConfig;
import org.apache.shindig.gadgets.oauth.OAuthProtocolException;
import org.apache.shindig.gadgets.oauth.OAuthResponseParams;
import org.apache.shindig.gadgets.oauth.OAuthStore;
import org.apache.shindig.gadgets.oauth.OAuthUtil;
import org.apache.shindig.gadgets.oauth.UserVisibleOAuthException;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class OAuthFetcher
extends ChainedContentFetcher {
    private static final Logger logger = Logger.getLogger(OAuthFetcher.class.getName());
    private static final int MAX_ATTEMPTS = 2;
    public static final String XOAUTH_APP_URL = "xoauth_app_url";
    protected static final String OPENSOCIAL_OWNERID = "opensocial_owner_id";
    protected static final String OPENSOCIAL_VIEWERID = "opensocial_viewer_id";
    protected static final String OPENSOCIAL_APPID = "opensocial_app_id";
    protected static final String OPENSOCIAL_APPURL = "opensocial_app_url";
    protected static final String XOAUTH_PUBLIC_KEY = "xoauth_signature_publickey";
    protected static final Pattern ALLOWED_PARAM_NAME = Pattern.compile("[-:\\w~!@$*()_\\[\\]:,./]+");
    protected final OAuthClientState clientState;
    protected final OAuthFetcherConfig fetcherConfig;
    protected final OAuthResponseParams responseParams;
    private AccessorInfo accessorInfo;
    private HttpRequest realRequest;

    public OAuthFetcher(OAuthFetcherConfig fetcherConfig, HttpFetcher nextFetcher, HttpRequest request) {
        super(nextFetcher);
        this.fetcherConfig = fetcherConfig;
        this.clientState = new OAuthClientState(fetcherConfig.getStateCrypter(), request.getOAuthArguments().getOrigClientState());
        this.responseParams = new OAuthResponseParams(fetcherConfig.getStateCrypter());
    }

    private void lookupOAuthMetadata() throws GadgetException {
        this.accessorInfo = this.fetcherConfig.getTokenStore().getOAuthAccessor(this.realRequest.getSecurityToken(), this.realRequest.getOAuthArguments(), this.clientState);
    }

    @Override
    public HttpResponse fetch(HttpRequest request) throws GadgetException {
        boolean retry;
        this.realRequest = request;
        HttpCacheKey cacheKey = this.makeCacheKey();
        HttpResponse response = this.fetcherConfig.getHttpCache().getResponse(cacheKey, request);
        if (response != null) {
            return response;
        }
        try {
            this.lookupOAuthMetadata();
        }
        catch (GadgetException e) {
            this.responseParams.setError(OAuthError.BAD_OAUTH_CONFIGURATION);
            return this.buildErrorResponse(e);
        }
        int attempts = 0;
        do {
            retry = false;
            ++attempts;
            try {
                response = this.attemptFetch();
            }
            catch (OAuthProtocolException pe) {
                retry = this.handleProtocolException(pe, attempts);
                if (retry) continue;
                response = pe.getResponseForGadget();
            }
        } while (retry);
        if (response == null) {
            throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR, "No response for OAuth fetch to " + this.realRequest.getUri());
        }
        return this.fetcherConfig.getHttpCache().addResponse(cacheKey, request, response);
    }

    private HttpCacheKey makeCacheKey() {
        HttpCacheKey key = new HttpCacheKey(this.realRequest);
        SecurityToken st = this.realRequest.getSecurityToken();
        key.set("authentication", "oauth");
        if (this.realRequest.getOAuthArguments().getSignOwner()) {
            key.set("owner", st.getOwnerId());
        }
        if (this.realRequest.getOAuthArguments().getSignViewer()) {
            key.set("viewer", st.getViewerId());
        }
        if (st.getOwnerId() != null && st.getOwnerId().equals(st.getViewerId()) && this.realRequest.getOAuthArguments().mayUseToken()) {
            key.set("tokenOwner", st.getOwnerId());
        }
        key.set("gadget", st.getAppUrl());
        key.set("instance", Long.toString(st.getModuleId()));
        key.set("service", this.realRequest.getOAuthArguments().getServiceName());
        key.set("token", this.realRequest.getOAuthArguments().getTokenName());
        return key;
    }

    private HttpResponse buildErrorResponse(GadgetException e) {
        if (this.responseParams.getError() == null) {
            this.responseParams.setError(OAuthError.UNKNOWN_PROBLEM);
        }
        if (this.responseParams.getErrorText() == null && e instanceof UserVisibleOAuthException) {
            this.responseParams.setErrorText(e.getMessage());
        }
        logger.log(Level.WARNING, "OAuth error", e);
        return this.buildNonDataResponse(403);
    }

    private boolean handleProtocolException(OAuthProtocolException pe, int attempts) throws GadgetException {
        if (pe.startFromScratch()) {
            this.fetcherConfig.getTokenStore().removeToken(this.realRequest.getSecurityToken(), this.accessorInfo.getConsumer(), this.realRequest.getOAuthArguments());
            this.accessorInfo.getAccessor().accessToken = null;
            this.accessorInfo.getAccessor().requestToken = null;
            this.accessorInfo.getAccessor().tokenSecret = null;
        }
        return attempts < 2 && pe.canRetry();
    }

    private HttpResponse attemptFetch() throws GadgetException, OAuthProtocolException {
        if (this.needApproval()) {
            this.checkCanApprove();
            this.fetchRequestToken();
            this.buildClientApprovalState();
            this.buildAznUrl();
            return this.buildOAuthApprovalResponse();
        }
        if (this.needAccessToken()) {
            this.checkCanApprove();
            this.exchangeRequestToken();
            this.saveAccessToken();
            this.buildClientAccessState();
        }
        return this.fetchData();
    }

    private boolean needApproval() {
        return this.realRequest.getOAuthArguments().mustUseToken() && this.accessorInfo.getAccessor().requestToken == null && this.accessorInfo.getAccessor().accessToken == null;
    }

    private void checkCanApprove() throws GadgetException {
        String pageOwner = this.realRequest.getSecurityToken().getOwnerId();
        String pageViewer = this.realRequest.getSecurityToken().getViewerId();
        String stateOwner = this.clientState.getOwner();
        if (!pageOwner.equals(pageViewer)) {
            throw new UserVisibleOAuthException("Only page owners can grant OAuth approval");
        }
        if (stateOwner != null && !stateOwner.equals(pageOwner)) {
            throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR, "Client state belongs to a different person.");
        }
    }

    private void fetchRequestToken() throws GadgetException, OAuthProtocolException {
        try {
            OAuthAccessor accessor = this.accessorInfo.getAccessor();
            HttpRequest request = new HttpRequest(Uri.parse((String)accessor.consumer.serviceProvider.requestTokenURL));
            request.setMethod(this.accessorInfo.getHttpMethod().toString());
            if (this.accessorInfo.getHttpMethod() == AccessorInfo.HttpMethod.POST) {
                request.setHeader("Content-Type", "application/x-www-form-urlencoded");
            }
            HttpRequest signed = this.sanitizeAndSign(request, null);
            OAuthMessage reply = this.sendOAuthMessage(signed);
            accessor.requestToken = OAuthUtil.getParameter(reply, "oauth_token");
            accessor.tokenSecret = OAuthUtil.getParameter(reply, "oauth_token_secret");
        }
        catch (OAuthException e) {
            throw new UserVisibleOAuthException(e.getMessage(), (Throwable)e);
        }
    }

    private List<OAuth.Parameter> sanitize(List<OAuth.Parameter> params) throws RequestSigningException {
        ArrayList<OAuth.Parameter> list = new ArrayList<OAuth.Parameter>();
        for (OAuth.Parameter p : params) {
            String name = p.getKey();
            if (this.allowParam(name)) {
                list.add(p);
                continue;
            }
            throw new RequestSigningException("invalid parameter name " + name);
        }
        return list;
    }

    private boolean allowParam(String paramName) {
        String canonParamName = paramName.toLowerCase();
        return !canonParamName.startsWith("oauth") && !canonParamName.startsWith("xoauth") && !canonParamName.startsWith("opensocial") && ALLOWED_PARAM_NAME.matcher(canonParamName).matches();
    }

    private void addIdentityParams(List<OAuth.Parameter> params) {
        String appUrl;
        String app;
        String viewer;
        String owner = this.realRequest.getSecurityToken().getOwnerId();
        if (owner != null && this.realRequest.getOAuthArguments().getSignOwner()) {
            params.add(new OAuth.Parameter(OPENSOCIAL_OWNERID, owner));
        }
        if ((viewer = this.realRequest.getSecurityToken().getViewerId()) != null && this.realRequest.getOAuthArguments().getSignViewer()) {
            params.add(new OAuth.Parameter(OPENSOCIAL_VIEWERID, viewer));
        }
        if ((app = this.realRequest.getSecurityToken().getAppId()) != null) {
            params.add(new OAuth.Parameter(OPENSOCIAL_APPID, app));
        }
        if ((appUrl = this.realRequest.getSecurityToken().getAppUrl()) != null) {
            params.add(new OAuth.Parameter(OPENSOCIAL_APPURL, appUrl));
        }
        if (this.accessorInfo.getConsumer().getConsumer().consumerKey == null) {
            params.add(new OAuth.Parameter("oauth_consumer_key", this.realRequest.getSecurityToken().getDomain()));
        }
    }

    private void addSignatureParams(List<OAuth.Parameter> params) {
        if (this.accessorInfo.getConsumer().getKeyName() != null) {
            params.add(new OAuth.Parameter(XOAUTH_PUBLIC_KEY, this.accessorInfo.getConsumer().getKeyName()));
        }
    }

    private String getAuthorizationHeader(List<Map.Entry<String, String>> oauthParams) {
        StringBuilder result = new StringBuilder("OAuth ");
        boolean first = true;
        for (Map.Entry<String, String> parameter : oauthParams) {
            if (!first) {
                result.append(", ");
            } else {
                first = false;
            }
            result.append(OAuth.percentEncode((String)parameter.getKey())).append("=\"").append(OAuth.percentEncode((String)parameter.getValue())).append('\"');
        }
        return result.toString();
    }

    public HttpRequest sanitizeAndSign(HttpRequest base, List<OAuth.Parameter> params) throws GadgetException {
        if (params == null) {
            params = new ArrayList<OAuth.Parameter>();
        }
        UriBuilder target = new UriBuilder(base.getUri());
        String query = target.getQuery();
        target.setQuery(null);
        params.addAll(this.sanitize(OAuth.decodeForm((String)query)));
        if (OAuth.isFormEncoded((String)base.getHeader("Content-Type"))) {
            params.addAll(this.sanitize(OAuth.decodeForm((String)base.getPostBodyAsString())));
        }
        this.addIdentityParams(params);
        this.addSignatureParams(params);
        try {
            OAuthMessage signed = OAuthUtil.newRequestMessage(this.accessorInfo.getAccessor(), base.getMethod(), target.toString(), params);
            HttpRequest oauthHttpRequest = this.createHttpRequest(base, this.selectOAuthParams(signed));
            oauthHttpRequest.setFollowRedirects(false);
            return oauthHttpRequest;
        }
        catch (OAuthException e) {
            throw new GadgetException(GadgetException.Code.INTERNAL_SERVER_ERROR, (Throwable)e);
        }
    }

    private HttpRequest createHttpRequest(HttpRequest base, List<Map.Entry<String, String>> oauthParams) throws GadgetException {
        AccessorInfo.OAuthParamLocation paramLocation = this.accessorInfo.getParamLocation();
        HttpRequest result = new HttpRequest(base);
        if (paramLocation == AccessorInfo.OAuthParamLocation.POST_BODY && !result.getMethod().equals("POST")) {
            paramLocation = AccessorInfo.OAuthParamLocation.AUTH_HEADER;
        }
        switch (paramLocation) {
            case AUTH_HEADER: {
                result.addHeader("Authorization", this.getAuthorizationHeader(oauthParams));
                break;
            }
            case POST_BODY: {
                String contentType = result.getHeader("Content-Type");
                if (!OAuth.isFormEncoded((String)contentType)) {
                    throw new UserVisibleOAuthException("OAuth param location can only be post_body if post body if of type x-www-form-urlencoded");
                }
                String oauthData = OAuthUtil.formEncode(oauthParams);
                if (result.getPostBodyLength() == 0) {
                    result.setPostBody(CharsetUtil.getUtf8Bytes((String)oauthData));
                    break;
                }
                result.setPostBody((result.getPostBodyAsString() + '&' + oauthData).getBytes());
                break;
            }
            case URI_QUERY: {
                result.setUri(Uri.parse((String)OAuthUtil.addParameters(result.getUri().toString(), oauthParams)));
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private OAuthMessage sendOAuthMessage(HttpRequest request) throws GadgetException, OAuthProtocolException, OAuthProblemException {
        HttpResponse response = this.nextFetcher.fetch(request);
        boolean done = false;
        try {
            this.checkForProtocolProblem(response);
            OAuthMessage reply = new OAuthMessage(null, null, null);
            reply.addParameters((Collection)OAuth.decodeForm((String)response.getResponseAsString()));
            reply = this.parseAuthHeader(reply, response);
            OAuthUtil.requireParameters(reply, "oauth_token", "oauth_token_secret");
            done = true;
            OAuthMessage oAuthMessage = reply;
            return oAuthMessage;
        }
        finally {
            if (!done) {
                this.logServiceProviderError(request, response);
            }
        }
    }

    private OAuthMessage parseAuthHeader(OAuthMessage msg, HttpResponse resp) {
        if (msg == null) {
            msg = new OAuthMessage(null, null, null);
        }
        for (String auth : resp.getHeaders("WWW-Authenticate")) {
            msg.addParameters((Collection)OAuthMessage.decodeAuthorization((String)auth));
        }
        return msg;
    }

    private void buildClientApprovalState() {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        this.responseParams.getNewClientState().setRequestToken(accessor.requestToken);
        this.responseParams.getNewClientState().setRequestTokenSecret(accessor.tokenSecret);
        this.responseParams.getNewClientState().setOwner(this.realRequest.getSecurityToken().getOwnerId());
    }

    private void buildAznUrl() {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        StringBuilder azn = new StringBuilder(accessor.consumer.serviceProvider.userAuthorizationURL);
        if (azn.indexOf("?") == -1) {
            azn.append('?');
        } else {
            azn.append('&');
        }
        azn.append("oauth_token");
        azn.append('=');
        azn.append(OAuth.percentEncode((String)accessor.requestToken));
        this.responseParams.setAznUrl(azn.toString());
    }

    private HttpResponse buildOAuthApprovalResponse() {
        return this.buildNonDataResponse(200);
    }

    private HttpResponse buildNonDataResponse(int status) {
        HttpResponseBuilder response = new HttpResponseBuilder().setHttpStatusCode(status);
        this.responseParams.addToResponse(response);
        response.setStrictNoCache();
        return response.create();
    }

    private boolean needAccessToken() {
        return this.realRequest.getOAuthArguments().mustUseToken() && this.accessorInfo.getAccessor().requestToken != null && this.accessorInfo.getAccessor().accessToken == null;
    }

    private void exchangeRequestToken() throws GadgetException, OAuthProtocolException {
        try {
            OAuthAccessor accessor = this.accessorInfo.getAccessor();
            HttpRequest request = new HttpRequest(Uri.parse((String)accessor.consumer.serviceProvider.accessTokenURL));
            request.setMethod(this.accessorInfo.getHttpMethod().toString());
            if (this.accessorInfo.getHttpMethod() == AccessorInfo.HttpMethod.POST) {
                request.setHeader("Content-Type", "application/x-www-form-urlencoded");
            }
            ArrayList<OAuth.Parameter> msgParams = new ArrayList<OAuth.Parameter>();
            msgParams.add(new OAuth.Parameter("oauth_token", accessor.requestToken));
            HttpRequest signed = this.sanitizeAndSign(request, msgParams);
            OAuthMessage reply = this.sendOAuthMessage(signed);
            accessor.accessToken = OAuthUtil.getParameter(reply, "oauth_token");
            accessor.tokenSecret = OAuthUtil.getParameter(reply, "oauth_token_secret");
        }
        catch (OAuthException e) {
            throw new UserVisibleOAuthException(e.getMessage(), (Throwable)e);
        }
    }

    private void saveAccessToken() throws GadgetException {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        OAuthStore.TokenInfo tokenInfo = new OAuthStore.TokenInfo(accessor.accessToken, accessor.tokenSecret);
        this.fetcherConfig.getTokenStore().storeTokenKeyAndSecret(this.realRequest.getSecurityToken(), this.accessorInfo.getConsumer(), this.realRequest.getOAuthArguments(), tokenInfo);
    }

    private void buildClientAccessState() {
        OAuthAccessor accessor = this.accessorInfo.getAccessor();
        this.responseParams.getNewClientState().setAccessToken(accessor.accessToken);
        this.responseParams.getNewClientState().setAccessTokenSecret(accessor.tokenSecret);
        this.responseParams.getNewClientState().setOwner(this.realRequest.getSecurityToken().getOwnerId());
    }

    private HttpResponse fetchData() throws GadgetException, OAuthProtocolException {
        HttpRequest signed = this.sanitizeAndSign(this.realRequest, null);
        HttpResponse response = this.nextFetcher.fetch(signed);
        try {
            this.checkForProtocolProblem(response);
        }
        catch (OAuthProtocolException e) {
            this.logServiceProviderError(signed, response);
            throw e;
        }
        HttpResponseBuilder builder = new HttpResponseBuilder(response);
        this.responseParams.addToResponse(builder);
        return builder.create();
    }

    private void checkForProtocolProblem(HttpResponse response) throws OAuthProtocolException {
        if (this.isFullOAuthError(response)) {
            OAuthMessage message = this.parseAuthHeader(null, response);
            if (OAuthUtil.getParameter(message, "oauth_problem") != null) {
                throw new OAuthProtocolException(message);
            }
            throw new OAuthProtocolException(response.getHttpStatusCode());
        }
    }

    private boolean isFullOAuthError(HttpResponse response) {
        if (response.getHttpStatusCode() != 400 && response.getHttpStatusCode() != 401 && response.getHttpStatusCode() != 403) {
            return false;
        }
        if (this.realRequest.getOAuthArguments().mustUseToken()) {
            return true;
        }
        return this.accessorInfo.getAccessor().accessToken != null;
    }

    private List<Map.Entry<String, String>> selectOAuthParams(OAuthMessage message) {
        ArrayList<Map.Entry<String, String>> result = new ArrayList<Map.Entry<String, String>>();
        for (Map.Entry<String, String> param : OAuthUtil.getParameters(message)) {
            if (!this.isContainerInjectedParameter(param.getKey())) continue;
            result.add(param);
        }
        return result;
    }

    private boolean isContainerInjectedParameter(String key) {
        return (key = key.toLowerCase()).startsWith("oauth") || key.startsWith("xoauth") || key.startsWith("opensocial");
    }

    private void logServiceProviderError(HttpRequest request, HttpResponse response) {
        logger.log(Level.INFO, "OAuth request failed:\n" + request + "\nresponse:\n" + response);
    }
}

