/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.hdfs.spout;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.AlreadyBeingCreatedException;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.storm.hdfs.common.HdfsUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileLock {
    private final FileSystem fs;
    private final String componentID;
    private final Path lockFile;
    private final FSDataOutputStream lockFileStream;
    private LogEntry lastEntry;
    private static final Logger LOG = LoggerFactory.getLogger(FileLock.class);

    private FileLock(FileSystem fs, Path lockFile, FSDataOutputStream lockFileStream, String spoutId) throws IOException {
        this.fs = fs;
        this.lockFile = lockFile;
        this.lockFileStream = lockFileStream;
        this.componentID = spoutId;
        this.logProgress("0", false);
    }

    private FileLock(FileSystem fs, Path lockFile, String spoutId, LogEntry entry) throws IOException {
        this.fs = fs;
        this.lockFile = lockFile;
        this.lockFileStream = fs.append(lockFile);
        this.componentID = spoutId;
        LOG.info("Acquired abandoned lockFile {}, Spout {}", (Object)lockFile, (Object)spoutId);
        this.logProgress(entry.fileOffset, true);
    }

    public void heartbeat(String fileOffset) throws IOException {
        this.logProgress(fileOffset, true);
    }

    private void logProgress(String fileOffset, boolean prefixNewLine) throws IOException {
        long now = System.currentTimeMillis();
        LogEntry entry = new LogEntry(now, this.componentID, fileOffset);
        String line = entry.toString();
        if (prefixNewLine) {
            this.lockFileStream.writeBytes(System.lineSeparator() + line);
        } else {
            this.lockFileStream.writeBytes(line);
        }
        this.lockFileStream.hflush();
        this.lastEntry = entry;
    }

    public void release() throws IOException {
        this.lockFileStream.close();
        if (!this.fs.delete(this.lockFile, false)) {
            LOG.warn("Unable to delete lock file, Spout = {}", (Object)this.componentID);
            throw new IOException("Unable to delete lock file");
        }
        LOG.debug("Released lock file {}. Spout {}", (Object)this.lockFile, (Object)this.componentID);
    }

    private void forceCloseLockFile() throws IOException {
        this.lockFileStream.close();
    }

    public static FileLock tryLock(FileSystem fs, Path fileToLock, Path lockDirPath, String spoutId) throws IOException {
        Path lockFile = new Path(lockDirPath, fileToLock.getName());
        try {
            FSDataOutputStream ostream = HdfsUtils.tryCreateFile(fs, lockFile);
            if (ostream != null) {
                LOG.debug("Acquired lock on file {}. LockFile= {}, Spout = {}", new Object[]{fileToLock, lockFile, spoutId});
                return new FileLock(fs, lockFile, ostream, spoutId);
            }
            LOG.debug("Cannot lock file {} as its already locked. Spout = {}", (Object)fileToLock, (Object)spoutId);
            return null;
        }
        catch (IOException e) {
            LOG.error("Error when acquiring lock on file " + fileToLock + " Spout = " + spoutId, (Throwable)e);
            throw e;
        }
    }

    public static LogEntry getLastEntryIfStale(FileSystem fs, Path lockFile, long olderThan) throws IOException {
        long modifiedTime = fs.getFileStatus(lockFile).getModificationTime();
        if (modifiedTime <= olderThan) {
            LogEntry lastEntry = FileLock.getLastEntry(fs, lockFile);
            if (lastEntry == null) {
                LOG.warn("Empty lock file found. Deleting it. {}", (Object)lockFile);
                try {
                    if (!fs.delete(lockFile, false)) {
                        throw new IOException("Empty lock file deletion failed");
                    }
                }
                catch (Exception e) {
                    LOG.error("Unable to delete empty lock file " + lockFile, (Throwable)e);
                }
            }
            if (lastEntry.eventTime <= olderThan) {
                return lastEntry;
            }
        }
        return null;
    }

    public static LogEntry getLastEntry(FileSystem fs, Path lockFile) throws IOException {
        FSDataInputStream in = fs.open(lockFile);
        BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)in));
        String lastLine = null;
        String line = reader.readLine();
        while (line != null) {
            lastLine = line;
            line = reader.readLine();
        }
        return LogEntry.deserialize(lastLine);
    }

    public static FileLock takeOwnership(FileSystem fs, Path lockFile, LogEntry lastEntry, String spoutId) throws IOException {
        try {
            if (fs instanceof DistributedFileSystem && !((DistributedFileSystem)fs).recoverLease(lockFile)) {
                LOG.warn("Unable to recover lease on lock file {} right now. Cannot transfer ownership. Will need to try later. Spout = {}", (Object)lockFile, (Object)spoutId);
                return null;
            }
            return new FileLock(fs, lockFile, spoutId, lastEntry);
        }
        catch (IOException e) {
            if (e instanceof RemoteException && ((RemoteException)((Object)e)).unwrapRemoteException() instanceof AlreadyBeingCreatedException) {
                LOG.warn("Lock file " + lockFile + "is currently open. Cannot transfer ownership now. Will need to try later. Spout= " + spoutId, (Throwable)e);
                return null;
            }
            LOG.warn("Cannot transfer ownership now for lock file " + lockFile + ". Will need to try later. Spout =" + spoutId, (Throwable)e);
            throw e;
        }
    }

    public static FileLock acquireOldestExpiredLock(FileSystem fs, Path lockFilesDir, int locktimeoutSec, String spoutId) throws IOException {
        long now = System.currentTimeMillis();
        long olderThan = now - (long)(locktimeoutSec * 1000);
        ArrayList<Path> listing = HdfsUtils.listFilesByModificationTime(fs, lockFilesDir, olderThan);
        for (Path file : listing) {
            FileLock lock;
            LogEntry lastEntry;
            if (file.getName().equalsIgnoreCase("DIRLOCK") || (lastEntry = FileLock.getLastEntryIfStale(fs, file, olderThan)) == null || (lock = FileLock.takeOwnership(fs, file, lastEntry, spoutId)) == null) continue;
            return lock;
        }
        if (listing.isEmpty()) {
            LOG.debug("No abandoned lock files found by Spout {}", (Object)spoutId);
        }
        return null;
    }

    public static HdfsUtils.Pair<Path, LogEntry> locateOldestExpiredLock(FileSystem fs, Path lockFilesDir, int locktimeoutSec) throws IOException {
        long now = System.currentTimeMillis();
        long olderThan = now - (long)(locktimeoutSec * 1000);
        ArrayList<Path> listing = HdfsUtils.listFilesByModificationTime(fs, lockFilesDir, olderThan);
        for (Path file : listing) {
            LogEntry lastEntry;
            if (file.getName().equalsIgnoreCase("DIRLOCK") || (lastEntry = FileLock.getLastEntryIfStale(fs, file, olderThan)) == null) continue;
            return new HdfsUtils.Pair<Path, LogEntry>(file, lastEntry);
        }
        LOG.debug("No abandoned files found");
        return null;
    }

    public LogEntry getLastLogEntry() {
        return this.lastEntry;
    }

    public Path getLockFile() {
        return this.lockFile;
    }

    public static class LogEntry {
        private static final int NUM_FIELDS = 3;
        public final long eventTime;
        public final String componentID;
        public final String fileOffset;

        public LogEntry(long eventtime, String componentID, String fileOffset) {
            this.eventTime = eventtime;
            this.componentID = componentID;
            this.fileOffset = fileOffset;
        }

        public String toString() {
            return this.eventTime + "," + this.componentID + "," + this.fileOffset;
        }

        public static LogEntry deserialize(String line) {
            String[] fields = line.split(",", 3);
            return new LogEntry(Long.parseLong(fields[0]), fields[1], fields[2]);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof LogEntry)) {
                return false;
            }
            LogEntry logEntry = (LogEntry)o;
            if (this.eventTime != logEntry.eventTime) {
                return false;
            }
            if (!this.componentID.equals(logEntry.componentID)) {
                return false;
            }
            return this.fileOffset.equals(logEntry.fileOffset);
        }

        public int hashCode() {
            int result = (int)(this.eventTime ^ this.eventTime >>> 32);
            result = 31 * result + this.componentID.hashCode();
            result = 31 * result + this.fileOffset.hashCode();
            return result;
        }
    }
}

