/*
 * Copyright (c) 2015, 2020 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package jakarta.security.enterprise.identitystore;

import java.util.Set;

import jakarta.security.auth.message.module.ServerAuthModule;

import jakarta.security.enterprise.CallerPrincipal;
import jakarta.security.enterprise.authentication.mechanism.http.HttpAuthenticationMechanism;
import jakarta.security.enterprise.authentication.mechanism.http.RememberMe;
import jakarta.security.enterprise.credential.RememberMeCredential;

/**
 * <code>RememberMeIdentityStore</code> is a mechanism for validating a caller's
 * credentials and accessing a caller's identity attributes that's specifically
 * tailored for the "remember me" feature.
 * 
 * <p>
 * This is not intended to be directly used by an authentication mechanism such as 
 * the Jakarta Security {@link HttpAuthenticationMechanism} or the Jakarta Authentication 
 * {@link ServerAuthModule}. Instead, the interceptor implementation backing the 
 * {@link RememberMe} annotation is intended to use this. 
 * 
 */
public interface RememberMeIdentityStore {

    /**
     * Validates the given credential.
     *
     * @param credential The credential to validate.
     * @return The validation result, including associated caller groups.
     */
    CredentialValidationResult validate(RememberMeCredential credential);
    
    /**
     * Associates the given principal and groups with a token.
     * <p>
     * The token generated by this method is intended to be used with the
     * {@link RememberMeCredential} and passed into the {@link #validate(RememberMeCredential)}
     * method.
     * 
     * @param callerPrincipal The principal to be associated.
     * @param groups The groups the principal is in.
     * @return A token that can be used to get the principal and groups back at a later time.
     */
    String generateLoginToken(CallerPrincipal callerPrincipal, Set<String> groups);
    
    /**
     * Dissociates the principal and groups that were associated with the token before
     * and removes the token itself.
     * <p>
     * If the token did not exist (i.e. no principal and groups were associated with that token)
     * no exception will be thrown.
     * 
     * @param token The token that is to be removed.
     */
    void removeLoginToken(String token);
    
}
