/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.crypto.sodium;

import com.google.common.base.Preconditions;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import javax.security.auth.Destroyable;
import jnr.ffi.Pointer;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.crypto.sodium.Allocated;
import org.apache.tuweni.crypto.sodium.Sodium;
import org.apache.tuweni.crypto.sodium.SodiumException;

public final class KeyDerivation {
    public static boolean isAvailable() {
        try {
            return Sodium.supportsVersion(Sodium.VERSION_10_0_12);
        }
        catch (UnsatisfiedLinkError e) {
            return false;
        }
    }

    private static void assertAvailable() {
        if (!KeyDerivation.isAvailable()) {
            throw new UnsupportedOperationException("Sodium key derivation is not available (requires sodium native library version >= 10.0.12)");
        }
    }

    public static int contextLength() {
        long contextbytes = Sodium.crypto_kdf_contextbytes();
        if (contextbytes > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("crypto_kdf_bytes_min: " + contextbytes + " is too large");
        }
        return (int)contextbytes;
    }

    public static int minSubKeyLength() {
        long length = Sodium.crypto_kdf_bytes_min();
        if (length > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("crypto_kdf_bytes_min: " + length + " is too large");
        }
        return (int)length;
    }

    public static int maxSubKeyLength() {
        long length = Sodium.crypto_kdf_bytes_max();
        if (length > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("crypto_kdf_bytes_max: " + length + " is too large");
        }
        return (int)length;
    }

    private static void assertContextLength(byte[] context) {
        long contextBytes = Sodium.crypto_kdf_contextbytes();
        if ((long)context.length != contextBytes) {
            throw new IllegalArgumentException("context must be " + contextBytes + " bytes, got " + context.length);
        }
    }

    private static void assertSubKeyLength(int length) {
        long minLength = Sodium.crypto_kdf_bytes_min();
        long maxLength = Sodium.crypto_kdf_bytes_max();
        if ((long)length < minLength || (long)length > maxLength) {
            throw new IllegalArgumentException("length is out of range [" + minLength + ", " + maxLength + "]");
        }
    }

    public static final class MasterKey
    implements Destroyable {
        final Allocated value;

        private MasterKey(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        @Override
        public void destroy() {
            this.value.destroy();
        }

        @Override
        public boolean isDestroyed() {
            return this.value.isDestroyed();
        }

        public static MasterKey fromBytes(Bytes bytes) {
            return MasterKey.fromBytes(bytes.toArrayUnsafe());
        }

        public static MasterKey fromBytes(byte[] bytes) {
            KeyDerivation.assertAvailable();
            if ((long)bytes.length != Sodium.crypto_kdf_keybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_kdf_keybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, MasterKey::new);
        }

        public static int length() {
            KeyDerivation.assertAvailable();
            long keybytes = Sodium.crypto_kdf_keybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_kdf_keybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public static MasterKey random() {
            KeyDerivation.assertAvailable();
            int length = MasterKey.length();
            Pointer ptr = Sodium.malloc(length);
            try {
                Sodium.randombytes_buf(ptr, length);
                return new MasterKey(ptr, length);
            }
            catch (Throwable e) {
                Sodium.sodium_free(ptr);
                throw e;
            }
        }

        public Bytes deriveKey(int length, long subkeyId, byte[] context) {
            return Bytes.wrap((byte[])this.deriveKeyArray(length, subkeyId, context));
        }

        public byte[] deriveKeyArray(int length, long subkeyId, byte[] context) {
            Preconditions.checkState((!this.value.isDestroyed() ? 1 : 0) != 0, (Object)"MasterKey has been destroyed");
            KeyDerivation.assertSubKeyLength(length);
            KeyDerivation.assertContextLength(context);
            byte[] subKey = new byte[length];
            int rc = Sodium.crypto_kdf_derive_from_key(subKey, subKey.length, subkeyId, context, this.value.pointer());
            if (rc != 0) {
                throw new SodiumException("crypto_kdf_derive_from_key: failed with result " + rc);
            }
            return subKey;
        }

        public Bytes deriveKey(int length, long subkeyId, String context) {
            return Bytes.wrap((byte[])this.deriveKeyArray(length, subkeyId, context));
        }

        public byte[] deriveKeyArray(int length, long subkeyId, String context) {
            int contextLen = KeyDerivation.contextLength();
            byte[] contextBytes = context.getBytes(StandardCharsets.UTF_8);
            if (context.length() > contextLen) {
                throw new IllegalArgumentException("context must be " + contextLen + " bytes, got " + context.length());
            }
            byte[] ctx = contextBytes.length == contextLen ? contextBytes : Arrays.copyOf(contextBytes, contextLen);
            return this.deriveKeyArray(length, subkeyId, ctx);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof MasterKey)) {
                return false;
            }
            MasterKey other = (MasterKey)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }
}

