//
// JODConverter - Java OpenDocument Converter
// Copyright 2004-2011 Mirko Nasato and contributors
//
// JODConverter is free software: you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation, either version 3 of
// the License, or (at your option) any later version.
//
// JODConverter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with JODConverter.  If not, see
// <http://www.gnu.org/licenses/>.
//
package org.artofsolving.jodconverter;

import static org.artofsolving.jodconverter.office.OfficeUtils.cast;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.artofsolving.jodconverter.document.DocumentFamily;
import org.artofsolving.jodconverter.document.DocumentFormat;
import org.artofsolving.jodconverter.office.OfficeContext;
import org.artofsolving.jodconverter.office.OfficeException;

import com.sun.star.awt.Size;
import com.sun.star.beans.PropertyValue;
import com.sun.star.beans.XPropertySet;
import com.sun.star.container.XIndexAccess;
import com.sun.star.graphic.XGraphicProvider;
import com.sun.star.lang.XComponent;
import com.sun.star.lang.XServiceInfo;
import com.sun.star.text.XTextGraphicObjectsSupplier;
import com.sun.star.util.XRefreshable;

public class StandardConversionTask extends AbstractConversionTask {

    private final DocumentFormat outputFormat;

    private Map<String,?> defaultLoadProperties;
    private DocumentFormat inputFormat;

    public StandardConversionTask(File inputFile, File outputFile, DocumentFormat outputFormat) {
        super(inputFile, outputFile);
        this.outputFormat = outputFormat;
    }

    public void setDefaultLoadProperties(Map<String, ?> defaultLoadProperties) {
        this.defaultLoadProperties = defaultLoadProperties;
    }

    public void setInputFormat(DocumentFormat inputFormat) {
        this.inputFormat = inputFormat;
    }

    @Override
    protected void modifyDocument(XComponent document, OfficeContext context) throws OfficeException {
        XRefreshable refreshable = cast(XRefreshable.class, document);
        if (refreshable != null) {
            refreshable.refresh();
        }

        if (cast(XServiceInfo.class, document).supportsService("com.sun.star.text.GenericTextDocument")) {
            embedWriterImages(document, context);
        }
    }

    private void embedWriterImages(XComponent document, OfficeContext context)
    {
        XIndexAccess indexAccess =
            cast(XIndexAccess.class, cast(XTextGraphicObjectsSupplier.class, document).getGraphicObjects());
        XGraphicProvider graphicProvider =
            cast(XGraphicProvider.class, context.getService("com.sun.star.graphic.GraphicProvider"));
        PropertyValue[] queryProperties = new PropertyValue[] {new PropertyValue()};
        queryProperties[0].Name = "URL";
        for (int i = 0; i < indexAccess.getCount(); i++) {
            try {
                XPropertySet graphicProperties = cast(XPropertySet.class, indexAccess.getByIndex(i));
                String graphicURL = (String) graphicProperties.getPropertyValue("GraphicURL");
                if (!graphicURL.contains("vnd.sun.star.GraphicObject")) {
                    queryProperties[0].Value = graphicURL;
                    // Before embedding the image, the "ActualSize" property holds the image size specified in the
                    // document content. If the width or height are not specified then their actual values will be 0.
                    Size specifiedSize = cast(Size.class, graphicProperties.getPropertyValue("ActualSize"));
                    graphicProperties.setPropertyValue("Graphic", graphicProvider.queryGraphic(queryProperties));
                    // Images are embedded as characters (see TextContentAnchorType.AS_CHARACTER) and their size is
                    // messed up if it's not explicitly specified (e.g. if the image height is not specified then it
                    // takes the line height).
                    adjustImageSize(graphicProperties, specifiedSize);
                }
            } catch (Exception e) {
                // Skip this graphic.
            }
        }
    }

    private void adjustImageSize(XPropertySet graphicProperties, Size specifiedSize)
    {
        try {
            // After embedding the image, the "ActualSize" property holds the actual image size.
            Size size = cast(Size.class, graphicProperties.getPropertyValue("ActualSize"));
            // Compute the width and height if not specified, preserving aspect ratio.
            if (specifiedSize.Width == 0 && specifiedSize.Height == 0) {
                specifiedSize.Width = size.Width;
                specifiedSize.Height = size.Height;
            } else if (specifiedSize.Width == 0) {
                specifiedSize.Width = specifiedSize.Height * size.Width / size.Height;
            } else if (specifiedSize.Height == 0) {
                specifiedSize.Height = specifiedSize.Width * size.Height / size.Width;
            }
            graphicProperties.setPropertyValue("Size", specifiedSize);
        } catch (Exception e) {
            // Ignore this image.
        }
    }

    @Override
    protected Map<String,?> getLoadProperties(File inputFile) {
        Map<String,Object> loadProperties = new HashMap<String,Object>();
        if (defaultLoadProperties != null) {
            loadProperties.putAll(defaultLoadProperties);
        }
        if (inputFormat != null && inputFormat.getLoadProperties() != null) {
            loadProperties.putAll(inputFormat.getLoadProperties());
        }
        return loadProperties;
    }

    @Override
    protected Map<String,?> getStoreProperties(File outputFile, XComponent document) {
        DocumentFamily family = OfficeDocumentUtils.getDocumentFamily(document);
        return outputFormat.getStoreProperties(family);
    }

}
