/*
 * This file is part of CycloneDX Core (Java).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) OWASP Foundation. All Rights Reserved.
 */
package org.cyclonedx.model.vulnerability;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlElementWrapper;
import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlProperty;
import org.cyclonedx.model.OrganizationalContact;
import org.cyclonedx.model.OrganizationalEntity;
import org.cyclonedx.model.Property;
import org.cyclonedx.model.Tool;
import org.cyclonedx.model.VersionFilter;
import org.cyclonedx.util.CustomDateSerializer;

/**
 * @since 6.0.0
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({
    "bomRef",
    "id",
    "source",
    "references",
    "ratings",
    "cwes",
    "description",
    "detail",
    "recommendation",
    "advisories",
    "created",
    "published",
    "updated",
    "credits",
    "tools",
    "analysis",
    "affects",
    "properties"
})
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public class Vulnerability
{
  public Vulnerability() {}

  @JacksonXmlProperty(isAttribute = true, localName = "bom-ref")
  @JsonProperty("bom-ref")
  private String bomRef;
  private String id;
  private Source source;
  private List<Reference> references;
  private List<Rating> ratings;
  private List<Integer> cwes;
  private String description;
  private String detail;
  private String recommendation;
  private List<Advisory> advisories;
  @JsonSerialize(using = CustomDateSerializer.class)
  @VersionFilter(versions = { "1.4"})
  private Date created;
  @JsonSerialize(using = CustomDateSerializer.class)
  @VersionFilter(versions = {"1.4"})
  private Date published;
  @JsonSerialize(using = CustomDateSerializer.class)
  @VersionFilter(versions = {"1.4"})
  private Date updated;
  private Credits credits;
  @JacksonXmlElementWrapper(localName = "tools")
  @JacksonXmlProperty(localName = "tool")
  private List<Tool> tools;
  private Analysis analysis;
  private List<Affect> affects;
  private List<Property> properties;

  public String getBomRef() {
    return bomRef;
  }

  public void setBomRef(final String bomRef) {
    this.bomRef = bomRef;
  }

  public String getId() {
    return id;
  }

  public void setId(final String id) {
    this.id = id;
  }

  public Source getSource() {
    return source;
  }

  public void setSource(final Source source) {
    this.source = source;
  }

  @JacksonXmlElementWrapper(localName = "references")
  @JacksonXmlProperty(localName = "reference")
  public List<Reference> getReferences() {
    return references;
  }

  public void setReferences(final List<Reference> references) {
    this.references = references;
  }

  @JacksonXmlElementWrapper(localName = "ratings")
  @JacksonXmlProperty(localName = "rating")
  public List<Rating> getRatings() {
    return ratings;
  }

  public void setRatings(final List<Rating> ratings) {
    this.ratings = ratings;
  }

  public void addRating(final Rating rating) {
    if (this.ratings == null) {
      this.ratings = new ArrayList<>();
    }
    this.ratings.add(rating);
  }

  @JacksonXmlElementWrapper(localName = "cwes")
  @JacksonXmlProperty(localName = "cwe")
  public List<Integer> getCwes() {
    return cwes;
  }

  public void setCwes(final List<Integer> cwes) {
    this.cwes = cwes;
  }

  public void addCwe(final int cweId) {
    if (this.cwes == null) {
      this.cwes = new ArrayList<>();
    }
    this.cwes.add(cweId);
  }

  public String getDescription() {
    return description;
  }

  public void setDescription(final String description) {
    this.description = description;
  }

  public String getDetail() {
    return detail;
  }

  public void setDetail(final String detail) {
    this.detail = detail;
  }

  public String getRecommendation() {
    return recommendation;
  }

  public void setRecommendation(final String recommendation) {
    this.recommendation = recommendation;
  }

  @JacksonXmlElementWrapper(localName = "advisories")
  @JacksonXmlProperty(localName = "advisory")
  public List<Advisory> getAdvisories() {
    return advisories;
  }

  public void setAdvisories(final List<Advisory> advisories) {
    this.advisories = advisories;
  }

  public Date getCreated() {
    return created;
  }

  public void setCreated(final Date created) {
    this.created = created;
  }

  public Date getPublished() {
    return published;
  }

  public void setPublished(final Date published) {
    this.published = published;
  }

  public Date getUpdated() {
    return updated;
  }

  public void setUpdated(final Date updated) {
    this.updated = updated;
  }

  public Credits getCredits() {
    return credits;
  }

  public void setCredits(final Credits credits) {
    this.credits = credits;
  }

  public List<Tool> getTools() {
    return tools;
  }

  public void setTools(final List<Tool> tools) {
    this.tools = tools;
  }

  public Analysis getAnalysis() {
    return analysis;
  }

  public void setAnalysis(final Analysis analysis) {
    this.analysis = analysis;
  }

  @JacksonXmlElementWrapper(localName = "affects")
  @JacksonXmlProperty(localName = "target")
  public List<Affect> getAffects() {
    return affects;
  }

  public void setAffects(final List<Affect> affects) {
    this.affects = affects;
  }

  @JacksonXmlElementWrapper(localName = "properties")
  @JacksonXmlProperty(localName = "property")
  public List<Property> getProperties() {
    return properties;
  }

  public void setProperties(final List<Property> properties) {
    this.properties = properties;
  }

  @JsonInclude(JsonInclude.Include.NON_NULL)
  public static class Reference {
    private String id;
    private Source source;

    public String getId() {
      return id;
    }

    public void setId(final String id) {
      this.id = id;
    }

    public Source getSource() {
      return source;
    }

    public void setSource(final Source source) {
      this.source = source;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_NULL)
  public static class Source {
    private String name;
    private String url;

    public String getName() {
      return name;
    }

    public void setName(final String name) {
      this.name = name;
    }

    public String getUrl() {
      return url;
    }

    public void setUrl(final String url) {
      this.url = url;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_NULL)
  public static class Advisory {
    private String title;
    private String url;

    public String getTitle() {
      return title;
    }

    public void setTitle(final String title) {
      this.title = title;
    }

    public String getUrl() {
      return url;
    }

    public void setUrl(final String url) {
      this.url = url;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_NULL)
  public static class Rating {
    private Source source;
    private Double score;

    public enum Severity {
      @JsonProperty("critical")
      CRITICAL("critical"),
      @JsonProperty("high")
      HIGH("high"),
      @JsonProperty("medium")
      MEDIUM("medium"),
      @JsonProperty("low")
      LOW("low"),
      @JsonProperty("info")
      INFO("info"),
      @JsonProperty("none")
      NONE("none"),
      @JsonProperty("unknown")
      UNKNOWN("unknown");

      private final String name;

      public String getSeverityName() {
        return this.name;
      }

      Severity(String name) {
        this.name = name;
      }

      public static Rating.Severity fromString(String text) {
        for (Rating.Severity s : Rating.Severity.values()) {
          if (s.name.equals(text)) {
            return s;
          }
        }
        return null;
      }
    }

    private Severity severity;

    public enum Method {
      @JsonProperty("CVSSv2")
      CVSSV2("CVSSv2"),
      @JsonProperty("CVSSv3")
      CVSSV3("CVSSv3"),
      @JsonProperty("CVSSv31")
      CVSSV31("CVSSv31"),
      @JsonProperty("OWASP")
      OWASP("OWASP"),
      @JsonProperty("other")
      OTHER("other");

      private final String name;

      public String getMethodName() {
        return this.name;
      }

      Method(String name) {
        this.name = name;
      }

      public static Rating.Method fromString(String text) {
        for (Rating.Method m : Rating.Method.values()) {
          if (m.name.equals(text)) {
            return m;
          }
        }
        return null;
      }
    }

    private Method method;
    private String vector;
    private String justification;

    public Source getSource() {
      return source;
    }

    public void setSource(final Source source) {
      this.source = source;
    }

    public Double getScore() {
      return score;
    }

    public void setScore(final Double score) {
      this.score = score;
    }

    public Severity getSeverity() {
      return severity;
    }

    public void setSeverity(final Severity severity) {
      this.severity = severity;
    }

    public Method getMethod() {
      return method;
    }

    public void setMethod(final Method method) {
      this.method = method;
    }

    public String getVector() {
      return vector;
    }

    public void setVector(final String vector) {
      this.vector = vector;
    }

    public String getJustification() {
      return justification;
    }

    public void setJustification(final String justification) {
      this.justification = justification;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_NULL)
  @JsonPropertyOrder({
      "state",
      "justification",
      "responses",
      "detail"})
  public static class Analysis {

    public enum State {
      @JsonProperty("resolved")
      RESOLVED("resolved"),
      @JsonProperty("resolved_with_pedigree")
      RESOLVED_WITH_PEDIGREE("resolved_with_pedigree"),
      @JsonProperty("exploitable")
      EXPLOITABLE("exploitable"),
      @JsonProperty("in_triage")
      IN_TRIAGE("in_triage"),
      @JsonProperty("false_positive")
      FALSE_POSITIVE("false_positive"),
      @JsonProperty("not_affected")
      NOT_AFFECTED("not_affected");

      private final String name;

      public String getStateName() {
        return this.name;
      }

      State(String name) {
        this.name = name;
      }

      public static Analysis.State fromString(String text) {
        for (Analysis.State a : Analysis.State.values()) {
          if (a.name.equals(text)) {
            return a;
          }
        }
        return null;
      }
    }

    private State state;

    public enum Justification {
      @JsonProperty("code_not_present")
      CODE_NOT_PRESENT("code_not_present"),
      @JsonProperty("code_not_reachable")
      CODE_NOT_REACHABLE("code_not_reachable"),
      @JsonProperty("requires_configuration")
      REQUIRES_CONFIGURATION("requires_configuration"),
      @JsonProperty("requires_dependency")
      REQUIRES_DEPENDENCY("requires_dependency"),
      @JsonProperty("requires_environment")
      REQUIRES_ENVIRONMENT("requires_environment"),
      @JsonProperty("protected_by_compiler")
      PROTECTED_BY_COMPILER("protected_by_compiler"),
      @JsonProperty("protected_at_runtime")
      PROTECTED_AT_RUNTIME("protected_at_runtime"),
      @JsonProperty("protected_at_perimeter")
      PROTECTED_AT_PERIMETER("protected_at_perimeter"),
      @JsonProperty("protected_by_mitigating_control")
      PROTECTED_BY_MITIGATING_CONTROL("protected_by_mitigating_control");

      private final String name;

      public String getJustificationName() {
        return this.name;
      }

      Justification(String name) {
        this.name = name;
      }

      public static Analysis.Justification fromString(String text) {
        for (Analysis.Justification j : Analysis.Justification.values()) {
          if (j.name.equals(text)) {
            return j;
          }
        }
        return null;
      }
    }

    private Justification justification;

    public enum Response {
      @JsonProperty("can_not_fix")
      CAN_NOT_FIX("can_not_fix"),
      @JsonProperty("will_not_fix")
      WILL_NOT_FIX("will_not_fix"),
      @JsonProperty("update")
      UPDATE("update"),
      @JsonProperty("rollback")
      ROLLBACK("rollback"),
      @JsonProperty("workaround_available")
      WORKAROUND_AVAILABLE("workaround_available");

      private final String name;

      public String getResponseName() {
        return this.name;
      }

      Response(String name) {
        this.name = name;
      }

      public static Analysis.Response fromString(String text) {
        for (Analysis.Response r : Analysis.Response.values()) {
          if (r.name.equals(text)) {
            return r;
          }
        }
        return null;
      }
    }

    private List<Response> responses;
    private String detail;

    public State getState() {
      return state;
    }

    public void setState(final State state) {
      this.state = state;
    }

    public Justification getJustification() {
      return justification;
    }

    public void setJustification(final Justification justification) {
      this.justification = justification;
    }

    @JacksonXmlElementWrapper(localName = "responses")
    @JsonProperty("response")
    @JacksonXmlProperty(localName = "response")
    public List<Response> getResponses() {
      return responses;
    }

    public void setResponses(final List<Response> responses) {
      this.responses = responses;
    }

    public String getDetail() {
      return detail;
    }

    public void setDetail(final String detail) {
      this.detail = detail;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_EMPTY)
  public static class Affect {
    private String ref;
    @JacksonXmlElementWrapper(localName = "versions")
    @JacksonXmlProperty(localName = "version")
    private List<Version> versions;

    public String getRef() {
      return ref;
    }

    public void setRef(final String ref) {
      this.ref = ref;
    }

    public List<Version> getVersions() {
      return versions;
    }

    public void setVersions(final List<Version> versions) {
      this.versions = versions;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_NULL)
  public static class Version {
    private String version;
    private String range;

    public enum Status {
      @JsonProperty("affected")
      AFFECTED("affected"),
      @JsonProperty("unaffected")
      UNAFFECTED("unaffected"),
      @JsonProperty("unknown")
      UNKNOWN("unknown");

      private final String name;

      public String getStatusName() {
        return this.name;
      }

      Status(String name) {
        this.name = name;
      }

      public static Version.Status fromString(String text) {
        for (Version.Status s : Version.Status.values()) {
          if (s.name.equals(text)) {
            return s;

          }
        }
        return null;
      }
    }

    private Status status;

    public String getVersion() {
      return version;
    }

    public void setVersion(final String version) {
      this.version = version;
    }

    public String getRange() {
      return range;
    }

    public void setRange(final String range) {
      this.range = range;
    }

    public Status getStatus() {
      return status;
    }

    public void setStatus(final Status status) {
      this.status = status;
    }
  }

  @JsonInclude(JsonInclude.Include.NON_EMPTY)
  public static class Credits
  {
    @JacksonXmlElementWrapper(localName = "organizations")
    @JacksonXmlProperty(localName = "organization")
    private List<OrganizationalEntity> organizations;
    @JacksonXmlElementWrapper(localName = "individuals")
    @JacksonXmlProperty(localName = "individual")
    private List<OrganizationalContact> individuals;

    public List<OrganizationalContact> getIndividuals() {
      return individuals;
    }

    public void setIndividuals(final List<OrganizationalContact> individuals) {
      this.individuals = individuals;
    }

    public List<OrganizationalEntity> getOrganizations() {
      return organizations;
    }

    public void setOrganizations(final List<OrganizationalEntity> organizations) {
      this.organizations = organizations;
    }
  }
}
