/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Erik Bengtson - added getObjectByAID
2004 Andy Jefferson - added MetaDataManager
2005 Andy Jefferson - javadocs
   ...
**********************************************************************/
package org.datanucleus;

import java.util.Collection;
import java.util.Set;

import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.ClassNotPersistableException;
import org.datanucleus.exceptions.NucleusOptimisticException;
import org.datanucleus.exceptions.NoPersistenceInformationException;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.state.CallbackHandler;
import org.datanucleus.state.FetchPlanState;
import org.datanucleus.store.FieldValues;
import org.datanucleus.store.StoreManager;

/**
 * Definition of an ObjectManager.
 * Provides the basics of object persistence internally, upon which are built
 * javax.jdo.PersistenceManager and javax.persistence.EntityManager.
 **/
public interface ObjectManager
{
    /**
     * Method to return the owner object.
     * For JDO this will return the PersistenceManager owning this ObjectManager.
     * For JPA this will return the EntityManager owning this ObjectManager.
     * @return The owner manager object
     */
    Object getOwner();

    /**
     * Method to close the ObjectManager.
     */
    void close();

    /**
     * Accessor for whether this ObjectManager is closed.
     * @return Whether this manager is closed.
     */
    boolean isClosed();

    /**
     * Accessor for the current transaction for this ObjectManager.
     * @return The current transaction
     */
    Transaction getTransaction();

    /**
     * Accessor for the Store Manager.
     * @return Store Manager
     */
    StoreManager getStoreManager();

    /**
     * Accessor for the MetaData Manager.
     * @return The MetaData Manager
     */
    MetaDataManager getMetaDataManager();

    /**
     * Accessor for the context in which this ObjectManager is running.
     * @return Returns the context.
     */
    OMFContext getOMFContext();

    /**
     * Accessor for the ObjectManagerFactory to which this manager belongs.
     * @return The OMF
     */
    ObjectManagerFactoryImpl getObjectManagerFactory();

    /**
     * Accessor for the API adapter.
     * @return API adapter.
     */
    ApiAdapter getApiAdapter();

    /**
     * Acessor for the current FetchPlan
     * @return FetchPlan
     */
    FetchPlan getFetchPlan();

    /**
     * Accessor for whether to ignore the cache.
     * @return Whether to ignore the cache.
     */
    boolean getIgnoreCache();

    /**
     * Accessor for whether the ObjectManager will detach all objects on commit
     * @return Whether we detach all on commit
     */
    public boolean getDetachAllOnCommit();

    /**
     * Accessor for whether the ObjectManager should copy on attaching.
     * @return Whether we copy on attaching
     */
    public boolean getCopyOnAttach();

    /**
     * Accessor for whether the ObjectManager is multithreaded
     * @return Whether the ObjectManager is multithreaded
     */
    public boolean getMultithreaded();

    /**
     * Method to set DetachAllOnCommit.
     * @param flag Whether to detach all objects on commit of the transaction.
     */
    public void setDetachAllOnCommit(boolean flag);

    /**
     * Method whether to copy on attaching.
     * @param flag Whether to copy on attaching.
     */
    public void setCopyOnAttach(boolean flag);

    /**
     * Method to set whether to ignore the L1 cache.
     * @param ignore Whether to ignore the L1 cache
     */
    public void setIgnoreCache(boolean ignore);

    /**
     * Method to set whether the ObjectManager should operate multithreaded
     * @param multi Whether to operate multithreaded
     */
    public void setMultithreaded(boolean multi);

    /**
     * Method to set DetachOnClose.
     * @param flag Whether to detach all objects on close of the ObjectManager.
     */
    void setDetachOnClose(boolean flag);

    /**
     * Accessor for the ClassLoader resolver to use in class loading issues.
     * @return The ClassLoader resolver
     */
    ClassLoaderResolver getClassLoaderResolver();

    /**
     * Retrieve the callback handler for this PM
     * @return the callback handler
     */
    CallbackHandler getCallbackHandler();

    /**
     * Convenience method to assert if the passed class is not persistable.
     * @param cls The class of which we want to persist objects
     * @throws ClassNotPersistableException When the class is not persistable
     * @throws NoPersistenceInformationException When the class has no available persistence information
     */
    void assertClassPersistable(Class cls);

    /**
     * Utility method to check if the specified class has reachable metadata or annotations.
     * @param cls The class to check
     * @return Whether the class has reachable metadata or annotations
     */
    public boolean hasPersistenceInformationForClass(Class cls);

    // -------------------------- Object Persistence operations --------------------------------

    /**
     * Method to evict the passed object.
     * @param pc The object
     */
    void evictObject(Object pc);

    /**
     * Method to evict all objects of the specified type (and optionaly its subclasses).
     * @param cls Type of persistable object
     * @param subclasses Whether to include subclasses
     */
    void evictObjects(Class cls, boolean subclasses);

    /**
     * Method to evict all L1 cache objects
     */
    void evictAllObjects();

    /**
     * Method to refresh the passed object.
     * @param pc The object
     */
    void refreshObject(Object pc);

    /**
     * Method to refresh all L1 cache objects
     */
    void refreshAllObjects();

    /**
     * Method to retrieve the passed object.
     * @param pc The object
     * @param fgOnly Just retrieve the current fetch group
     */
    void retrieveObject(Object pc, boolean fgOnly);

    /**
     * Method to persist the passed object.
     * @param pc The object
     * @return The persisted object
     */
    Object persistObject(Object pc);

    /**
     * Method to persist the passed object (internally).
     * @param pc The object
     * @param preInsertChanges Changes to be made before inserting
     * @param ownerSM StateManager of the owner when embedded
     * @param ownerFieldNum Field number in the owner where this is embedded (or -1 if not embedded)
     * @param objectType Type of object (see org.datanucleus.StateManager, e.g StateManager.PC)
     * @return The persisted object
     */
    Object persistObjectInternal(Object pc, FieldValues preInsertChanges, StateManager ownerSM, int ownerFieldNum, int objectType);

    /**
     * Method to make transient the passed object.
     * @param pc The object
     * @param state Object containing the state of the fetchplan processing
     */
    void makeObjectTransient(Object pc, FetchPlanState state);

    /**
     * Method to make the passed object transactional.
     * @param pc The object
     */
    void makeObjectTransactional(Object pc);

    /**
     * Method to make the passed object nontransactional.
     * @param pc The object
     */
    void makeObjectNontransactional(Object pc);

    /**
     * Method to delete an object from the datastore.
     * @param obj The object
     */
    void deleteObject(Object obj);

    /**
     * Method to delete the passed object (internally).
     * @param pc The object
     */
    void deleteObjectInternal(Object pc);

    /**
     * Method to delete an array of objects from the datastore.
     * @param objs The objects to delete
     */
    void deleteObjects(Object[] objs);

    /**
     * Method to delete a collection of objects from the datastore.
     * @param objs The objects to delete
     */
    void deleteObjects(Collection objs);

    /**
     * Method to attach the passed object (and related objects).
     * Throws an exception if another (persistent) object with the same id exists in the L1 cache already.
     * @param pc The (detached) object
     * @param sco Whether the object has no identity (embedded or serialised)
     */
    void attachObject(Object pc, boolean sco);

    /**
     * Method to attach a copy of the passed object (and related objects).
     * @param pc The object
     * @param sco Whether it has no identity (second-class object)
     * @return The attached copy of the input object
     */
    Object attachObjectCopy(Object pc, boolean sco);

    /**
     * Method to detach the passed object.
     * @param pc The object to detach
     * @param state State for the detachment process.
     */
    void detachObject(Object pc, FetchPlanState state);

    /**
     * Method to detach a copy of the passed object using the provided state.
     * @param pc The object
     * @param state State for the detachment process
     * @return The detached copy of the object
     */
    Object detachObjectCopy(Object pc, FetchPlanState state);

    /**
     * Method to detach all objects in the PM.
     */
    void detachAll();

    /**
     * Method to return if the specified object exists in the datastore.
     * @param obj The (persistable) object
     * @return Whether it exists
     */
    boolean exists(Object obj);

    /**
     * Accessor for the currently managed objects for the current transaction.
     * If the transaction is not active this returns null.
     * @return Collection of managed objects enlisted in the current transaction
     */
    Set getManagedObjects();

    /**
     * Accessor for the currently managed objects for the current transaction.
     * If the transaction is not active this returns null.
     * @param classes Classes that we want the objects for
     * @return Collection of managed objects enlisted in the current transaction
     */
    Set getManagedObjects(Class[] classes);

    /**
     * Accessor for the currently managed objects for the current transaction.
     * If the transaction is not active this returns null.
     * @param states States that we want the objects for
     * @return Collection of managed objects enlisted in the current transaction
     */
    Set getManagedObjects(String[] states);

    /**
     * Accessor for the currently managed objects for the current transaction.
     * If the transaction is not active this returns null.
     * @param states States that we want the objects for
     * @param classes Classes that we want the objects for
     * @return Collection of managed objects enlisted in the current transaction
     */
    Set getManagedObjects(String[] states, Class[] classes);

    /**
     * Accessor for an object given the object id.
     * @param id Id of the object.
     * @param fv the FieldValues
     * @return the Object
     */
    Object findObject(Object id, FieldValues fv);

    /**
     * Accessor for an object given the object id.
     * @param id Id of the object.
     * @param validate Whether to validate the object state
     * @param checkInheritance Whether look to the database to determine which
     * class this object is. This parameter is a hint. Set false, if it's
     * already determined the correct pcClass for this pc "object" in a certain
     * level in the hierarchy. Set to true and it will look to the database.
     * @param objectClassName Class name for the object with this id (if known, optional)
     * @return The Object
     */
    Object findObject(Object id, boolean validate, boolean checkInheritance, String objectClassName);

    /**
     * Accessor for an object given the object id.
     * @param id Id of the object.
     * @param fv the FieldValues
     * @param pcClass the type which the object is. This type will be used to instanciat the object
     * @param ignoreCache true if the cache is ignored
     * @return the Object
     */
    Object findObject(Object id, FieldValues fv, Class pcClass, boolean ignoreCache);

    /**
     * Accessor for an object given the object id. The object uses application identity
     * @param fv the FieldValues containing at minimum the primary key fields
     * @param pcClass the type which the object is
     * @param ignoreCache true if the cache is ignored
     * @param checkInheritance true if the cache is ignored
     * @return the Object
     */
    Object findObjectUsingAID(Class pcClass, FieldValues fv, boolean ignoreCache, boolean checkInheritance);

    /**
     * This method returns an object id instance corresponding to the pcClass and key arguments.
     * Operates in 2 modes :-
     * <ul>
     * <li>The class uses SingleFieldIdentity and the key is the value of the key field</li>
     * <li>In all other cases the key is the String form of the object id instance</li>
     * </ul>
     * @param pcClass Class of the PersistenceCapable to create the identity for
     * @param key Value of the key for SingleFieldIdentity (or the toString value)
     * @return The new object-id instance
     */
    Object newObjectId(Class pcClass, Object key);

    /**
     * This method returns an object id instance corresponding to the class name, and the passed
     * object (when using app identity).
     * @param className Name of the class of the object.
     * @param pc The persistable object. Used for application-identity
     * @return A new object ID.
     */
    Object newObjectId(String className, Object pc);

    /**
     * Method to generate an instance of an interface, abstract class, or concrete PC class.
     * @param persistenceCapable The class of the interface or abstract class, or concrete class defined in MetaData
     * @return The instance of this type
     */
    Object newInstance(Class persistenceCapable);

    /**
     * Method to enlist the specified StateManager in the current transaction.
     * @param sm The StateManager
     */
    void enlistInTransaction(StateManager sm);

    /**
     * Method to evict the specified StateManager from the current transaction.
     * @param sm The StateManager
     */
    void evictFromTransaction(StateManager sm);

    /**
     * Method to return if an object is enlisted in the current transaction.
     * @param id Identity for the object
     * @return Whether it is enlisted in the current transaction
     */
    boolean isEnlistedInTransaction(Object id);

    /**
     * Method to find the StateManager for the passed persistable object when it is managed by this manager.
     * @param pc The persistable object
     * @return The StateManager
     */
    StateManager findStateManager(Object pc);

    /**
     * Method to register the StateManager as being for the passed object.
     * Used during the process of identifying StateManager for persistable object.
     * @param sm The StateManager
     * @param pc The object managed by the StateManager
     */
    void hereIsStateManager(StateManager sm, Object pc);

    /**
     * Method to add the object managed by the specified StateManager to the cache.
     * @param sm The StateManager
     */    
    void addStateManager(StateManager sm);

    /**
     * Method to remove the object managed by the specified StateManager from the cache.
     * @param sm The StateManager
     */
    void removeStateManager(StateManager sm);

    /**
     * Accessor for the StateManager of an object given the object id.
     * @param myID Id of the object.
     * @return The StateManager
     */
    StateManager getStateManagerById(Object myID);

    /**
     * Mark the specified StateManager as dirty
     * @param sm The StateManager
     * @param directUpdate Whether the object has had a direct update made on it (if known)
     */
    void markDirty(StateManager sm, boolean directUpdate);

    /**
     * Mark the specified StateManager as clean.
     * @param sm The StateManager
     */
    void clearDirty(StateManager sm);

    /**
     * Method to mark as clean all StateManagers of dirty objects.
     */
    void clearDirty();

    /**
     * Method to mark the specified StateManager as needing an update due to managed relation constraints.
     * @param sm The StateManager
     */
    void markManagedRelationDirty(StateManager sm);

    /**
     * Returns whether this ObjectManager is currently performing the manage relationships task.
     * @return Whether in the process of managing relations
     */
    boolean isManagingRelations();

    /**
     * Tests whether this persistable object is being inserted.
     * @param pc the object to verify the status
     * @return true if this instance is inserting.
     */
    boolean isInserting(Object pc);

    /**
     * Whether the datastore operations are delayed until commit.
     * In optimistic transactions this is automatically enabled.
     * @return true if datastore operations are delayed until commit
     */
    boolean isDelayDatastoreOperationsEnabled();

    /**
     * Method called during the rollback process, after the actual datastore rollback.
     */
    public void preRollback();

    /**
     * Method called during the begin process, after the actual begin.
     */
    public void postBegin();
    
    /**
     * Method called during the commit process, before the actual datastore commit.
     */
    public void preCommit();

    /**
     * Method called during the commit process, after the actual datastore commit.
     */
    public void postCommit();

    /**
     * Method called during the close process.
     */
    void postClose();

    /**
     * Method callable from external APIs for user-management of flushing.
     * Called by JDO PM.flush, or JPA EM.flush().
     * Performs management of relations, prior to performing internal flush of all dirty/new/deleted
     * instances to the datastore.
     */
    void flush();

    /**
     * Method to flushes all dirty, new, and deleted instances to the datastore.
     * It has no effect if a transaction is not active. 
     * If a datastore transaction is active, this method synchronizes the cache with
     * the datastore and reports any exceptions. 
     * If an optimistic transaction is active, this method obtains a datastore connection
     * and synchronizes the cache with the datastore using this connection.
     * The connection obtained by this method is held until the end of the transaction.
     * @param flushToDatastore Whether to ensure any changes reach the datastore
     *     Otherwise they will be flushed to the datastore manager and leave it to
     *     decide the opportune moment to actually flush them to teh datastore
     * @throws NucleusOptimisticException when optimistic locking error(s) occur
     */
    void flushInternal(boolean flushToDatastore);

    /**
     * Accessor for whether the ObjectManager is flushing changes to the datastore.
     * @return Whether it is currently flushing
     */
    boolean isFlushing();

    /**
     * Accessor for whether this ObjectManager is currently running detachAllOnCommit.
     * @return Whether running detachAllOnCommit
     */
    boolean isRunningDetachAllOnCommit();

    // -------------------------------- Query Support -------------------------------------

    /**
     * Accessor for a new Query.
     * @return The new Query
     */
    org.datanucleus.store.query.Query newQuery();

    /**
     * Accessor for the Extent for a class (and optionally its subclasses).
     * @param candidateClass The class
     * @param includeSubclasses Whether to include subclasses
     * @return The Extent
     */
    org.datanucleus.store.Extent getExtent(Class candidateClass, boolean includeSubclasses);

    // --------------------------------- Caching ---------------------------------------

    /**
     * Replace the previous object id for a PC object to a new
     * @param pc The Persistable object
     * @param oldID the old id
     * @param newID the new id
     */
    void replaceObjectId(Object pc, Object oldID, Object newID);

    /**
     * Method to put a Persistable object associated to the StateManager into the L1 cache.
     * @param sm The State Manager
     */
    void putObjectIntoCache(StateManager sm);

    /**
     * Method to remove an object from the L1 cache.
     * @param pc The object
     * @param id The id of the object
     */
    void removeObjectFromCache(Object pc, Object id);

    /**
     * Convenience method to access an object in the cache.
     * Firstly looks in the L1 cache for this PM, and if not found looks in the L2 cache.
     * @param id Id of the object
     * @return Persistable object (with connected StateManager).
     */
    public Object getObjectFromCache(Object id);

    /**
     * Disconnect SM instances, clear cache and reset settings 
     */
    public void disconnectSMCache();

    // -------------------------------- Callback Interface --------------------------------------

    /**
     * Listener of ObjectManager events
     */
    public interface ObjectManagerListener
    {
        /**
         * Invoked before closing the ObjectManager
         * @param om the ObjectManager being closed
         */     
        void objectManagerPreClose(ObjectManager om);
    }
    
    /**
     * Method to register a listener for instances of the specified classes.
     * @param listener The listener to sends events to
     * @param classes The classes that it is interested in
     */
    void addListener(Object listener, Class[] classes);

    /**
     * Method to remove a currently registered listener.
     * @param listener The instance lifecycle listener to remove.
     */
    void removeListener(Object listener);

    /**
     * Disconnect the registered LifecycleListener
     */
    public void disconnectLifecycleListener();

    /**
     * Accessor for an internal fetch group for the specified class.
     * @param cls The class
     * @param name Name of the group
     * @return The FetchGroup
     */
    FetchGroup getInternalFetchGroup(Class cls, String name);

    /**
     * Method to add an internal fetch group to this ObjectManager.
     * @param grp The internal fetch group
     */
    void addInternalFetchGroup(FetchGroup grp);

    /**
     * Accessor for the fetch groups for the specified name.
     * @param name Name of the group
     * @return The FetchGroup
     */
    Set getFetchGroupsWithName(String name);
}