/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Kelly Grizzle (TJDO)
2003 Erik Bengtson - refactored the persistent id generator System property
2003 Erik Bengtson - added PMFConfiguration, PMFContext
2003 Andy Jefferson - commented, and added ApplicationId to supported options
2003 Andy Jefferson - introduction of localiser
2004 Andy Jefferson - update to getProperties() method
2004 Andy Jefferson - added LifecycleListener
2004 Andy Jefferson - added Level 2 Cache
2005 Marco Schulze - implemented copying the lifecycle listeners in j2ee environment
    ...
**********************************************************************/
package org.datanucleus.jdo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.jdo.FetchGroup;
import javax.jdo.JDOFatalUserException;
import javax.jdo.JDOUnsupportedOptionException;
import javax.jdo.JDOUserException;
import javax.jdo.datastore.DataStoreCache;
import javax.jdo.datastore.Sequence;
import javax.jdo.listener.InstanceLifecycleListener;
import javax.jdo.spi.JDOPermission;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.TransactionIsolationNotSupportedException;
import org.datanucleus.properties.CorePropertyValidator;
import org.datanucleus.store.StoreManager;
import org.datanucleus.util.Localiser;

/**
 * Factory used to obtain {@link javax.jdo.PersistenceManager} instances.
 * The factory is configurable up to a point when it is frozen. Thereafter nothing can be changed.
 */
public abstract class AbstractPersistenceManagerFactory extends ObjectManagerFactoryImpl
{
    /** Localisation utility for output messages from jdo. */
    protected static final Localiser LOCALISER_JDO = Localiser.getInstance("org.datanucleus.jdo.Localisation",
        AbstractPersistenceManagerFactory.class.getClassLoader());

    private static final String VERSION_NUMBER_PROPERTY = "VersionNumber";
    private static final String VENDOR_NAME_PROPERTY = "VendorName";

    /** The cache of PM's in use */
    private Set pmCache = new HashSet();

    /** Lifecycle Listeners */
    protected List lifecycleListeners;

    /** Map of user-defined sequences keyed by the factory class name. */
    private Map sequenceByFactoryClass;

    /** Level 2 Cache. */
    private DataStoreCache datastoreCache = null;

    /** JDO Fetch Groups. */
    private Set jdoFetchGroups = null;

    /**
     * Constructor.
     */
    public AbstractPersistenceManagerFactory()
    {
        super();

        try
        {
            Class cls = javax.jdo.PersistenceManager.class;
            cls.getMethod("detachCopy",new Class[]{Object.class});
        }
        catch (SecurityException e)
        {
            throw new JDOFatalUserException(LOCALISER_JDO.msg("012003"));
        }
        catch (NoSuchMethodException e)
        {
            throw new JDOFatalUserException(LOCALISER_JDO.msg("012003"));
        }
    }

    /**
     * Freezes the current configuration.
     * @throws NucleusException if the configuration was invalid or inconsistent in some way
     */
    protected void freezeConfiguration()
    {
        if (configurable)
        {
            try
            {
                // Log the PMF configuration
                logConfiguration();

                // Set user classloader
                ClassLoaderResolver clr = getOMFContext().getClassLoaderResolver(null);
                clr.registerUserClassLoader((ClassLoader)getProperty("datanucleus.primaryClassLoader"));

                // Set up the StoreManager
                initialiseStoreManager(clr);

                // Set up the Level 2 Cache
                initialiseLevel2Cache();

                if (cache != null)
                {
                    datastoreCache = new JDODataStoreCache(cache);
                }

                configurable = false;
            }
            catch (TransactionIsolationNotSupportedException inse)
            {
                throw new JDOUnsupportedOptionException(inse.getMessage());
            }
            catch (NucleusException jpe)
            {
                throw NucleusJDOHelper.getJDOExceptionForNucleusException(jpe);
            }
        }
    }

    /**
     * Return non-configurable properties of this PersistenceManagerFactory.
     * Properties with keys VendorName and VersionNumber are required.  Other keys are optional.
     * @return the non-configurable properties of this PersistenceManagerFactory.
     */
    public Properties getProperties()
    {
        Properties props = new Properties();

        props.setProperty(VENDOR_NAME_PROPERTY, getVendorName());
        props.setProperty(VERSION_NUMBER_PROPERTY, getVersionNumber());

        return props;
    }

    /**
     * The application can determine from the results of this method which
     * optional features, and which query languages are supported by the JDO
     * implementation. Se esection 11.6 of the JDO 2 specification.
     * @return A Collection of String representing the supported options.
     */
    public Collection supportedOptions()
    {
        // Generate the list of supported options, taking the general options we support and removing
        // any that the particular StoreManager doesn't support
        List options = new ArrayList(Arrays.asList(OPTION_ARRAY));
        StoreManager storeMgr = getOMFContext().getStoreManager();
        if (storeMgr != null)
        {
            Collection storeMgrOptions = storeMgr.getSupportedOptions();
            if (!storeMgrOptions.contains("DatastoreIdentity"))
            {
                options.remove("javax.jdo.option.DatastoreIdentity");
            }
            if (!storeMgrOptions.contains("ApplicationIdentity"))
            {
                options.remove("javax.jdo.option.ApplicationIdentity");
            }
            if (!storeMgrOptions.contains("OptimisticTransaction"))
            {
                options.remove("javax.jdo.option.Optimistic");
            }
            if (!storeMgr.supportsQueryLanguage("JDOQL"))
            {
                options.remove("javax.jdo.query.JDOQL");
            }
            if (!storeMgr.supportsQueryLanguage("SQL"))
            {
                options.remove("javax.jdo.query.SQL");
            }

            if (storeMgrOptions.contains("TransactionIsolationLevel.read-committed"))
            {
                options.add("javax.jdo.option.TransactionIsolationLevel.read-committed");
            }
            if (storeMgrOptions.contains("TransactionIsolationLevel.read-uncommitted"))
            {
                options.add("javax.jdo.option.TransactionIsolationLevel.read-uncommitted");
            }
            if (storeMgrOptions.contains("TransactionIsolationLevel.repeatable-read"))
            {
                options.add("javax.jdo.option.TransactionIsolationLevel.repeatable-read");
            }
            if (storeMgrOptions.contains("TransactionIsolationLevel.serializable"))
            {
                options.add("javax.jdo.option.TransactionIsolationLevel.serializable");
            }
            if (storeMgrOptions.contains("TransactionIsolationLevel.snapshot"))
            {
                options.add("javax.jdo.option.TransactionIsolationLevel.snapshot");
            }
        }

        return Collections.unmodifiableList(options);
    }

    /**
     * The JDO spec optional features that DataNucleus supports.
     * See JDO 2.0 spec section 11.6 for the full list of possibilities.
     **/
    private static final String[] OPTION_ARRAY =
    {
        "javax.jdo.option.TransientTransactional",
        "javax.jdo.option.NontransactionalWrite",
        "javax.jdo.option.NontransactionalRead",
        "javax.jdo.option.RetainValues",
        "javax.jdo.option.Optimistic",
        "javax.jdo.option.ApplicationIdentity",
        "javax.jdo.option.DatastoreIdentity",
        "javax.jdo.option.NonDurableIdentity",
        "javax.jdo.option.BinaryCompatibility",
        "javax.jdo.option.GetDataStoreConnection",
        "javax.jdo.option.GetJDBCConnection",
        "javax.jdo.option.version.DateTime",
        "javax.jdo.option.PreDirtyEvent",
        // "javax.jdo.option.ChangeApplicationIdentity",

        // Types
        "javax.jdo.option.ArrayList",
        "javax.jdo.option.LinkedList",
        "javax.jdo.option.TreeSet",
        "javax.jdo.option.TreeMap",
        "javax.jdo.option.Vector",
        "javax.jdo.option.List",
        "javax.jdo.option.Stack", // Not a listed JDO2 feature
        "javax.jdo.option.Map", // Not a listed JDO2 feature
        "javax.jdo.option.HashMap", // Not a listed JDO2 feature
        "javax.jdo.option.Hashtable", // Not a listed JDO2 feature
        "javax.jdo.option.SortedSet", // Not a listed JDO2 feature
        "javax.jdo.option.SortedMap", // Not a listed JDO2 feature
        "javax.jdo.option.Array",
        "javax.jdo.option.NullCollection",

        // ORM
        "javax.jdo.option.mapping.HeterogeneousObjectType",
        "javax.jdo.option.mapping.HeterogeneousInterfaceType",
        "javax.jdo.option.mapping.JoinedTablePerClass",
        "javax.jdo.option.mapping.JoinedTablePerConcreteClass",
        "javax.jdo.option.mapping.NonJoinedTablePerConcreteClass",
        // "javax.jdo.option.mapping.RelationSubclassTable", // Not yet supported for multiple subclasses

        // Query Languages
        "javax.jdo.query.SQL",
        "javax.jdo.query.JDOQL", // Not a listed optional feature
        "javax.jdo.option.UnconstrainedQueryVariables"
    };

    /**
     * Return the Set of PersistenceManagers actually in cache
     * @return Set this contains instances of PersistenceManagers in the cache
     */
    protected Set getPmCache()
    {
        return pmCache;
    }

    /**
     * Remove a PersistenceManager from the cache
     * Only the PersistenceManager is allowed to call this method
     * @param om  the PersistenceManager to be removed from cache
     */
    public void releasePersistenceManager(JDOPersistenceManager om)
    {
        getPmCache().remove(om);
    }

    /**
     * Asserts that the PMF is open.
     * @throws JDOUserException if it is already closed
     */
    protected void assertIsOpen()
    {
        try
        {
            super.assertIsOpen();
        }
        catch (NucleusException jpe)
        {
            // Comply with Section 11.4 of the JDO2 spec (throw JDOUserException if already closed)
            throw NucleusJDOHelper.getJDOExceptionForNucleusException(jpe);
        }
    }

    /**
     * Close this PersistenceManagerFactory. Check for JDOPermission("closePersistenceManagerFactory") 
     * and if not authorized, throw SecurityException.
     * <P>If the authorization check succeeds, check to see that all PersistenceManager instances obtained 
     * from this PersistenceManagerFactory have no active transactions. If any PersistenceManager instances
     * have an active transaction, throw a JDOUserException, with one nested JDOUserException for each 
     * PersistenceManager with an active Transaction.
     * <P>If there are no active transactions, then close all PersistenceManager instances obtained from 
     * this PersistenceManagerFactory, mark this PersistenceManagerFactory as closed, disallow 
     * getPersistenceManager methods, and allow all other get methods. If a set method or getPersistenceManager
     * method is called after close, then JDOUserException is thrown.
     * @see javax.jdo.PersistenceManagerFactory#close()
     */
    public synchronized void close()
    {
        assertIsOpen();

        SecurityManager secmgr = System.getSecurityManager();
        if (secmgr != null)
        {
            // checkPermission will throw SecurityException if not authorized
            secmgr.checkPermission(JDOPermission.CLOSE_PERSISTENCE_MANAGER_FACTORY);
        }

        // iterate though the list of pms to release resources
        Iterator pms = new HashSet(getPmCache()).iterator();
        Set exceptions = new HashSet();
        while (pms.hasNext())
        {
            try
            {
                ((JDOPersistenceManager)pms.next()).close();
            }
            catch (JDOUserException ex)
            {
                exceptions.add(ex);
            }
        }
        if (!exceptions.isEmpty())
        {
            throw new JDOUserException(LOCALISER_JDO.msg("012002"),(Throwable[])exceptions.toArray(new Throwable[exceptions.size()]));
        }

        if (jdoFetchGroups != null)
        {
            jdoFetchGroups.clear();
        }

        // Let superclass close its resources
        super.close();
    }

    /**
     * Accessor for the DataStore (level 2) Cache
     * @return The datastore cache
     * @since 1.1
     */
    public DataStoreCache getDataStoreCache()
    {
        freezeConfiguration();

        return datastoreCache;
    }

    /**
     * Set the user name for the data store connection.
     * @param userName the user name for the data store connection.
     */
    public synchronized void setConnectionUserName(String userName)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionUserName", userName);
    }

    /**
     * Set the password for the data store connection.
     * @param password the password for the data store connection.
     */
    public synchronized void setConnectionPassword(String password)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionPassword", password);
    }

    /**
     * Set the URL for the data store connection.
     * @param url the URL for the data store connection.
     */
    public synchronized void setConnectionURL(String url)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionURL", url);
    }

    /**
     * Set the driver name for the data store connection.
     * @param driverName the driver name for the data store connection.
     */
    public synchronized void setConnectionDriverName(String driverName)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionDriverName", driverName);
    }

    /**
     * Set the name for the data store connection factory.
     * @param connectionFactoryName name of the data store connection factory.
     */
    public synchronized void setConnectionFactoryName(String connectionFactoryName)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionFactoryName", connectionFactoryName);
    }

    /**
     * Set the data store connection factory.  JDO implementations
     * will support specific connection factories.  The connection
     * factory interfaces are not part of the JDO specification.
     * @param connectionFactory the data store connection factory.
     */
    public synchronized void setConnectionFactory(Object connectionFactory)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionFactory", connectionFactory);
    }

    /**
     * Set the name for the second data store connection factory.  This is
     * needed for managed environments to get nontransactional connections for
     * optimistic transactions.
     * @param connectionFactoryName name of the data store connection factory.
     */
    public synchronized void setConnectionFactory2Name(String connectionFactoryName)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionFactory2Name", connectionFactoryName);
    }

    /**
     * Set the second data store connection factory.  This is
     * needed for managed environments to get nontransactional connections for
     * optimistic transactions.  JDO implementations
     * will support specific connection factories.  The connection
     * factory interfaces are not part of the JDO specification.
     * @param connectionFactory the data store connection factory.
     */
    public synchronized void setConnectionFactory2(Object connectionFactory)
    {
        assertConfigurable();
        setProperty("datanucleus.ConnectionFactory2", connectionFactory);
    }

    /**
     * Set the default Multithreaded setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @param flag the default Multithreaded setting.
     */
    public synchronized void setMultithreaded(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.Multithreaded", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default Optimistic setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @param flag the default Optimistic setting.
     */
    public synchronized void setOptimistic(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.Optimistic", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default RetainValues setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @param flag the default RetainValues setting.
     */
    public synchronized void setRetainValues(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.RetainValues", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default RestoreValues setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @param flag the default RestoreValues setting.
     */
    public synchronized void setRestoreValues(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.RestoreValues", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default NontransactionalRead setting for all
     * <tt>PersistenceManager</tt> instances obtained from this factory.
     * @param flag the default NontransactionalRead setting.
     */
    public synchronized void setNontransactionalRead(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.NontransactionalRead", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default NontransactionalWrite setting for all
     * <tt>PersistenceManager</tt> instances obtained from this factory.
     * @param flag the default NontransactionalWrite setting.
     */
    public synchronized void setNontransactionalWrite(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.NontransactionalWrite", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default IgnoreCache setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @param flag the default IgnoreCache setting.
     */
    public synchronized void setIgnoreCache(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.IgnoreCache", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Mutator for the DetachAllOnCommit setting.
     * @param flag the default DetachAllOnCommit setting.
     * @since 1.1
     */
    public synchronized void setDetachAllOnCommit(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.DetachAllOnCommit", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Mutator for the CopyOnAttach setting.
     * @param flag the default CopyOnAttach setting.
     * @since 1.2
     */
    public synchronized void setCopyOnAttach(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.CopyOnAttach", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the name for any mapping, used in searching for ORM/Query metadata files.
     * @param mapping the mapping name
     */
    public synchronized void setMapping(String mapping)
    {
        assertConfigurable();
        setProperty("datanucleus.Mapping", mapping);
    }

    /**
     * Mutator for the catalog to use for this persistence factory.
     * @param catalog Name of the catalog
     * @since 1.1
     */
    public synchronized void setCatalog(String catalog)
    {
        assertConfigurable();
        setProperty("datanucleus.mapping.Catalog", catalog);
    }

    /**
     * Mutator for the schema to use for this persistence factory.
     * @param schema Name of the schema
     * @since 1.1
     */
    public synchronized void setSchema(String schema)
    {
        assertConfigurable();
        setProperty("datanucleus.mapping.Schema", schema);
    }

    /**
     * Mutator for the transaction type to use for this persistence factory.
     * @param type Transaction type
     */
    public synchronized void setTransactionType(String type)
    {
        assertConfigurable();
        boolean validated = new CorePropertyValidator().validate("datanucleus.TransactionType", type);
        if (validated)
        {
            setProperty("datanucleus.TransactionType", type);
        }
        else
        {
            throw new JDOUserException(LOCALISER.msg("008012", "javax.jdo.option.TransactionType", type));
        }
    }

    /**
     * Mutator for the name of the persistence unit.
     * @param name Name of the persistence unit
     */
    public synchronized void setPersistenceUnitName(String name)
    {
        assertConfigurable();
        setProperty("datanucleus.PersistenceUnitName", name);
    }

    /**
     * Mutator for the filename of the persistence.xml file.
     * This is for the case where an application has placed the persistence.xml somewhere else maybe
     * outside the CLASSPATH.
     * @param name Filename of the persistence unit
     */
    public synchronized void setPersistenceXmlFilename(String name)
    {
        assertConfigurable();
        setProperty("datanucleus.persistenceXmlFilename", name);
    }

    /**
     * Mutator for the name of the persistence factory.
     * @param name Name of the persistence factory (if any)
     */
    public synchronized void setName(String name)
    {
        assertConfigurable();
        setProperty("datanucleus.Name", name);
    }

    /**
     * Mutator for the timezone id of the datastore server.
     * If not set assumes that it is running in the same timezone as this JVM.
     * @param id Timezone Id to use
     */
    public void setServerTimeZoneID(String id)
    {
        boolean validated = new CorePropertyValidator().validate("datanucleus.ServerTimeZoneID", id);
        if (validated)
        {
            setProperty("datanucleus.ServerTimeZoneID", id);
        }
        else
        {
            throw new JDOUserException("Invalid TimeZone ID specified");
        }
    }

    /**
     * Set the readOnly setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @param flag the default readOnly setting.
     */
    public synchronized void setReadOnly(boolean flag)
    {
        assertConfigurable();
        setProperty("datanucleus.ReadOnly", flag ? Boolean.TRUE : Boolean.FALSE);
    }

    /**
     * Set the default isolation level for transactions.
     * @param level Level
     */
    public void setTransactionIsolationLevel(String level)
    {
        assertConfigurable();

        if (omfContext != null && omfContext.getStoreManager() != null &&
            !omfContext.getStoreManager().getSupportedOptions().contains("TransactionIsolationLevel." + level))
        {
            throw new JDOUnsupportedOptionException("Isolation level \"" + level + "\" is not supported for this datastore");
        }

        // Reset to "read-committed" if passed in as null
        setProperty("datanucleus.transactionIsolation", level != null ? level : "read-committed");
    }

    /**
     * Get the user name for the data store connection.
     * @return the user name for the data store connection.
     */
    public String getConnectionUserName()
    {
        return getStringProperty("datanucleus.ConnectionUserName");
    }

    /**
     * Get the password for the data store connection.
     * @return the password for the data store connection.
     */
    public String getConnectionPassword()
    {
        return getStringProperty("datanucleus.ConnectionPassword");
    }

    /**
     * Get the URL for the data store connection.
     * @return the URL for the data store connection.
     */
    public String getConnectionURL()
    {
        return getStringProperty("datanucleus.ConnectionURL");
    }

    /**
     * Get the driver name for the data store connection.
     * @return the driver name for the data store connection.
     */
    public String getConnectionDriverName()
    {
        return getStringProperty("datanucleus.ConnectionDriverName");
    }

    /**
     * Get the name for the data store connection factory.
     * @return the name of the data store connection factory.
     */
    public String getConnectionFactoryName()
    {
        return getStringProperty("datanucleus.ConnectionFactoryName");
    }

    /**
     * Get the name for the second data store connection factory.  This is
     * needed for managed environments to get nontransactional connections for
     * optimistic transactions.
     * @return the name of the data store connection factory.
     */
    public String getConnectionFactory2Name()
    {
        return getStringProperty("datanucleus.ConnectionFactory2Name");
    }

    /**
     * Get the data store connection factory.
     * @return the data store connection factory.
     */
    public Object getConnectionFactory()
    {
        return getProperty("datanucleus.ConnectionFactory");
    }

    /**
     * Get the second data store connection factory.  This is
     * needed for managed environments to get nontransactional connections for
     * optimistic transactions.
     * @return the data store connection factory.
     */
    public Object getConnectionFactory2()
    {
        return getProperty("datanucleus.ConnectionFactory2");
    }

    /**
     * Get the default Multithreaded setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @return the default Multithreaded setting.
     */
    public boolean getMultithreaded()
    {
        return getBooleanProperty("datanucleus.Multithreaded");
    }

    /**
     * Get the default Optimistic setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @return the default Optimistic setting.
     */
    public boolean getOptimistic()
    {
        return getBooleanProperty("datanucleus.Optimistic");
    }

    /**
     * Get the default RetainValues setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @return the default RetainValues setting.
     */
    public boolean getRetainValues()
    {
        return getBooleanProperty("datanucleus.RetainValues");
    }

    /**
     * Get the default RestoreValues setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @return the default RestoreValues setting.
     */
    public boolean getRestoreValues()
    {
        return getBooleanProperty("datanucleus.RestoreValues");
    }

    /**
     * Get the default NontransactionalRead setting for all
     * <tt>PersistenceManager</tt> instances obtained from this factory.
     * @return the default NontransactionalRead setting.
     */
    public boolean getNontransactionalRead()
    {
        return getBooleanProperty("datanucleus.NontransactionalRead");
    }

    /**
     * Get the default NontransactionalWrite setting for all
     * <tt>PersistenceManager</tt> instances obtained from this factory.
     * @return the default NontransactionalWrite setting.
     */
    public boolean getNontransactionalWrite()
    {
        return getBooleanProperty("datanucleus.NontransactionalWrite");
    }

    /**
     * Get the default IgnoreCache setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @return the IgnoreCache setting.
     */
    public boolean getIgnoreCache()
    {
        return getBooleanProperty("datanucleus.IgnoreCache");
    }

    /**
     * Accessor for the DetachAllOnCommit setting.
     * @return the DetachAllOnCommit setting.
     */
    public boolean getDetachAllOnCommit()
    {
        return getBooleanProperty("datanucleus.DetachAllOnCommit");
    }

    /**
     * Accessor for the CopyOnAttach setting.
     * @return the CopyOnAttach setting.
     */
    public boolean getCopyOnAttach()
    {
        return getBooleanProperty("datanucleus.CopyOnAttach");
    }

    /**
     * Get the name for any mapping, used in retrieving metadata files for ORM/Query data.
     * @return the name for the mapping.
     */
    public String getMapping()
    {
        return getStringProperty("datanucleus.Mapping");
    }

    /**
     * Accessor for the catalog to use for this persistence factory.
     * @return the name of the catalog
     */
    public String getCatalog()
    {
        return getStringProperty("datanucleus.mapping.Catalog");
    }

    /**
     * Accessor for the schema to use for this persistence factory.
     * @return the name of the schema
     */
    public String getSchema()
    {
        return getStringProperty("datanucleus.mapping.Schema");
    }

    /**
     * Accessor for the name of the persistence factory (if any).
     * @return the name of the persistence factory
     */
    public String getName()
    {
        return getStringProperty("datanucleus.Name");
    }

    /**
     * Accessor for the name of the persistence unit
     * @return the name of the persistence unit
     */
    public String getPersistenceUnitName()
    {
        return getStringProperty("datanucleus.PersistenceUnitName");
    }

    /**
     * Accessor for the filename of the persistence.xml file.
     * This is for the case where an application has placed the persistence.xml somewhere else maybe
     * outside the CLASSPATH.
     * @return the filename of the persistence unit
     */
    public String getPersistenceXmlFilename()
    {
        return getStringProperty("datanucleus.persistenceXmlFilename");
    }

    /**
     * Accessor for the timezone "id" of the datastore server (if any).
     * If not set assumes the same as the JVM we are running in.
     * @return Server timezone id
     */
    public String getServerTimeZoneID()
    {
        return getStringProperty("datanucleus.ServerTimeZoneID");
    }

    /**
     * Get the default readOnly setting for all <tt>PersistenceManager</tt>
     * instances obtained from this factory.
     * @return the default readOnly setting.
     */
    public boolean getReadOnly()
    {
        return getBooleanProperty("datanucleus.ReadOnly");
    }

    /**
     * Accessor for the transaction type to use with this persistence factory.
     * @return transaction type
     */
    public String getTransactionType()
    {
        return getStringProperty("datanucleus.TransactionType");
    }

    /**
     * Accessor for the transaction isolation level default.
     * @return Transaction isolation level
     */
    public String getTransactionIsolationLevel()
    {
        return getStringProperty("datanucleus.transactionIsolation");
    }

    /**
     * Asserts that a change to a configuration property is allowed.
     */
    protected void assertConfigurable()
    {
        if (!configurable)
        {
            throw new JDOUserException(LOCALISER.msg("008016"));
        }
    }

    // -------------------------------- Lifecycle Listeners -------------------------------

    /**
     * @return Returns either <tt>null</tt> or a <tt>List</tt> with instances of
     *		<tt>LifecycleListenerSpecification</tt>.
     */
    public List getLifecycleListenerSpecifications()
    {
    	return lifecycleListeners;
    }

    /**
     * Method to add lifecycle listeners for particular classes.
     * Adds the listener to all PMs already created.
     * @param listener The listener
     * @param classes The classes that the listener relates to
     * @since 1.1
     */
    public void addInstanceLifecycleListener(InstanceLifecycleListener listener, Class[] classes)
    {
        if (listener == null)
        {
            return;
        }

        synchronized (pmCache)
        {
            // Add to any PMs - make sure that the PM Cache isnt changed while doing so
            Iterator pms = getPmCache().iterator();
            while (pms.hasNext())
            {
                JDOPersistenceManager pm = (JDOPersistenceManager)pms.next();
                pm.addInstanceLifecycleListener(listener, classes);
            }
        }

        // Register the lifecycle listener
        if (lifecycleListeners == null)
        {
            lifecycleListeners = new ArrayList(5);
        }
        lifecycleListeners.add(new LifecycleListenerForClass(listener, classes));
    }

    /**
     * Method to remove a lifecycle listener. Removes the listener from all PM's as well.
     * @param listener The Listener
     * @since 1.1
     */
    public void removeInstanceLifecycleListener(InstanceLifecycleListener listener)
    {
        if (listener == null || lifecycleListeners == null)
        {
            return;
        }

        synchronized (pmCache)
        {
            // Remove from any PMs - make sure that the PM Cache isnt changed while doing so
            Iterator pms = getPmCache().iterator();
            while (pms.hasNext())
            {
                JDOPersistenceManager pm = (JDOPersistenceManager)pms.next();
                pm.removeInstanceLifecycleListener(listener);
            }
        }

        // Remove from the PMF
        Iterator iter = lifecycleListeners.iterator();
        while (iter.hasNext())
        {
            LifecycleListenerForClass classListener = (LifecycleListenerForClass) iter.next();
            if (classListener.getListener() == listener)
            {
                iter.remove();
            }
        }
    }

    // --------------------------- Sequences ----------------------------------

    /**
     * Method to register a sequence for a particular factory class.
     * @param factoryClassName Name of the factory class
     * @param sequence The sequence
     */
    public void addSequenceForFactoryClass(String factoryClassName, Sequence sequence)
    {
        if (sequenceByFactoryClass == null)
        {
            sequenceByFactoryClass = new HashMap();
        }

        sequenceByFactoryClass.put(factoryClassName, sequence);
    }

    /**
     * Accessor for the sequence for a factory class.
     * @param factoryClassName The name of the factory class
     * @return The sequence
     */
    public Sequence getSequenceForFactoryClass(String factoryClassName)
    {
        if (sequenceByFactoryClass == null)
        {
            return null;
        }

        return (Sequence)sequenceByFactoryClass.get(factoryClassName);
    }

    // --------------------------- Fetch Groups ----------------------------------

    /* (non-Javadoc)
     * @see javax.jdo.PersistenceManagerFactory#getFetchGroups()
     */
    public Set getFetchGroups()
    {
        if (jdoFetchGroups == null)
        {
            return null;
        }

        // TODO Should return a copy
        return jdoFetchGroups;
    }

    /* (non-Javadoc)
     * @see javax.jdo.PersistenceManagerFactory#getFetchGroup(java.lang.Class, java.lang.String)
     */
    public javax.jdo.FetchGroup getFetchGroup(Class cls, String name)
    {
        if (jdoFetchGroups == null)
        {
            jdoFetchGroups = new HashSet();
        }

        Iterator iter = jdoFetchGroups.iterator();
        while (iter.hasNext())
        {
            JDOFetchGroup jdoGrp = (JDOFetchGroup)iter.next();
            if (jdoGrp.getName().equals(name) && jdoGrp.getType() == cls)
            {
                return jdoGrp;
            }
        }

        // Create new FetchGroup, but don't add to set of groups yet - user should add via addFetchGroups()
        try
        {
            org.datanucleus.FetchGroup internalGrp = getInternalFetchGroup(cls, name);
            if (!internalGrp.isUnmodifiable())
            {
                // Return existing internal group since still modifiable
                return new JDOFetchGroup(internalGrp);
            }
            else
            {
                // Create a new internal group (modifiable) and return a JDO group based on that
                internalGrp = createInternalFetchGroup(cls, name);
                addInternalFetchGroup(internalGrp);
                JDOFetchGroup jdoGrp = new JDOFetchGroup(internalGrp);
                return jdoGrp;
            }
        }
        catch (NucleusException ne)
        {
            throw NucleusJDOHelper.getJDOExceptionForNucleusException(ne);
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.PersistenceManagerFactory#addFetchGroups(javax.jdo.FetchGroup[])
     */
    public void addFetchGroups(FetchGroup[] groups)
    {
        if (groups == null || groups.length == 0)
        {
            return;
        }

        if (jdoFetchGroups == null)
        {
            jdoFetchGroups = new HashSet();
        }
        for (int i=0;i<groups.length;i++)
        {
            addInternalFetchGroup(((JDOFetchGroup)groups[i]).getInternalFetchGroup());
            jdoFetchGroups.add(groups[i]);
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.PersistenceManagerFactory#removeFetchGroups(javax.jdo.FetchGroup[])
     */
    public void removeFetchGroups(FetchGroup[] groups)
    {
        if (groups == null || groups.length == 0 || jdoFetchGroups == null)
        {
            return;
        }

        for (int i=0;i<groups.length;i++)
        {
            JDOFetchGroup jdoGrp = (JDOFetchGroup)groups[i];
            removeInternalFetchGroup(jdoGrp.getInternalFetchGroup());
            jdoFetchGroups.remove(jdoGrp);
        }
    }

    /* (non-Javadoc)
     * @see javax.jdo.PersistenceManagerFactory#removeAllFetchGroups()
     */
    public void removeAllFetchGroups()
    {
        if (jdoFetchGroups != null)
        {
            Iterator iter = jdoFetchGroups.iterator();
            while (iter.hasNext())
            {
                JDOFetchGroup jdoGrp = (JDOFetchGroup)iter.next();
                removeInternalFetchGroup(jdoGrp.getInternalFetchGroup());
            }
            jdoFetchGroups.clear();
            jdoFetchGroups = null;
        }
    }
}