/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.jdo;

import java.util.Properties;

import javax.jdo.JDODataStoreException;
import javax.jdo.datastore.Sequence;

import org.datanucleus.ManagedConnection;
import org.datanucleus.ObjectManager;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.ExtensionMetaData;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.plugin.ConfigurationElement;
import org.datanucleus.store.NucleusSequence;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.valuegenerator.ValueGenerationConnectionProvider;
import org.datanucleus.store.valuegenerator.ValueGenerator;
import org.datanucleus.store.valuegenerator.ValueGenerationManager;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;

/**
 * Basic generic implementation of a JDO2 datastore sequence for DB4O datastores.
 * Utilises the "org.datanucleus.store_valuegenerator" extensions.
 */
public class JDOSequenceImpl implements Sequence, NucleusSequence
{
    /** Localisation of messages */
    protected static final Localiser LOCALISER = Localiser.getInstance("org.datanucleus.Localisation",
        ObjectManagerFactoryImpl.class.getClassLoader());

    /** Store Manager where we obtain our sequence. */
    protected final StoreManager storeManager;

    /** Name of the sequence. */
    protected final SequenceMetaData seqMetaData;

    /** The generator for the sequence. */
    protected final ValueGenerator generator;

    /** The controlling Object Manager. */
    protected final ObjectManager om;

    /**
     * Constructor.
     * @param objectMgr The Object Manager managing the sequence
     * @param storeMgr Manager of the store where we obtain the sequence
     * @param seqmd MetaData defining the sequence
     */
    public JDOSequenceImpl(ObjectManager objectMgr, final StoreManager storeMgr, SequenceMetaData seqmd)
    {
        this.om = objectMgr;
        this.storeManager = storeMgr;
        this.seqMetaData = seqmd;

        // Allocate the PoidManager for this sequence
        String poidGeneratorName = "sequence";

        // Create the controlling properties for this sequence
        Properties props = new Properties();
        ExtensionMetaData[] seqExtensions = seqmd.getExtensions();
        if (seqExtensions != null && seqExtensions.length > 0)
        {
            // Add all MetaData extension properties provided
            for (int i=0;i<seqExtensions.length;i++)
            {
                props.put(seqExtensions[i].getKey(), seqExtensions[i].getValue());
            }
        }
        props.put("sequence-name", seqMetaData.getDatastoreSequence());

        // Get a PoidManager to create the generator
        ValueGenerationManager mgr = storeMgr.getPoidManager();
        final ObjectManager thisOM = objectMgr;

        ValueGenerationConnectionProvider connProvider = new ValueGenerationConnectionProvider()
        {
            ManagedConnection mconn;
            public ManagedConnection retrieveConnection()
            {
                mconn = storeMgr.getConnection(thisOM);
                return mconn;
            }

            public void releaseConnection()
            {
                this.mconn.release();
                this.mconn = null;
            }
        };

        Class cls = null;
        ConfigurationElement elem =
            objectMgr.getOMFContext().getPluginManager().getConfigurationElementForExtension(
                "org.datanucleus.store_valuegenerator", 
                new String[]{"name", "datastore"}, 
                new String[] {poidGeneratorName, storeManager.getStoreManagerKey()});
        if (elem != null)
        {
            cls = objectMgr.getOMFContext().getPluginManager().loadClass(
                elem.getExtension().getPlugin().getSymbolicName(), elem.getAttribute("class-name"));
        }
        if (cls == null)
        {
            throw new NucleusException("Cannot create Poid Generator for strategy "+poidGeneratorName);
        }
        generator = mgr.createValueGenerator(seqMetaData.getName(), cls, props, storeManager, connProvider);

        if (NucleusLogger.JDO.isDebugEnabled())
        {
            NucleusLogger.JDO.debug(LOCALISER.msg("017003", seqMetaData.getName(), poidGeneratorName));
        }
    }

    /**
     * Accessor for the sequence name.
     * @return The sequence name
     */
    public String getName()
    {
        return seqMetaData.getName();
    }

    /**
     * Method to allocate a set of elements.
     * @param additional The number of additional elements to allocate
     */
    public void allocate(int additional)
    {
        try
        {
            generator.allocate(additional);
        }
        catch (NucleusException de)
        {
        }
    }

    /**
     * Accessor for the next element in the sequence.
     * @return The next element
     */
    public Object next()
    {
        try
        {
            return generator.next();
        }
        catch (NucleusDataStoreException dse)
        {
            if (dse.getFailedObject() != null)
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getFailedObject());
            }
            else
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getNestedExceptions());
            }
        }
    }

    /**
     * Accessor for the next element in the sequence as a long.
     * @return The next element
     * @throws JDODataStoreException Thrown if not numeric
     */
    public long nextValue()
    {
        try
        {
            return generator.nextValue();
        }
        catch (NucleusDataStoreException dse)
        {
            if (dse.getFailedObject() != null)
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getFailedObject());
            }
            else
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getNestedExceptions());
            }
        }
    }

    /**
     * Accessor for the current element.
     * @return The current element.
     */
    public Object current()
    {
        try
        {
            return generator.current();
        }
        catch (NucleusDataStoreException dse)
        {
            if (dse.getFailedObject() != null)
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getFailedObject());
            }
            else
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getNestedExceptions());
            }
        }
    }

    /**
     * Accessor for the current element in the sequence as a long.
     * @return The current element
     * @throws JDODataStoreException Thrown if not numeric
     */
    public long currentValue()
    {
        try
        {
            return generator.currentValue();
        }
        catch (NucleusDataStoreException dse)
        {
            if (dse.getFailedObject() != null)
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getFailedObject());
            }
            else
            {
                throw new JDODataStoreException(dse.getMessage(), dse.getNestedExceptions());
            }
        }
    }
}