/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.jdo.metadata;

import org.xml.sax.Attributes;
import org.xml.sax.EntityResolver;
import org.xml.sax.SAXException;

import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractElementMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ArrayMetaData;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.CollectionMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.metadata.DiscriminatorMetaData;
import org.datanucleus.metadata.ElementMetaData;
import org.datanucleus.metadata.EmbeddedMetaData;
import org.datanucleus.metadata.FetchGroupMetaData;
import org.datanucleus.metadata.FetchPlanMetaData;
import org.datanucleus.metadata.FieldMetaData;
import org.datanucleus.metadata.FileMetaData;
import org.datanucleus.metadata.ForeignKeyMetaData;
import org.datanucleus.metadata.IdentityMetaData;
import org.datanucleus.metadata.ImplementsMetaData;
import org.datanucleus.metadata.IndexMetaData;
import org.datanucleus.metadata.InheritanceMetaData;
import org.datanucleus.metadata.InterfaceMetaData;
import org.datanucleus.metadata.JoinMetaData;
import org.datanucleus.metadata.KeyMetaData;
import org.datanucleus.metadata.MapMetaData;
import org.datanucleus.metadata.MetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.OrderMetaData;
import org.datanucleus.metadata.PackageMetaData;
import org.datanucleus.metadata.PrimaryKeyMetaData;
import org.datanucleus.metadata.PropertyMetaData;
import org.datanucleus.metadata.QueryMetaData;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.metadata.UniqueMetaData;
import org.datanucleus.metadata.ValueMetaData;
import org.datanucleus.metadata.VersionMetaData;
import org.datanucleus.metadata.xml.AbstractMetaDataHandler;
import org.datanucleus.util.NucleusLogger;

/**
 * Parser handler for JDO MetaData.
 * Implements DefaultHandler and handles the extracting of MetaData for JDO
 * from the XML elements/attributes. This class simply constructs the MetaData
 * representation mirroring what is in the MetaData file. It has no knowledge
 * of the class(es) that it represents, simply the information in the MetaData
 * file. The knowledge of the classes is imposed on the representation at a
 * later stage where necessary.
 * <P>Operates the parse process using a Stack. MetaData components are added
 * to the stack as they are encountered and created. They are then popped off
 * the stack when the end element is encountered.</P>
 *
 * @since 1.1
 * @version $Revision: 1.2 $
 */
public class JDOMetaDataHandler extends AbstractMetaDataHandler
{
    /**
     * Constructor. Protected to prevent instantiation.
     * @param mgr the metadata manager
     * @param filename The name of the file to parse
     * @param resolver Entity Resolver to use (null if not available)
     */
    public JDOMetaDataHandler(MetaDataManager mgr, String filename, EntityResolver resolver)
    {
        super(mgr, filename, resolver);
        metadata = new FileMetaData(filename, mgr, null, null);
        pushStack(metadata);
    }
    
    /**
     * Utility to create a new class component.
     * @param pmd The parent PackageMetaData
     * @param attrs The attributes
     * @return The ClassMetaData
     */
    protected ClassMetaData newClassObject(PackageMetaData pmd, Attributes attrs)
    {
        ClassMetaData cmd = mgr.getMetaDataFactory().newClassObject(pmd,
                            getAttr(attrs,"name"),
                            getAttr(attrs,"identity-type"),
                            getAttr(attrs,"objectid-class"),
                            getAttr(attrs,"requires-extent"),
                            getAttr(attrs,"detachable"),
                            getAttr(attrs,"embedded-only"),
                            getAttr(attrs,"persistence-modifier"),
                            getAttr(attrs,"persistence-capable-superclass"),
                            getAttr(attrs,"catalog"),
                            getAttr(attrs,"schema"),
                            getAttr(attrs,"table"),
                            null);
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            cmd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return cmd;
    }

    /**
     * Utility to create a new interface component.
     * @param pmd The parent PackageMetaData
     * @param attrs The attributes
     * @return The InterfaceMetaData
     */
    protected InterfaceMetaData newInterfaceObject(PackageMetaData pmd, Attributes attrs)
    {
        InterfaceMetaData imd = mgr.getMetaDataFactory().newInterfaceObject(pmd,
            getAttr(attrs, "name"),
            getAttr(attrs, "identity-type"),
            getAttr(attrs, "objectid-class"),
            getAttr(attrs, "requires-extent"),
            getAttr(attrs, "detachable"),
            getAttr(attrs, "embedded-only"),
            getAttr(attrs, "catalog"),
            getAttr(attrs, "schema"),
            getAttr(attrs, "table"),
            null);
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            imd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return imd;
    }

    /**
     * Utility to create a new field component.
     * @param md The parent MetaData
     * @param attrs The attributes
     * @return The FieldMetaData
     */
    protected FieldMetaData newFieldObject(MetaData md, Attributes attrs)
    {
        FieldMetaData fmd = mgr.getMetaDataFactory().newFieldObject(md,
                            getAttr(attrs,"name"),
                            getAttr(attrs,"primary-key"),
                            getAttr(attrs,"persistence-modifier"),
                            getAttr(attrs,"default-fetch-group"),
                            getAttr(attrs,"null-value"),
                            getAttr(attrs,"embedded"),
                            getAttr(attrs,"serialized"),
                            getAttr(attrs,"dependent"),
                            getAttr(attrs,"mapped-by"),
                            getAttr(attrs,"column"),
                            getAttr(attrs,"table"),
                            null, // Catalog not supported in JDO2 metadata
                            null, // Schema not supported in JDO2 metadata
                            getAttr(attrs,"delete-action"),
                            getAttr(attrs,"indexed"),
                            getAttr(attrs,"unique"),
                            getAttr(attrs,"recursion-depth"),
                            getAttr(attrs,"load-fetch-group"),
                            getAttr(attrs,"value-strategy"),
                            getAttr(attrs,"sequence"),
                            getAttr(attrs,"field-type"));
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            fmd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return fmd;
    }

    /**
     * Utility to create a new property component.
     * @param md The parent MetaData
     * @param attrs The attributes
     * @return The PropertyMetaData
     */
    protected PropertyMetaData newPropertyObject(MetaData md, Attributes attrs)
    {
        PropertyMetaData pmd = mgr.getMetaDataFactory().newPropertyObject(md,
                            getAttr(attrs,"name"),
                            getAttr(attrs,"primary-key"),
                            getAttr(attrs,"persistence-modifier"),
                            getAttr(attrs,"default-fetch-group"),
                            getAttr(attrs,"null-value"),
                            getAttr(attrs,"embedded"),
                            getAttr(attrs,"serialized"),
                            getAttr(attrs,"dependent"),
                            getAttr(attrs,"mapped-by"),
                            getAttr(attrs,"column"),
                            getAttr(attrs,"table"),
                            null, // Catalog not supported in JDO2 metadata
                            null, // Schema not supported in JDO2 metadata
                            getAttr(attrs,"delete-action"),
                            getAttr(attrs,"indexed"),
                            getAttr(attrs,"unique"),
                            getAttr(attrs,"recursion-depth"),
                            getAttr(attrs,"load-fetch-group"),
                            getAttr(attrs,"value-strategy"),
                            getAttr(attrs,"sequence"),
                            getAttr(attrs,"field-type"),
                            getAttr(attrs,"field-name"));
        String cacheableAttr = getAttr(attrs, "cacheable");
        if (cacheableAttr != null)
        {
            pmd.setCacheable(cacheableAttr.equalsIgnoreCase("false") ? false : true);
        }
        return pmd;
    }    
    
    /**
     * Handler method called at the start of an element.
     * @param uri URI of the tag
     * @param localName Local name
     * @param qName Element name
     * @param attrs Attributes for this element 
     * @throws SAXException in parsing errors
     */
    public void startElement(String uri, 
                             String localName,
                             String qName,
                             Attributes attrs)
    throws SAXException 
    {
        if (charactersBuffer.length() > 0)
        {
            // Cater for subelements that appear before the end of the body text (save the body text with the parent)
            String currentString = getString().trim();
            if (getStack() instanceof QueryMetaData)
            {
                ((QueryMetaData)getStack()).setQuery(currentString.trim());
            }
        }

        if (NucleusLogger.METADATA.isDebugEnabled())
        {
            StringBuffer sb = new StringBuffer();
            sb.append("<" + qName);
            for (int i=0; i<attrs.getLength(); i++)
            {
                sb.append(" ");
                sb.append(attrs.getQName(i)).append("=\"").append(attrs.getValue(i)).append("\"");
            }
            sb.append(">");
            NucleusLogger.METADATA.debug(LOCALISER.msg("044034",sb.toString(), "" + stack.size()));
        }
        if (localName.length()<1)
        {
            localName = qName;
        }
        try
        {
            // New JDO file
            if (localName.equals("jdo"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                filemd.setType(FileMetaData.JDO_FILE);
                filemd.setCatalog(getAttr(attrs, "catalog"));
                filemd.setSchema(getAttr(attrs, "schema"));
            }
            // New JDO O/R mapping file
            else if (localName.equals("orm"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                filemd.setType(FileMetaData.ORM_FILE);
                filemd.setCatalog(getAttr(attrs, "catalog"));
                filemd.setSchema(getAttr(attrs, "schema"));
            }
            // New JDO Query file
            else if (localName.equals("jdoquery"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                filemd.setType(FileMetaData.JDOQUERY_FILE);
            }
            // New fetch-plan for this file
            else if (localName.equals("fetch-plan"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                FetchPlanMetaData fpmd = new FetchPlanMetaData(filemd, getAttr(attrs, "name"),
                    getAttr(attrs, "max-fetch-depth"), getAttr(attrs, "fetch-size"));
                ((FileMetaData)metadata).addFetchPlan(fpmd);
                pushStack(fpmd);
            }
            // New package for this file
            else if (localName.equals("package"))
            {
                FileMetaData filemd = (FileMetaData)getStack();
                PackageMetaData pmd = new PackageMetaData(filemd, getAttr(attrs, "name"),
                    getAttr(attrs, "catalog"), getAttr(attrs, "schema"));
                ((FileMetaData)metadata).addPackage(pmd);
                pushStack(pmd);
            }
            // New class for this package
            else if (localName.equals("class"))
            {
                PackageMetaData pmd = (PackageMetaData)getStack();
                ClassMetaData cmd = newClassObject(pmd,attrs);
                pmd.addClass(cmd);
    
                pushStack(cmd);
            }
            // New interface for this package
            else if (localName.equals("interface"))
            {
                PackageMetaData pmd = (PackageMetaData)getStack();
                InterfaceMetaData imd = newInterfaceObject(pmd, attrs);
                pmd.addInterface(imd);
                pushStack(imd);
            }
            // New primary-key for class/interface
            else if (localName.equals("primary-key"))
            {
                MetaData md = getStack();
                PrimaryKeyMetaData pkmd = 
                    new PrimaryKeyMetaData(md, getAttr(attrs,"name"), getAttr(attrs, "column"));
                if (md instanceof AbstractClassMetaData)
                {
                    ((AbstractClassMetaData)md).setPrimaryKeyMetaData(pkmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    ((JoinMetaData)md).setPrimaryKeyMetaData(pkmd);
                }
                pushStack(pkmd);
            }
            // New implements for this class
            else if (localName.equals("implements"))
            {
                ClassMetaData cmd = (ClassMetaData)getStack();
                ImplementsMetaData imd = 
                    new ImplementsMetaData(cmd, getAttr(attrs,"name"));
                cmd.addImplements(imd);
                pushStack(imd);
            }
            // New property for this interface
            else if (localName.equals("property"))
            {
                MetaData parent = getStack();
                if (parent instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData acmd = (AbstractClassMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(acmd, attrs);
                    acmd.addMember(propmd);
                    pushStack(propmd);
                }
                else if (parent instanceof EmbeddedMetaData)
                {
                    EmbeddedMetaData emd = (EmbeddedMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(emd, attrs);
                    emd.addMember(propmd);
                    pushStack(propmd);
                }
                else if (parent instanceof ImplementsMetaData)
                {
                    ImplementsMetaData implmd = (ImplementsMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(implmd, attrs);
                    implmd.addProperty(propmd);
                    pushStack(propmd);
                }
                else if (parent instanceof FetchGroupMetaData)
                {
                    FetchGroupMetaData fgmd = (FetchGroupMetaData)parent;
                    PropertyMetaData propmd = newPropertyObject(fgmd, attrs);
                    fgmd.addMember(propmd);
                    pushStack(propmd);
                }
            }
            // New datastore-identity for this class
            else if (localName.equals("datastore-identity"))
            {
                AbstractClassMetaData acmd = (AbstractClassMetaData)getStack();
                IdentityMetaData imd = new IdentityMetaData(acmd,
                                getAttr(attrs,"column"),
                                getAttr(attrs,"strategy"),
                                getAttr(attrs,"sequence"));
                acmd.setIdentityMetaData(imd);
                pushStack(imd);
            }
            // New inheritance for this class
            else if (localName.equals("inheritance"))
            {
                MetaData parent = getStack();
                AbstractClassMetaData acmd = (AbstractClassMetaData)parent;
                InheritanceMetaData imd = new InheritanceMetaData(acmd,
                                getAttr(attrs,"strategy"));
                acmd.setInheritanceMetaData(imd);
                pushStack(imd);
            }
            // Discriminator for this inheritance
            else if (localName.equals("discriminator"))
            {
                InheritanceMetaData imd = (InheritanceMetaData)getStack();
                DiscriminatorMetaData dismd = new DiscriminatorMetaData(imd,
                                getAttr(attrs,"column"),
                                getAttr(attrs,"value"),
                                getAttr(attrs,"strategy"),
                                getAttr(attrs,"indexed"));
                imd.setDiscriminatorMetaData(dismd);
                pushStack(dismd);
            }
            // New query for this class
            else if (localName.equals("query"))
            {
                MetaData emd = getStack();
                QueryMetaData qmd = new QueryMetaData(emd,
                                (emd instanceof ClassMetaData ? ((ClassMetaData)emd).getFullClassName() : null),
                                getAttr(attrs, "name"),
                                getAttr(attrs, "language"),
                                getAttr(attrs, "unmodifiable"),
                                getAttr(attrs, "result-class"),
                                null,
                                getAttr(attrs, "unique"),
                                getAttr(attrs, "fetch-plan"));
                if (emd instanceof ClassMetaData)
                {
                    ClassMetaData cmd = (ClassMetaData)emd;
                    cmd.addQuery(qmd);
                }
                else if (emd instanceof FileMetaData)
                {
                    FileMetaData fmd = (FileMetaData)emd;
                    fmd.addQuery(qmd);
                }
                pushStack(qmd);
            }
            // New sequence for this class
            else if (localName.equals("sequence"))
            {
                MetaData emd = getStack();
                SequenceMetaData seqmd = new SequenceMetaData(emd,
                                getAttr(attrs,"name"),
                                getAttr(attrs,"datastore-sequence"),
                                getAttr(attrs,"factory-class"),
                                getAttr(attrs,"strategy"),
                                null, null);
                if (emd instanceof PackageMetaData)
                {
                    PackageMetaData pmd = (PackageMetaData)emd;
                    pmd.addSequence(seqmd);
                }
                pushStack(seqmd);
            }
            // New field for this class
            else if (localName.equals("field"))
            {
                MetaData md = getStack();
                FieldMetaData fmd = newFieldObject(md,attrs);
                if (md instanceof ClassMetaData)
                {
                    ClassMetaData cmd = (ClassMetaData)md;
                    cmd.addMember(fmd);
                }
                else if (md instanceof FetchGroupMetaData)
                {
                    FetchGroupMetaData fgmd = (FetchGroupMetaData)md;
                    fgmd.addMember(fmd);
                }
                else if (md instanceof EmbeddedMetaData)
                {
                    EmbeddedMetaData emd = (EmbeddedMetaData)md;
                    emd.addMember(fmd);
                }
                else if (md instanceof ForeignKeyMetaData)
                {
                    ForeignKeyMetaData fkmd = (ForeignKeyMetaData)md;
                    fkmd.addMember(fmd);
                }
                else if (md instanceof IndexMetaData)
                {
                    IndexMetaData imd = (IndexMetaData)md;
                    imd.addMember(fmd);
                }
                else if (md instanceof UniqueMetaData)
                {
                    UniqueMetaData umd = (UniqueMetaData)md;
                    umd.addMember(fmd);
                }
                pushStack(fmd);
            }
            // New join
            else if (localName.equals("join"))
            {
                MetaData parent = getStack();
                JoinMetaData jnmd = new JoinMetaData(parent,
                                                   getAttr(attrs, "table"),
                                                   null, null,
                                                   getAttr(attrs, "column"),
                                                   getAttr(attrs, "outer"),
                                                   getAttr(attrs, "delete-action"),
                                                   getAttr(attrs, "indexed"),
                                                   getAttr(attrs, "unique"));
                if (parent instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)parent;
                    fmd.setJoinMetaData(jnmd);
                }
                else if (parent instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)parent;
                    cmd.addJoin(jnmd);
                }
                else if (parent instanceof InheritanceMetaData)
                {
                    InheritanceMetaData imd = (InheritanceMetaData)parent;
                    imd.setJoinMetaData(jnmd);
                }
                pushStack(jnmd);
            }
            // New map container for this field
            else if (localName.equals("map"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                MapMetaData map = new MapMetaData(fmd,
                                getAttr(attrs,"key-type"),
                                getAttr(attrs,"embedded-key"),
                                getAttr(attrs,"dependent-key"),
                                getAttr(attrs,"serialized-key"),
                                getAttr(attrs,"value-type"),
                                getAttr(attrs,"embedded-value"),
                                getAttr(attrs,"dependent-value"),
                                getAttr(attrs,"serialized-value"));
                fmd.setContainer(map);
                pushStack(map);
            }
            // New array container for this field
            else if (localName.equals("array"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                ArrayMetaData amd = new ArrayMetaData(fmd,
                                getAttr(attrs, "element-type"),
                                getAttr(attrs, "embedded-element"),
                                getAttr(attrs, "dependent-element"),
                                getAttr(attrs, "serialized-element"));
                fmd.setContainer(amd);
                pushStack(amd);
            }
            // New collection container for this field
            else if (localName.equals("collection"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                CollectionMetaData colmd = new CollectionMetaData(fmd,
                                getAttr(attrs,"element-type"),
                                getAttr(attrs,"embedded-element"),
                                getAttr(attrs,"dependent-element"),
                                getAttr(attrs,"serialized-element"));
                fmd.setContainer(colmd);
                pushStack(colmd);
            }
            // New column
            else if (localName.equals("column"))
            {
                MetaData md = getStack();
                ColumnMetaData clnmd = new ColumnMetaData(md,
                                getAttr(attrs,"name"),
                                getAttr(attrs,"target"),
                                getAttr(attrs,"target-field"),
                                getAttr(attrs,"jdbc-type"),
                                getAttr(attrs,"sql-type"),
                                getAttr(attrs,"length"),
                                getAttr(attrs,"scale"),
                                getAttr(attrs,"allows-null"),
                                getAttr(attrs,"default-value"),
                                getAttr(attrs,"insert-value"),
                                null, null, null);
                if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.addColumn(clnmd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elemd = (AbstractElementMetaData)md;
                    elemd.addColumn(clnmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jnmd = (JoinMetaData)md;
                    jnmd.addColumn(clnmd);
                }
                else if (md instanceof IdentityMetaData)
                {
                    IdentityMetaData idmd = (IdentityMetaData)md;
                    idmd.addColumn(clnmd);
                }
                else if (md instanceof ForeignKeyMetaData)
                {
                    ForeignKeyMetaData fkmd = (ForeignKeyMetaData)md;
                    fkmd.addColumn(clnmd);
                }
                else if (md instanceof IndexMetaData)
                {
                    IndexMetaData idxmd = (IndexMetaData)md;
                    idxmd.addColumn(clnmd);
                }
                else if (md instanceof UniqueMetaData)
                {
                    UniqueMetaData unimd = (UniqueMetaData)md;
                    unimd.addColumn(clnmd);
                }
                else if (md instanceof OrderMetaData)
                {
                    OrderMetaData ormd = (OrderMetaData)md;
                    ormd.addColumn(clnmd);
                }
                else if (md instanceof DiscriminatorMetaData)
                {
                    DiscriminatorMetaData dismd = (DiscriminatorMetaData)md;
                    dismd.setColumnMetaData(clnmd);
                }
                else if (md instanceof VersionMetaData)
                {
                    VersionMetaData vermd = (VersionMetaData)md;
                    vermd.addColumn(clnmd);
                }
                else if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addUnmappedColumn(clnmd);
                }
                else if (md instanceof PrimaryKeyMetaData)
                {
                    PrimaryKeyMetaData pkmd = (PrimaryKeyMetaData)md;
                    pkmd.addColumn(clnmd);
                }
                pushStack(clnmd);
            }
            // New element
            else if (localName.equals("element"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                ElementMetaData elemd = new ElementMetaData(fmd,
                                getAttr(attrs, "column"),
                                getAttr(attrs, "delete-action"),
                                getAttr(attrs, "update-action"),
                                getAttr(attrs, "indexed"),
                                getAttr(attrs, "unique"),
                                getAttr(attrs, "mapped-by"));
                fmd.setElementMetaData(elemd);
                pushStack(elemd);
            }
            // New key
            else if (localName.equals("key"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                KeyMetaData keymd = new KeyMetaData(fmd,
                                getAttr(attrs, "column"),
                                getAttr(attrs, "delete-action"),
                                getAttr(attrs, "update-action"),
                                getAttr(attrs, "indexed"),
                                getAttr(attrs, "unique"),
                                getAttr(attrs, "mapped-by"));
                fmd.setKeyMetaData(keymd);
                pushStack(keymd);
            }
            // New value
            else if (localName.equals("value"))
            {
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                ValueMetaData valuemd = new ValueMetaData(fmd,
                                getAttr(attrs, "column"),
                                getAttr(attrs, "delete-action"),
                                getAttr(attrs, "update-action"),
                                getAttr(attrs, "indexed"),
                                getAttr(attrs, "unique"),
                                getAttr(attrs, "mapped-by"));
                fmd.setValueMetaData(valuemd);
                pushStack(valuemd);
            }
            // New fetch-group
            else if (localName.equals("fetch-group"))
            {
                MetaData md = getStack();
                FetchGroupMetaData fgmd = new FetchGroupMetaData(md, getAttr(attrs,"post-load"), getAttr(attrs,"name"));
                if (md instanceof FetchGroupMetaData)
                {
                    FetchGroupMetaData fgmdParent = (FetchGroupMetaData)md;
                    fgmdParent.addFetchGroup(fgmd);
                }
                else if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addFetchGroup(fgmd);
                }
                else if (md instanceof FetchPlanMetaData)
                {
                    FetchPlanMetaData fpmd = (FetchPlanMetaData)md;
                    fpmd.addFetchGroup(fgmd);
                }
                pushStack(fgmd);
            }
            // New extension for this tag
            else if (localName.equals("extension"))
            {
                MetaData md = getStack();
                md.addExtension(getAttr(attrs,"vendor-name"), getAttr(attrs,"key"), getAttr(attrs,"value"));
            }
            // New version
            else if (localName.equals("version"))
            {
                AbstractClassMetaData cmd = (AbstractClassMetaData)getStack();
                VersionMetaData vermd = new VersionMetaData(
                                getAttr(attrs,"strategy"),
                                getAttr(attrs,"column"),
                                getAttr(attrs,"indexed"));
                cmd.setVersionMetaData(vermd);
                pushStack(vermd);
            }
            // New index
            else if (localName.equals("index"))
            {
                MetaData md = getStack();
                IndexMetaData idxmd = new IndexMetaData(
                                getAttr(attrs,"name"),
                                getAttr(attrs,"table"),
                                getAttr(attrs,"unique"));
                if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addIndex(idxmd);
                }
                else if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setIndexMetaData(idxmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jmd = (JoinMetaData)md;
                    jmd.setIndexMetaData(idxmd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elmd = (AbstractElementMetaData)md;
                    elmd.setIndexMetaData(idxmd);
                }
                else if (md instanceof OrderMetaData)
                {
                    OrderMetaData omd = (OrderMetaData)md;
                    omd.setIndexMetaData(idxmd);
                }
                else if (md instanceof VersionMetaData)
                {
                    VersionMetaData vermd = (VersionMetaData)md;
                    vermd.setIndexMetaData(idxmd);
                }
                else if (md instanceof DiscriminatorMetaData)
                {
                    DiscriminatorMetaData dismd = (DiscriminatorMetaData)md;
                    dismd.setIndexMetaData(idxmd);
                }
                pushStack(idxmd);
            }
            // New unique constraint
            else if (localName.equals("unique"))
            {
                MetaData md = getStack();
                UniqueMetaData unimd = new UniqueMetaData(getAttr(attrs,"name"),
                                                          getAttr(attrs,"table"),
                                                          getAttr(attrs,"deferred"));
                if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addUniqueConstraint(unimd);
                }
                else if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setUniqueMetaData(unimd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jmd = (JoinMetaData)md;
                    jmd.setUniqueMetaData(unimd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elmd = (AbstractElementMetaData)md;
                    elmd.setUniqueMetaData(unimd);
                }
                pushStack(unimd);
            }
            // New foreign key
            else if (localName.equals("foreign-key"))
            {
                MetaData md = getStack();
                ForeignKeyMetaData fkmd = new ForeignKeyMetaData(getAttr(attrs,"name"),
                                getAttr(attrs,"table"),
                                getAttr(attrs,"unique"),
                                getAttr(attrs,"deferred"),
                                getAttr(attrs,"delete-action"),
                                getAttr(attrs,"update-action"));
                if (md instanceof AbstractClassMetaData)
                {
                    AbstractClassMetaData cmd = (AbstractClassMetaData)md;
                    cmd.addForeignKey(fkmd);
                }
                else if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setForeignKeyMetaData(fkmd);
                }
                else if (md instanceof JoinMetaData)
                {
                    JoinMetaData jmd = (JoinMetaData)md;
                    jmd.setForeignKeyMetaData(fkmd);
                }
                else if (md instanceof AbstractElementMetaData)
                {
                    AbstractElementMetaData elmd = (AbstractElementMetaData)md;
                    elmd.setForeignKeyMetaData(fkmd);
                }
                pushStack(fkmd);
            }
            // New order
            else if (localName.equals("order"))
            {
                OrderMetaData ormd = new OrderMetaData(getAttr(attrs, "column"),
                                                       getAttr(attrs, "indexed"),
                                                       getAttr(attrs, "mapped-by"));
                AbstractMemberMetaData fmd = (AbstractMemberMetaData)getStack();
                fmd.setOrderMetaData(ormd);
                pushStack(ormd);
            }
            // Embedded
            else if (localName.equals("embedded"))
            {
                MetaData md = getStack();
                EmbeddedMetaData emd = new EmbeddedMetaData(md,
                                                            getAttr(attrs,"owner-field"),
                                                            getAttr(attrs,"null-indicator-column"),
                                                            getAttr(attrs,"null-indicator-value"));
                if (md instanceof AbstractMemberMetaData)
                {
                    AbstractMemberMetaData fmd = (AbstractMemberMetaData)md;
                    fmd.setEmbeddedMetaData(emd);
                }
                else if (md instanceof KeyMetaData)
                {
                    KeyMetaData kmd = (KeyMetaData)md;
                    kmd.setEmbeddedMetaData(emd);
                }
                else if (md instanceof ValueMetaData)
                {
                    ValueMetaData vmd = (ValueMetaData)md;
                    vmd.setEmbeddedMetaData(emd);
                }
                else if (md instanceof ElementMetaData)
                {
                    ElementMetaData elmd = (ElementMetaData)md;
                    elmd.setEmbeddedMetaData(emd);
                }
                pushStack(emd);
            }
            else
            {
                String message = LOCALISER.msg("044037",qName);
                NucleusLogger.METADATA.error(message);
                throw new RuntimeException(message);
            }
        }
        catch(RuntimeException ex)
        {
            NucleusLogger.METADATA.error(LOCALISER.msg("044042", qName, getStack(), uri), ex);
            throw ex;
        }
    }

    /**
     * Handler method called at the end of an element.
     * @param uri URI of the tag
     * @param localName local name
     * @param qName Name of element just ending
     * @throws SAXException in parsing errors
     */
    public void endElement(String uri, String localName, String qName)
    throws SAXException
    {
        if (NucleusLogger.METADATA.isDebugEnabled())
        {
            NucleusLogger.METADATA.debug(LOCALISER.msg("044035","<" + qName + ">", "" + stack.size()));
        }
        if (localName.length() < 1)
        {
            localName = qName;
        }

        // Save the current string for elements that have a body value
        String currentString = getString().trim();
        if (currentString.length() > 0)
        {
            MetaData md = getStack();
            if (localName.equals("query"))
            {
                ((QueryMetaData)md).setQuery(currentString);
            }
        }

        // Pop the tag
        // If startElement pushes an element onto the stack need a remove here for that type
        if (localName.equals("package") ||
            localName.equals("fetch-plan") ||
            localName.equals("class") ||
            localName.equals("interface") ||
            localName.equals("implements") ||
            localName.equals("property") ||
            localName.equals("datastore-identity") ||
            localName.equals("inheritance") ||
            localName.equals("primary-key") ||
            localName.equals("version") ||
            localName.equals("unmapped") ||
            localName.equals("query") ||
            localName.equals("sequence") ||
            localName.equals("field") ||
            localName.equals("map") ||
            localName.equals("element") ||
            localName.equals("embedded") ||
            localName.equals("key") ||
            localName.equals("value") ||
            localName.equals("array") ||
            localName.equals("collection") ||
            localName.equals("join") ||
            localName.equals("index") ||
            localName.equals("unique") ||
            localName.equals("foreign-key") ||
            localName.equals("order") ||
            localName.equals("fetch-group") ||
            localName.equals("column") ||
            localName.equals("discriminator"))
        {
            popStack();
        }
    }
}