/**********************************************************************
Copyright (c) 2007 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.jdo.state;

import javax.jdo.JDOUserException;

import org.datanucleus.StateManager;
import org.datanucleus.Transaction;
import org.datanucleus.state.LifeCycleState;


/**
 * Class representing the life cycle state of PersistentNontransactionalDirty
 **/
class PersistentNontransactionalDirty extends LifeCycleState
{
    /** Protected Constructor to prevent external instantiation. */
    protected PersistentNontransactionalDirty()
    {
		isPersistent = true;    	
        isDirty = true;
        isNew = false;
        isDeleted = false;
        isTransactional = false;

        stateType = P_NONTRANS_DIRTY;
    }


    /**
     * Method to transition to transactional.
     * @param sm StateManager.
     * @param refreshFields Whether to refresh loaded fields
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionMakeTransactional(StateManager sm, boolean refreshFields)
    {
        return this;
    }

    /**
     * Method to transition to commit state.
     * @param sm StateManager.
     * @param tx the Transaction been committed.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionCommit(StateManager sm, Transaction tx)
    {
        sm.clearSavedFields();

        if (tx.getRetainValues())
        {
            return changeState(sm, P_NONTRANS);
        }
        else
        {
            sm.clearNonPrimaryKeyFields();
            return changeState(sm, HOLLOW);
        }
    }

    /**
     * Method to transition to rollback state.
     * @param sm StateManager.
     * @param tx The transaction
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRollback(StateManager sm,Transaction tx)
    {
        if (tx.getRestoreValues())
        {
            sm.restoreFields();
            return changeState(sm, P_NONTRANS_DIRTY);
        }
        else
        {
            sm.clearNonPrimaryKeyFields();
            sm.clearSavedFields();
            return changeState(sm, HOLLOW);
        }
    }
    
    /**
     * Method to transition to evict state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionEvict(StateManager sm)
    {
        sm.clearNonPrimaryKeyFields();
        sm.clearSavedFields();
        return changeState(sm, HOLLOW);
    }

    /**
     * Method to transition to read-field state.
     * @param sm StateManager.
     * @param isLoaded if the field was previously loaded.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionReadField(StateManager sm, boolean isLoaded)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
		if (!tx.isActive() && !tx.getNontransactionalRead())
		{
	        throw new JDOUserException(LOCALISER.msg("027002"),sm.getInternalObjectId());
		}
        return this;
    }

    /**
     * Method to transition to transaction begin state.
     * @param sm StateManager.
     * @param tx Transaction.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionBegin(StateManager sm, org.datanucleus.Transaction tx)
    {
        sm.saveFields();
        sm.enlistInTransaction();
        return this;
    }
    
    /**
     * Method to transition to write-field state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionWriteField(StateManager sm)
    {
        return this;
    }

    /**
     * Method to transition to detached-clean.
     * @param sm StateManager.
     * @return new LifeCycle state.
     */
    public LifeCycleState transitionDetach(StateManager sm)
    {
        return changeState(sm, DETACHED_CLEAN);
    }

    /**
     * Method to return a string version of this object.
     * @return The string "P_NONTRANS_DIRTY".
     **/
    public String toString()
    {
        return "P_NONTRANS_DIRTY";
    }
}