/**********************************************************************
 Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Contributors:
 ...
 **********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.List;

import org.datanucleus.util.StringUtils;

/**
 * Abstract representation of an ORM constraint.
 * @version $Revision: 1.6 $
 */
public class AbstractConstraintMetaData extends MetaData
{
    /** the constraint name */
    protected String name;

    /** the constraint table name. Name of the table to which this applies (null implies the enclosing class' table). */
    protected String table;

    /** Contains the metadata for fields/properties. */
    protected AbstractMemberMetaData[] memberMetaData;

    /** Contains the metadata for columns */
    protected ColumnMetaData[] columnMetaData;

    // -------------------------------------------------------------------------
    // Fields below here are used in the metadata parse process where the parser
    // dynamically adds fields/columns as it encounters them in the MetaData files.
    // They are typically cleared at the point of initialise() and not used thereafter.

    /** The fields/properties for this constraint. */
    protected List members = new ArrayList();

    /** The columns for this constraint. */
    protected List columns = new ArrayList();

    /**
     * Constructor.
     * @param parent Parent (if required to be set)
     * @param name Name of the constraint
     * @param table Name of table for the constraint
     */
    public AbstractConstraintMetaData(MetaData parent, String name, String table)
    {
        super(parent);

        this.name = (StringUtils.isWhitespace(name) ? null : name);
        this.table = (StringUtils.isWhitespace(table) ? null : table);
    }

    /**
     * Add a new member that is part of this constraint.
     * @param mmd MetaData for the field/property
     */
    public void addMember(AbstractMemberMetaData mmd)
    {
        members.add(mmd);
    }

    /**
     * Add a new ColumnMetaData element
     * @param colmd MetaData for the column
     */
    public void addColumn(ColumnMetaData colmd)
    {
        columns.add(colmd);
    }

    /**
     * Mutator for the constraint name.
     * @param name Name of the constraint.
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Accessor for the name of the constraint
     * @return Returns the name.
     */
    public final String getName()
    {
        return name;
    }

    /**
     * Accessor for the name of the table
     * @return Returns the table.
     */
    public final String getTable()
    {
        return table;
    }

    /**
     * Accessor for metadata for all fields/properties that this constraint relates to.
     * @return Returns the memberMetaData.
     */
    public final AbstractMemberMetaData[] getMemberMetaData()
    {
        return memberMetaData;
    }

    /**
     * Accessor for columnMetaData
     * @return Returns the columnMetaData.
     */
    public final ColumnMetaData[] getColumnMetaData()
    {
        return columnMetaData;
    }

    /**
     * Accessor for the number of fields/properties for this constraint.
     * @return Number of fields/properties
     */
    public int getNumberOfMembers()
    {
        if (members != null)
        {
            return members.size();
        }
        else if (memberMetaData != null)
        {
            return memberMetaData.length;
        }
        return 0;
    }

    /**
     * Accessor for the number of columns for this constraint.
     * @return Number of columns
     */
    public int getNumberOfColumns()
    {
        if (columns != null)
        {
            return columns.size();
        }
        else if (columnMetaData != null)
        {
            return columnMetaData.length;
        }
        return 0;
    }
}