/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2004 Andy Jefferson - added javadocs.
2004 Andy Jefferson - added unique, indexed
2005 Andy Jefferson - changed foreignKey attr to delete-action
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.List;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.util.StringUtils;

/**
 * This element specifies the mapping for the element component of arrays and
 * collections.
 * 
 * If only one column is mapped, and no additional information is needed for the
 * column, then the column attribute can be used. Otherwise, the column
 * element(s) are used.
 * 
 * The serialized attribute specifies that the key values are to be serialized
 * into the named column.
 * 
 * The foreign-key attribute specifies the name of a foreign key to be
 * generated.
 * 
 * @since 1.1
 * @version $Revision: 1.24 $
 */
public abstract class AbstractElementMetaData extends MetaData implements ColumnMetaDataContainer
{
    /**
     * Columns ColumnMetaData
     */
    protected final List columns = new ArrayList();
    
    /**
     * IndexMetaData
     */
    protected IndexMetaData indexMetaData;

    /**
     * The indexing value
     */
    protected IndexedValue indexed=null;

    /**
     * UniqueMetaData.
     */
    protected UniqueMetaData uniqueMetaData;

    /**
     * Whether to add a unique constraint
     */
    protected final boolean uniqueConstraint;

    /**
     * ForeignKeyMetaData
     */
    protected ForeignKeyMetaData foreignKeyMetaData;

    /**
     * EmbeddedMetaData
     */
    protected EmbeddedMetaData embeddedMetaData;

    /** column name value. */
    protected String columnName;

    /** Field that this is mapped to. */
    protected String mappedBy;

    // -------------------------------------------------------------------------
    // Fields below here are not represented in the output MetaData. They are
    // for use internally in the operation of the JDO system. The majority are
    // for convenience to save iterating through the fields since the fields
    // are fixed once initialised.

    protected ColumnMetaData columnMetaData[];

    /**
     * Constructor to create a copy of the passed metadata object applying the passed parent.
     * @param parent The parent
     * @param aemd The metadata to copy
     */
    public AbstractElementMetaData(MetaData parent, AbstractElementMetaData aemd)
    {
        super(parent);
        this.columnName = aemd.columnName;
        this.uniqueConstraint = aemd.uniqueConstraint;
        this.indexed = aemd.indexed;
        this.mappedBy = aemd.mappedBy;
        // TODO These should be changed to be copies
        this.indexMetaData = aemd.indexMetaData;
        this.uniqueMetaData = aemd.uniqueMetaData;
        this.foreignKeyMetaData = aemd.foreignKeyMetaData;
        this.embeddedMetaData = aemd.embeddedMetaData;
        for (int i=0;i<columns.size();i++)
        {
            addColumn(new ColumnMetaData(this,(ColumnMetaData)aemd.columns.get(i)));
        }
    }

    /**
     * Constructor.
     * @param parent Parent element
     * @param columnName Name of column
     * @param deleteAction attribute delete-action value
     * @param updateAction attribute update-action value
     * @param indexed The indexing value
     * @param unique Whether to add a unique constraint
     * @param mappedBy Mapped-by field for this element/key/value
     */
    public AbstractElementMetaData(MetaData parent,
                                   String columnName,
                                   String deleteAction,
                                   String updateAction,
                                   String indexed,
                                   String unique,
                                   String mappedBy)
    {
        super(parent);

        this.columnName = (StringUtils.isWhitespace(columnName) ? null : columnName);

        if (deleteAction != null || updateAction != null)
        {
            foreignKeyMetaData = new ForeignKeyMetaData(null, null, null, null, deleteAction, updateAction);
        }

        this.indexed = IndexedValue.getIndexedValue(indexed);

        if (unique != null && unique.equalsIgnoreCase("true"))
        {
            uniqueConstraint = true;
        }
        else
        {
            uniqueConstraint = false;
        }
        this.mappedBy = (StringUtils.isWhitespace(mappedBy) ? null : mappedBy);
    }

    /**
     * Populate the metadata
     * @param clr the ClassLoaderResolver
     * @param primary the primary ClassLoader to use (or null)
     */
    public void populate(ClassLoaderResolver clr, ClassLoader primary)
    {
        if (embeddedMetaData != null)
        {
            embeddedMetaData.populate(clr, primary);
        }
    }

    /**
     * Method to initialise the object, creating any convenience arrays needed.
     * Initialises all sub-objects. 
     */
    public void initialise()
    {
        // Cater for user specifying column name, or columns
        if (columns.size() == 0 && columnName != null)
        {
            columnMetaData = new ColumnMetaData[1];
            columnMetaData[0] = new ColumnMetaData(this, columnName);
            columnMetaData[0].initialise();
        }
        else
        {
            columnMetaData = new ColumnMetaData[columns.size()];
        	for (int i=0; i<columnMetaData.length; i++)
        	{
            	columnMetaData[i] = (ColumnMetaData) columns.get(i);
            	columnMetaData[i].initialise();
        	}
        }

        // Interpret the "indexed" value to create our IndexMetaData where it wasn't specified that way
        if (indexMetaData == null && columnMetaData != null && indexed != null && indexed != IndexedValue.FALSE)
        {
            indexMetaData = new IndexMetaData(null, null, (indexed == IndexedValue.UNIQUE) ? "true" : "false");
            for (int i=0;i<columnMetaData.length;i++)
            {
                indexMetaData.addColumn(columnMetaData[i]);
            }
        }
        if (indexMetaData != null)
        {
            indexMetaData.initialise();
        }

        if (uniqueMetaData == null && uniqueConstraint)
        {
            uniqueMetaData = new UniqueMetaData(null, columnName, null);
            for (int i=0;i<columnMetaData.length;i++)
            {
                uniqueMetaData.addColumn(columnMetaData[i]);
            }
        }
        if (uniqueMetaData != null)
        {
            uniqueMetaData.initialise();
        }

        if (foreignKeyMetaData != null)
        {
            foreignKeyMetaData.initialise();
        }

        if (embeddedMetaData != null)
        {
            embeddedMetaData.initialise();
        }

        setInitialised();
    }

    // -------------------------------- Accessors ------------------------------

    /**
     * Accessor for column name.
     * @return Returns the column name.
     */
    public final String getColumnName()
    {
        return columnName;
    }

    /**
     * Accessor for the field in the value that stores the key
     * @return Field in the value that stores the key
     */
    public String getMappedBy()
    {
        return mappedBy;
    }

    /**
     * Accessor for columnMetaData
     * @return Returns the columnMetaData.
     */
    public final ColumnMetaData[] getColumnMetaData()
    {
        return columnMetaData;
    }

    /**
     * Accessor for embeddedMetaData
     * @return Returns the embeddedMetaData.
     */
    public final EmbeddedMetaData getEmbeddedMetaData()
    {
        return embeddedMetaData;
    }

    /**
     * Accessor for foreignKeyMetaData
     * @return Returns the foreignKeyMetaData.
     */
    public final ForeignKeyMetaData getForeignKeyMetaData()
    {
        return foreignKeyMetaData;
    }

    /**
     * Accessor for indexMetaData
     * @return Returns the indexMetaData.
     */
    public final IndexMetaData getIndexMetaData()
    {
        return indexMetaData;
    }

    /**
     * Accessor for uniqueMetaData
     * @return Returns the uniqueMetaData.
     */
    public final UniqueMetaData getUniqueMetaData()
    {
        return uniqueMetaData;
    }

    // ------------------------------ Mutators ---------------------------------
 
    /**
     * Add a new ColumnMetaData element
     * @param colmd The Column MetaData 
     */
    public void addColumn(ColumnMetaData colmd)
    {
        columns.add(colmd);
        colmd.parent = this;
    }

    /**
     * Mutator for the Embedded MetaData 
     * @param embeddedMetaData The embeddedMetaData to set.
     */
    public final void setEmbeddedMetaData(EmbeddedMetaData embeddedMetaData)
    {
        this.embeddedMetaData = embeddedMetaData;
    }
 
    /**
     * Mutator for the Foreign Key MetaData 
     * @param foreignKeyMetaData The foreignKeyMetaData to set.
     */
    public final void setForeignKeyMetaData(ForeignKeyMetaData foreignKeyMetaData)
    {
        this.foreignKeyMetaData = foreignKeyMetaData;
    }

    /**
     * Mutator for the Index MetaData 
     * @param indexMetaData The indexMetaData to set.
     */
    public final void setIndexMetaData(IndexMetaData indexMetaData)
    {
        this.indexMetaData = indexMetaData;
    }

    /**
     * Mutator for the Unique MetaData 
     * @param uniqueMetaData The uniqueMetaData to set.
     */
    public final void setUniqueMetaData(UniqueMetaData uniqueMetaData)
    {
        this.uniqueMetaData = uniqueMetaData;
    }
}