/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import org.datanucleus.util.StringUtils;

/**
 * Representation of the Meta-Data for a column mapping of a field.
 * JDO metadata represented is :-
 * <PRE>
 * &lt;!ELEMENT column (extension*)?&gt;
 * &lt;!ATTLIST column name CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column target CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column target-field CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column jdbc-type CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column sql-type CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column length CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column scale CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column allows-null CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column default-value CDATA #IMPLIED&gt;
 * &lt;!ATTLIST column insert-value CDATA #IMPLIED&gt;
 * </PRE>
 *
 * @since 1.1
 * @version $Revision: 1.20 $
 */
public class ColumnMetaData extends MetaData
{
    /** column name. */
    protected String name;

    /** target column name. */
    protected String target;

    /** target field/property name. */
    protected String targetMember;

    /** jdbc-type to use (if any). */
    protected String jdbcType;

    /** sql-type to use (if any). */
    protected String sqlType;

    /** length to use (if any). Also known as precision */
    protected Integer length;

    /** scale to use (if any). */
    protected Integer scale;
   
    /** allows-null tag value. */
    protected Boolean allowsNull;

    /** column default value (when constructing the table with this column). */
    protected String defaultValue;

    /** value to use when inserting this column in the datastore (the column is not mapped to a field/property) */
    protected String insertValue;

    /** Whether this column is to be inserted when the owning object is inserted. JPA 1.0 attribute. */
    protected boolean insertable = true;

    /** Whether this column can be updated when the owning object is updated. JPA 1.0 attribute. */
    protected boolean updateable = true;

    /** unique tag value. JPA 1.0 attribute. */
    protected boolean unique;

    /**
     * Creates a ColumnMetaData by copying contents from <code>colmd</code>.
     * @param parent Parent MetaData component
     * @param colmd MetaData for the column
     */
    public ColumnMetaData(final MetaData parent, final ColumnMetaData colmd)
    {
        super(parent);

        name = colmd.getName();
        target = colmd.getTarget();
        targetMember = colmd.getTargetMember();
        jdbcType = colmd.getJdbcType();
        sqlType = colmd.getSqlType();
        length = colmd.getLength();
        scale = colmd.getScale();
        allowsNull = colmd.allowsNull;
        defaultValue = colmd.getDefaultValue();
        insertValue = colmd.getInsertValue();
        insertable = colmd.getInsertable();
        updateable = colmd.getUpdateable();
        unique = colmd.getUnique();
    }

    /**
     * Convenience constructor specifying just the column name, and the parent metadata component.
     * Assigns nulls to other parameters.
     * @param parent Parent MetaData component
     * @param name Name of the column
     */
    public ColumnMetaData(final MetaData parent, final String name)
    {
        this(parent, name, null, null, null, null, null, null, null, null, null, null, null, null);
    }

    /**
     * Constructor.
     * @param parent parent MetaData instance
     * @param name field name 
     * @param target target
     * @param targetMember target field/property
     * @param jdbcType JDBC Type to use
     * @param sqlType SQL Type to use
     * @param length length of field
     * @param scale scale of field
     * @param allowsNull Whether nulls are allowed
     * @param defaultValue The default value for the column
     * @param insertValue The insert value for the column
     * @param insertable Whether this column is insertable
     * @param updateable Whether this column is updateable
     * @param unique Whether this column is unique
     */
    public ColumnMetaData(MetaData parent,
                          final String name,
                          final String target,
                          final String targetMember,
                          final String jdbcType,
                          final String sqlType,
                          final String length,
                          final String scale,
                          final String allowsNull,
                          final String defaultValue,
                          final String insertValue,
                          final String insertable,
                          final String updateable,
                          final String unique)
    {
        super(parent);

        this.name = (StringUtils.isWhitespace(name) ? null : name);

        this.target = (StringUtils.isWhitespace(target) ? null : target);
        this.targetMember = (StringUtils.isWhitespace(targetMember) ? null : targetMember);

        this.jdbcType = (StringUtils.isWhitespace(jdbcType) ? null : jdbcType);
        /*if (this.jdbcType != null && !JDBCUtils.isValidJDBCType(this.jdbcType))
        {
            // Invalid "jdbc-type" (maybe not supported by JPOX?)
            // TODO This uses RDBMS-specific code so is commented out
            throw new InvalidMetaDataException(LOCALISER, "044118", this.jdbcType);
        }*/

        this.sqlType = (StringUtils.isWhitespace(sqlType) ? null : sqlType);

        if (length != null && length.length() > 0)
        {
            this.length = Integer.valueOf(length);
        }

        if (scale != null && scale.length() > 0)
        {
            this.scale = Integer.valueOf(scale);
        }

        if (allowsNull != null)
        {
            if (allowsNull.equalsIgnoreCase("true"))
            {
                this.allowsNull = Boolean.TRUE;
            }
            else if (allowsNull.equalsIgnoreCase("false"))
            {
                this.allowsNull = Boolean.FALSE;
            }
        }

        this.defaultValue = (StringUtils.isWhitespace(defaultValue) ? null : defaultValue);
        this.insertValue = (StringUtils.isWhitespace(insertValue) ? null : insertValue);

        if (insertable != null && insertable.equalsIgnoreCase("false"))
        {
            this.insertable = false;
        }
        else
        {
            // Insertable unless specified otherwise
            this.insertable = true;
        }
        if (updateable != null && updateable.equalsIgnoreCase("false"))
        {
            this.updateable = false;
        }
        else
        {
            // Updateable unless specified otherwise
            this.updateable = true;
        }
        if (unique != null && unique.equalsIgnoreCase("true"))
        {
            this.unique = true;
        }
        else
        {
            // Non-unique unless specified otherwise
            this.unique = false;
        }
    }

    /**
     * Accessor for the name
     * @return name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Accessor for the column that is the target of this column in the referenced table.
     * @return target tag value
     */
    public String getTarget()
    {
        return target;
    }

    /**
     * Accessor for the field/property that is the target of this column in the referenced class.
     * @return field/property that is the target
     */
    public String getTargetMember()
    {
        return targetMember;
    }

    /**
     * Accessor for the jdbc-type tag value
     * @return jdbc-type tag value
     */
    public String getJdbcType()
    {
        return jdbcType;
    }

    /**
     * Accessor for the sql-type tag value
     * @return sql-type tag value
     */
    public String getSqlType()
    {
        return sqlType;
    }

    /**
     * Accessor for the length tag value. Also known as precision
     * @return length tag value
     */
    public Integer getLength()
    {
        return length;
    }

    /**
     * Accessor for the scale tag value
     * @return scale tag value
     */
    public Integer getScale()
    {
        return scale;
    }
 
    /**
     * Accessor for whether the nulls allowed flag has been set.
     * @return Whether it was set.
     */
    public boolean isAllowsNullSet()
    {
        return (allowsNull != null);
    }

    /**
     * Accessor for the nulls-allowed tag value
     * @return nulls-allowed tag value
     */
    public boolean isAllowsNull()
    {
        if (allowsNull == null)
        {
            return false;
        }
        else
        {
            return allowsNull.booleanValue();
        }
    }

    /**
     * Accessor for the default value
     * @return default value
     */
    public String getDefaultValue()
    {
        return defaultValue;
    }

    /**
     * Accessor for the insert value
     * @return insert value
     */
    public String getInsertValue()
    {
        return insertValue;
    }

    /**
     * Accessor for whether this column can be inserted when the owning object is inserted.
     * @return Whether it can be inserted.
     */
    public boolean getInsertable()
    {
        return insertable;
    }

    /**
     * Accessor for whether this column can be update when the owning object is updated.
     * @return Whether it can be updated.
     */
    public boolean getUpdateable()
    {
        return updateable;
    }

    /**
     * Accessor for the unique tag value
     * @return unique tag value
     */
    public boolean getUnique()
    {
        return unique;
    }

    // ------------------------------- Mutators --------------------------------

    /**
     * Mutator for the scale 
     * @param scale The scale to set.
     */
    public final void setScale(Integer scale)
    {
        this.scale = scale;
    }

    /**
     * Mutator for the scale
     * @param scale The scale to set.
     */
    public final void setScale(int scale)
    {
        this.scale = new Integer(scale);
    }

    /**
     * Mutator for the JDBC type
     * @param jdbcType The jdbcType to set.
     */
    public final void setJdbcType(String jdbcType)
    {
        this.jdbcType = jdbcType;
    }

    /**
     * Mutator for the length. Also known as precision 
     * @param length The length to set.
     */
    public final void setLength(Integer length)
    {
        this.length = length;
    }

    /**
     * Mutator for the length. Also known as precision 
     * @param length The length to set.
     */
    public final void setLength(int length)
    {
        this.length = new Integer(length);
    }    

    /**
     * Mutator for the name 
     * @param name The name to set.
     */
    public final void setName(String name)
    {
        this.name = name;
    }

    /**
     * Mutator for whether nulls are allowed.
     * @param allowsNull The allowsNull to set.
     */
    public final void setAllowsNull(Boolean allowsNull)
    {
        this.allowsNull = allowsNull;
    }

    /**
     * Mutator for the SQL type
     * @param sqlType The sqlType to set.
     */
    public final void setSqlType(String sqlType)
    {
        this.sqlType = sqlType;
    }

    // ------------------------------- Utilities -------------------------------

    /**
     * Returns a string representation of the object using a prefix
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<column");
        if (name != null)
        {
            sb.append(" name=\"" + name + "\"");
        }
        if (target != null)
        {
            sb.append(" target=\"" + target + "\"");
        }
        if (targetMember != null)
        {
            sb.append(" target-field=\"" + targetMember + "\"");
        }
        if (jdbcType != null)
        {
            sb.append(" jdbc-type=\"" + jdbcType + "\"");
        }
        if (sqlType != null)
        {
            sb.append(" sql-type=\"" + sqlType + "\"");
        }
        if (allowsNull != null)
        {
            sb.append(" allows-null=\"" + allowsNull + "\"");
        }
        if (length != null)
        {
            sb.append(" length=\"" + length + "\"");
        }
        if (scale != null)
        {
            sb.append(" scale=\"" + scale + "\"");
        }
        if (defaultValue != null)
        {
            sb.append(" default-value=\"" + defaultValue + "\"");
        }
        if (insertValue != null)
        {
            sb.append(" insert-value=\"" + insertValue + "\"");
        }

        if (extensions != null && extensions.size() > 0)
        {
            sb.append(">\n");

            // Add extensions
            sb.append(super.toString(prefix + indent,indent));

            sb.append(prefix).append("</column>\n");
        }
        else
        {
            sb.append("/>\n");
        }

        return sb.toString();
    }
}