/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.io.Serializable;

import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.StringUtils;

/**
 * Representation of the details of an object stored in a container.
 * This can be an element in a collection/array, or the key/value of a Map.
 * 
 * @version $Revision: 1.4 $
 */
class ContainerComponent implements Serializable
{
    /** Whether the component is stored embedded. */
    protected Boolean embedded;

    /** Whether the component is stored serialised. */
    protected Boolean serialized;

    /** Whether the component is dependent on the container (i.e should be deleted with the container). */
    protected Boolean dependent;

    /** Type of the component. */
    protected String type = "java.lang.Object";

    /** ClassMetaData for the component. */
    protected AbstractClassMetaData classMetaData;

    /**
     * Default constructor.
     */
    public ContainerComponent()
    {
        // Do nothing
    }

    /**
     * Simple constructor that sets the component using typical input metadata.
     * This does not set the "classMetaData" field.
     * @param packageName Name of the package that this component is in
     * @param type Type of the component
     * @param embedded Whether it is embedded "true", "false" or null
     * @param serialized Whether it is serialised "true", "false" or null
     * @param dependent Whether it is dependent on the container "true", "false" or null
     */
    public ContainerComponent(String packageName, String type, String embedded, String serialized, String dependent)
    {
        if (!StringUtils.isWhitespace(type))
        {
            if (ClassUtils.isPrimitiveArrayType(type))
            {
                this.type = type;
            }
            else if (ClassUtils.isPrimitiveType(type))
            {
                this.type = type;
            }
            else if (packageName != null)
            {
                // Apply naming rules - if no package, then assumed to be in same package as this class
                this.type = ClassUtils.createFullClassName(packageName, type);
            }
            else
            {
                this.type = type;
            }
        }
        if (embedded != null)
        {
            if (embedded.equalsIgnoreCase("true"))
            {
                this.embedded = Boolean.TRUE;
            }
            else if (embedded.equalsIgnoreCase("false"))
            {
                this.embedded = Boolean.FALSE;
            }
        }
        if (dependent != null)
        {
            if (dependent.equalsIgnoreCase("true"))
            {
                this.dependent = Boolean.TRUE;
            }
            else if (dependent.equalsIgnoreCase("false"))
            {
                this.dependent = Boolean.FALSE;
            }
        }
        if (serialized != null)
        {
            if (serialized.equalsIgnoreCase("true"))
            {
                this.serialized = Boolean.TRUE;
            }
            else if (serialized.equalsIgnoreCase("false"))
            {
                this.serialized = Boolean.FALSE;
            }
        }
    }

    public String toString()
    {
        return "Type=" + type + " embedded=" + embedded + " serialized=" + serialized + " dependent=" + dependent + 
            " cmd=" + classMetaData;
    }
}