/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

/**
 * Default implementation of MetaDataFactory.
 *
 * @version $Revision: 1.4 $
 */
public class DefaultMetaDataFactory implements MetaDataFactory
{
    /**
     * Constructor for a ClassMetaData.
     * Takes the basic string information found in the XML/annotations.
     * @param pmd MetaData for the package that this class belongs to
     * @param name Name of class
     * @param identityType Type of identity
     * @param objectidClass Class of the object id
     * @param requiresExtent Whether the class requires an extent
     * @param detachable Whether this is detachable
     * @param embeddedOnly embedded-only tag
     * @param modifier persistence modifier for the class
     * @param persistenceCapableSuperclass PC superclass (optional)
     * @param catalog Name for catalog
     * @param schema Name for schema
     * @param table Name of the table where to persist objects of this type
     * @param entityName the entity name required by JPA 4.3.1
     */
    public ClassMetaData newClassObject(PackageMetaData pmd, String name, String identityType, String objectidClass,
            String requiresExtent, String detachable, String embeddedOnly, String modifier, String persistenceCapableSuperclass,
            String catalog, String schema, String table, String entityName)
    {
        ClassMetaData cmd = new ClassMetaData(pmd, name, identityType, objectidClass, requiresExtent, detachable,
            embeddedOnly, modifier, persistenceCapableSuperclass, catalog, schema, table, entityName);
        return cmd;
    }

    /**
     * Constructor for a FieldMetaData.
     * @param md MetaData for the class that this field belongs to
     * @param name Name of the field
     * @param pk Whether it is a part of the PK
     * @param modifier persistence-modifier
     * @param defaultFetchGroup Whether it is in the DFG
     * @param nullValue Action on null value inserts
     * @param embedded Whether it is embedded
     * @param serialized Whether it is serialised
     * @param dependent Whether it is dependent for deletes
     * @param mappedBy Field in other class that it is mapped using
     * @param column Column name to store it
     * @param table Table where it is stored
     * @param catalog Catalog that the table is in
     * @param schema Schema that the table is in
     * @param deleteAction Any FK delete action
     * @param indexed Whether it is indexed
     * @param unique Whether it is unique
     * @param recursionDepth Recursion depth to apply on fetch-plan operations
     * @param loadFetchGroup Whether to load the fetch group
     * @param valueStrategy Strategy for generating values for this field
     * @param sequence Sequence name if the strategy is "sequence"
     * @param fieldType Type of the field
     * @return MetaData for the field
     */
    public FieldMetaData newFieldObject(MetaData md, String name, String pk, String modifier, String defaultFetchGroup,
            String nullValue, String embedded, String serialized, String dependent, String mappedBy, String column,
            String table, String catalog, String schema, 
            String deleteAction, String indexed, String unique, String recursionDepth, String loadFetchGroup,
            String valueStrategy, String sequence, String fieldType)
    {
        FieldMetaData fmd = new FieldMetaData(md, name, pk, modifier, defaultFetchGroup, nullValue, embedded, serialized,
            dependent, mappedBy, column, table, catalog, schema, deleteAction, indexed, unique, recursionDepth, loadFetchGroup, 
            valueStrategy, sequence, fieldType);
        return fmd;
    }

    /**
     * Method to create a new field object copying from the supplied field
     * @param md Parent metadata
     * @param referenceFmd The reference field
     * @return FieldMetaData object
     */
    public FieldMetaData newFieldObject(MetaData md, AbstractMemberMetaData referenceFmd)
    {
        FieldMetaData fmd = new FieldMetaData(md, referenceFmd);
        return fmd;
    }

    /**
     * Constructor for a PropertyMetaData.
     * @param md MetaData for the interface that this property belongs to
     * @param name Name of the field
     * @param pk Whether it is a part of the PK
     * @param modifier persistence-modifier
     * @param defaultFetchGroup Whether it is in the DFG
     * @param nullValue Action on null value inserts
     * @param embedded Whether it is embedded
     * @param serialized Whether it is serialised
     * @param dependent Whether it is dependent for deletes
     * @param mappedBy Field in other class that it is mapped using
     * @param column Column name to store it
     * @param table Table where it is stored
     * @param catalog Catalog that the table is in
     * @param schema Schema that the table is in
     * @param deleteAction Any FK delete action
     * @param indexed Whether it is indexed
     * @param unique Whether it is unique
     * @param recursionDepth Recursion depth to apply on fetch-plan operations
     * @param loadFetchGroup Whether to load the fetch group
     * @param valueStrategy Strategy for generating values for this field
     * @param sequence Sequence name if the strategy is "sequence"
     * @param fieldType Type of the field
     * @param fieldName Name of the field (relates to the implementation of this)
     * @return MetaData for the field
     */
    public PropertyMetaData newPropertyObject(MetaData md, String name, String pk, String modifier, String defaultFetchGroup,
            String nullValue, String embedded, String serialized, String dependent, String mappedBy, String column,
            String table, String catalog, String schema, 
            String deleteAction, String indexed, String unique, String recursionDepth, String loadFetchGroup,
            String valueStrategy, String sequence, String fieldType, String fieldName)
    {
        PropertyMetaData pmd = new PropertyMetaData(md, name, pk, modifier, defaultFetchGroup, nullValue, embedded, serialized,
            dependent, mappedBy, column, table, catalog, schema, deleteAction, indexed, unique, recursionDepth, loadFetchGroup, 
            valueStrategy, sequence, fieldType, fieldName);
        return pmd;
    }

    /**
     * Method to create a new property object copying from the supplied object.
     * @param md Parent metadata
     * @param referencePmd Property to copy from
     * @return PropertyMetaData object
     */
    public PropertyMetaData newPropertyObject(MetaData md, PropertyMetaData referencePmd)
    {
        PropertyMetaData pmd = new PropertyMetaData(md, referencePmd);
        return pmd;        
    }

    /**
     * Constructor for an InterfaceMetaData.
     * Takes the basic string information found in the XML/annotations.
     * @param pmd MetaData for the package that this class belongs to
     * @param name Name of class
     * @param identityType Type of identity
     * @param objectidClass Class of the object id
     * @param requiresExtent Whether the class requires an extent
     * @param detachable Whether this is detachable
     * @param embeddedOnly embedded-only tag
     * @param catalog Name for catalog
     * @param schema Name for schema
     * @param table Name of the table where to persist objects of this type
     * @param entityName the entity name required by JPA 4.3.1
     */
    public InterfaceMetaData newInterfaceObject(final PackageMetaData pmd,
                             final String name,
                             final String identityType,
                             final String objectidClass,
                             final String requiresExtent,
                             final String detachable,
                             final String embeddedOnly,
                             final String catalog,
                             final String schema,
                             final String table,
                             final String entityName)
    {
        InterfaceMetaData imd = new InterfaceMetaData(pmd,
            name,
            identityType,
            objectidClass,
            requiresExtent,
            detachable,
            embeddedOnly,
            catalog,
            schema,
            table,
            entityName);
        return imd;
    }
}