/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2004 Andy Jefferson - added initialise() method
    ...
**********************************************************************/
package org.datanucleus.metadata;

import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Representation of a discriminator in an inheritance strategy.
 * 
 * @since 1.1
 * @version $Revision: 1.21 $
 */
public class DiscriminatorMetaData extends MetaData
{
    /** strategy tag value. */
    protected DiscriminatorStrategy strategy=null;

    /** Column name of discriminator */
    protected String columnName=null;

    /** Value for discriminator column */
    protected String value=null;

    /** Discriminator column */
    protected ColumnMetaData columnMetaData=null;

    /** Whether the discriminator is indexed or not and whether it is unique */
    protected IndexedValue indexed=null;

    /** Definition of any indexing of the discriminator column. */
    protected IndexMetaData indexMetaData;

    /**
     * Constructor.
     * @param parent parent InheritanceMetaData instance
     * @param columnName Name of the column
     * @param value Value for discriminator column
     * @param strategy The strategy
     * @param indexed The indexed tag
     */
    public DiscriminatorMetaData(InheritanceMetaData parent,
            					 final String columnName,
            					 final String value,
                               	 final String strategy,
                               	 final String indexed)
    {
        super(parent);

        this.columnName = (StringUtils.isWhitespace(columnName) ? null : columnName);
        this.strategy = DiscriminatorStrategy.getDiscriminatorStrategy(strategy);
        if (value != null && strategy == null)
        {
            // Default to value strategy if a value is specified
            this.strategy = DiscriminatorStrategy.VALUE_MAP;
        }
        else if (strategy == null)
        {
            // Default to class strategy if no value is specified
            this.strategy = DiscriminatorStrategy.CLASS_NAME;
        }
        this.indexed = IndexedValue.getIndexedValue(indexed);
        if (StringUtils.isWhitespace(value))
        {
            if (this.strategy == DiscriminatorStrategy.VALUE_MAP)
            {
                String className = ((AbstractClassMetaData)parent.getParent()).getFullClassName();
                NucleusLogger.METADATA.warn(LOCALISER.msg("044103", className));
                this.value = className;
            }
            else
            {
                this.value = null;
            }
        }
        else
        {
            this.value = value;
        }
    }

    /**
     * Constructor.
     * @param parent parent InheritanceMetaData instance
     * @param dmd DiscriminatorMetaData
     */
    public DiscriminatorMetaData(InheritanceMetaData parent, final DiscriminatorMetaData dmd)
    {
        super(parent);

        this.columnName = dmd.columnName;
        this.value = dmd.value;
        this.strategy = dmd.strategy;
        this.indexed = dmd.indexed;
        columnMetaData = new ColumnMetaData(this, dmd.columnMetaData);
        if( dmd.indexMetaData != null)
        {
            indexMetaData = new IndexMetaData(dmd.indexMetaData);
        }
    }

    /**
     * Initialisation method. This should be called AFTER using the populate
     * method if you are going to use populate. It creates the internal
     * convenience arrays etc needed for normal operation.
     */
    public void initialise()
    {
        // Interpret the "indexed" value to create our IndexMetaData where it wasn't specified that way
        if (indexMetaData == null && columnMetaData != null && indexed != null && indexed != IndexedValue.FALSE)
        {
            indexMetaData = new IndexMetaData(null, null, (indexed == IndexedValue.UNIQUE) ? "true" : "false");
            // Note we only support a single column discriminator here
            indexMetaData.addColumn(columnMetaData);
        }
        if (indexMetaData != null)
        {
            indexMetaData.initialise();
        }

        if (columnMetaData == null && columnName != null)
        {
            columnMetaData = new ColumnMetaData(this, columnName);
            columnMetaData.initialise();
        }

        setInitialised();
    }

    // --------------------------- Accessors/Mutators ---------------------------

    /**
     * Accessor for column MetaData.
     * @return Returns the column MetaData.
     */
    public ColumnMetaData getColumnMetaData()
    {
        return columnMetaData;
    }

    /**
     * Mutator for column MetaData.
     * @param columnMetaData The column MetaData to set.
     */
    public void setColumnMetaData(ColumnMetaData columnMetaData)
    {
        this.columnMetaData = columnMetaData;
        this.columnMetaData.parent = this;
    }

    /**
     * Accessor for indexMetaData
     * @return Returns the indexMetaData.
     */
    public final IndexMetaData getIndexMetaData()
    {
        return indexMetaData;
    }

    /**
     * Mutator for the index MetaData 
     * @param indexMetaData The indexMetaData to set.
     */
    public final void setIndexMetaData(IndexMetaData indexMetaData)
    {
        this.indexMetaData = indexMetaData;
    }

    /**
     * Accessor for value.
     * @return Returns the value.
     */
    public String getValue()
    {
        return value;
    }

    /**
     * Accessor for columnName.
     * @return Returns the columnName.
     */
    public String getColumnName()
    {
        // Return the column variant if specified, otherwise the name field
        if (columnMetaData != null && columnMetaData.getName() != null)
        {
            return columnMetaData.getName();
        }
        return columnName;
    }

    /**
     * Accessor for strategy.
     * @return Returns the strategy.
     */
    public final DiscriminatorStrategy getStrategy()
    {
        return strategy;
    }

    /**
     * Accessor for indexed value.
     * @return Returns the indexed value.
     */
    public final IndexedValue getIndexedValue()
    {
        return indexed;
    }

    /**
     * Mutator for columnName.
     * @param columnName The columnName to set.
     */
    public void setColumnName(String columnName)
    {
        this.columnName = columnName;
    }

    //  ----------------------------- Utilities ---------------------------------

    /**
     * Returns a string representation of the object using a prefix
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<discriminator");
        if (strategy != null)
        {
            sb.append(" strategy=\"" + strategy.toString() + "\"");
        }
        if (columnName != null && columnMetaData == null)
        {
            sb.append(" column=\"" + columnName + "\"");
        }
        if (value != null)
        {
            sb.append(" value=\"" + value + "\"");
        }
        if (indexed != null)
        {
            sb.append(" indexed=\"" + indexed.toString() + "\"");
        }
        sb.append(">\n");

        // Column MetaData
        if (columnMetaData != null)
        {
            sb.append(columnMetaData.toString(prefix + indent,indent));
        }

        // Add index
        if (indexMetaData != null)
        {
            sb.append(indexMetaData.toString(prefix + indent,indent));
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</discriminator>\n");

        return sb.toString();
    }
}