/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2004 Andy Jefferson - toString(), MetaData, javadocs
2004 Andy Jefferson - nullIndicatorColumn/Value, ownerField
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * This element specifies the mapping for an embedded type. It contains multiple field elements, 
 * one for each field in the type.
 * <P>
 * The <B>null-indicator-column</B> optionally identifies the name of the column used to indicate 
 * whether the embedded instance is null. By default, if the value of this column is null, then the
 * embedded instance is null. This column might be mapped to a field of the embedded instance but 
 * might be a synthetic column for the sole purpose of indicating a null reference.
 * The <B>null-indicator-value</B> specifies the value to indicate that the embedded instance is null. 
 * This is only used for non-nullable columns.
 * If <B>null-indicator-column</B> is omitted, then the embedded instance is assumed always to exist.
 *
 * @since 1.1
 * @version $Revision: 1.42 $
 */
public class EmbeddedMetaData extends MetaData
{
    /** Fields/properties of the embedded object. */
    protected final List members = new ArrayList();

    /** Name of the owner field/property in the embedded object. */
    protected final String ownerMember;

    /** Name of a column used for determining if the embedded object is null */
    protected final String nullIndicatorColumn;

    /** Value in the null column indicating that the embedded object is null */
    protected final String nullIndicatorValue;

    // -------------------------------------------------------------------------
    // Fields below here are not represented in the output MetaData. They are
    // for use internally in the operation of the JDO system. The majority are
    // for convenience to save iterating through the fields since the fields
    // are fixed once initialised.

    protected AbstractMemberMetaData fieldMetaData[];
    
    /**
     * Constructor to create a copy of the passed metadata using the provided parent.
     * @param parent The parent
     * @param embmd The metadata to copy
     */
    public EmbeddedMetaData(MetaData parent, EmbeddedMetaData embmd)
    {
        super(parent);
        this.ownerMember = embmd.ownerMember;
        this.nullIndicatorColumn = embmd.nullIndicatorColumn;
        this.nullIndicatorValue = embmd.nullIndicatorValue;
        for (int i=0;i<embmd.members.size();i++)
        {
            if (embmd.members.get(i) instanceof FieldMetaData)
            {
                addMember(new FieldMetaData(this,(AbstractMemberMetaData)embmd.members.get(i)));
            }
            else
            {
                addMember(new PropertyMetaData(this,(PropertyMetaData)embmd.members.get(i)));
            }
        }
    }

    /**
     * Constructor.
     * @param parent Owning metadata
     * @param ownerField The owner field
     * @param nullColumn Column to use for determining null embedded object
     * @param nullValue Value of column to use for determining null embedded object
     */
    public EmbeddedMetaData(MetaData parent,
                            String ownerField,
                            String nullColumn,
                            String nullValue)
    {
        super(parent);
        this.ownerMember = (StringUtils.isWhitespace(ownerField) ? null : ownerField);
        nullIndicatorColumn = (StringUtils.isWhitespace(nullColumn) ? null : nullColumn);
        nullIndicatorValue = (StringUtils.isWhitespace(nullValue) ? null : nullValue);
    }

    /**
     * Method to populate the embedded MetaData.
     * This performs checks on the validity of the field types for embedding.
     * @param clr The class loader to use where necessary
      * @param primary the primary ClassLoader to use (or null)
    */
    public void populate(ClassLoaderResolver clr, ClassLoader primary)
    {
        // Find the class that the embedded fields apply to
        MetaData md = getParent();
        AbstractMemberMetaData apmd = null; // Field that has <embedded>
        AbstractClassMetaData embCmd = null; // The embedded class
        String embeddedType = null; // Name of the embedded type
        if (md instanceof AbstractMemberMetaData)
        {
            // PC embedded in PC object
            apmd = (AbstractMemberMetaData)md;
            embeddedType = apmd.getTypeName();
            MetaDataManager mmgr = apmd.getAbstractClassMetaData().getMetaDataManager();
            embCmd = mmgr.getMetaDataForClassInternal(apmd.getType(), clr);
            if (embCmd == null && apmd.getFieldTypes() != null && apmd.getFieldTypes().length == 1)
            {
                // The specified field is not embeddable, nor is it persistent-interface, so try field-type for embedding
                embCmd = mmgr.getMetaDataForClassInternal(clr.classForName(apmd.getFieldTypes()[0]), clr);
            }
            if (embCmd == null)
            {
                NucleusLogger.METADATA.error(LOCALISER.msg("044121", 
                    apmd.getFullFieldName(), apmd.getTypeName()));
                throw new InvalidMetaDataException(LOCALISER,
                    "044121", apmd.getFullFieldName(), apmd.getTypeName());
            }
        }
        else if (md instanceof ElementMetaData)
        {
            // PC element embedded in collection
            ElementMetaData elemmd = (ElementMetaData)md;
            apmd = (AbstractMemberMetaData)elemmd.getParent();
            embeddedType = apmd.getCollection().getElementType();
            try
            {
                Class cls = clr.classForName(embeddedType, primary);
                embCmd = apmd.getAbstractClassMetaData().getMetaDataManager().getMetaDataForClassInternal(cls, clr);
            }
            catch (ClassNotResolvedException cnre)
            {
                // Should be caught by populating the Collection
            }
            if (embCmd == null)
            {
                NucleusLogger.METADATA.error(LOCALISER.msg("044122", 
                    apmd.getFullFieldName(), apmd.getCollection().getElementType()));
                throw new InvalidMetaDataException(LOCALISER,
                    "044122", apmd.getFullFieldName(), apmd.getTypeName());
            }
        }
        else if (md instanceof KeyMetaData)
        {
            // PC key embedded in Map
            KeyMetaData keymd = (KeyMetaData)md;
            apmd = (AbstractMemberMetaData)keymd.getParent();
            embeddedType = apmd.getMap().getKeyType();
            try
            {
                Class cls = clr.classForName(embeddedType, primary);
                embCmd = apmd.getAbstractClassMetaData().getMetaDataManager().getMetaDataForClassInternal(cls, clr);
            }
            catch (ClassNotResolvedException cnre)
            {
                // Should be caught by populating the Map
            }
            if (embCmd == null)
            {
                NucleusLogger.METADATA.error(LOCALISER.msg("044123", 
                    apmd.getFullFieldName(), apmd.getMap().getKeyType()));
                throw new InvalidMetaDataException(LOCALISER,
                    "044123", apmd.getFullFieldName(), embeddedType);
            }
        }
        else if (md instanceof ValueMetaData)
        {
            // PC value embedded in Map
            ValueMetaData valuemd = (ValueMetaData)md;
            apmd = (AbstractMemberMetaData)valuemd.getParent();
            embeddedType = apmd.getMap().getValueType();
            try
            {
                Class cls = clr.classForName(embeddedType, primary);
                embCmd = apmd.getAbstractClassMetaData().getMetaDataManager().getMetaDataForClassInternal(cls, clr);
            }
            catch (ClassNotResolvedException cnre)
            {
                // Should be caught by populating the Map
            }
            if (embCmd == null)
            {
                NucleusLogger.METADATA.error(LOCALISER.msg("044124", 
                    apmd.getFullFieldName(), apmd.getMap().getValueType()));
                throw new InvalidMetaDataException(LOCALISER,
                    "044124", apmd.getFullFieldName(), embeddedType);
            }
        }

        // Check that all "members" are of the correct type for the embedded object
        Iterator memberIter = members.iterator();
        while (memberIter.hasNext())
        {
            Object fld = memberIter.next();
            // TODO Should allow PropertyMetaData here I think
            if (embCmd instanceof InterfaceMetaData && fld instanceof FieldMetaData)
            {
                // Cannot have a field within a persistent interface
                throw new InvalidMetaDataException(LOCALISER, "044129",
                    apmd.getFullFieldName(), embCmd.getFullClassName(), ((AbstractMemberMetaData)fld).getName());
            }
        }

        // Add fields for the class that aren't in the <embedded> block using Reflection.
        // NOTE 1 : We ignore fields in superclasses
        // NOTE 2 : We ignore "enhanced" fields (added by the JDO enhancer)
        // NOTE 3 : We ignore inner class fields (containing "$") 
        // NOTE 4 : We sort the fields into ascending alphabetical order
        Class embeddedClass = null;
        Collections.sort(members);
        try
        {
            // Load the embedded class
            embeddedClass = clr.classForName(embeddedType, primary);

            // TODO Cater for properties in the populating class when the user defines using setters

            // Get all (reflected) fields in the populating class
            Field[] cls_fields=embeddedClass.getDeclaredFields();
            for (int i=0;i<cls_fields.length;i++)
            {
                // Limit to fields in this class, that aren't enhanced fields
                // that aren't inner class fields, and that aren't static
                if (cls_fields[i].getDeclaringClass().getName().equals(embeddedType) &&
                    !cls_fields[i].getName().startsWith("jdo") &&
                    !ClassUtils.isInnerClass(cls_fields[i].getName()) &&
                    !Modifier.isStatic(cls_fields[i].getModifiers()))
                {
                    // Find if there is a AbstractMemberMetaData for this field.
                    // This is possible as AbstractMemberMetaData implements Comparable
                    if (Collections.binarySearch(members, cls_fields[i].getName()) < 0)
                    {
                        // Add a default FieldMetaData for this field.
                        NucleusLogger.METADATA.debug(LOCALISER.msg("044125", 
                            cls_fields[i].getName(), embeddedType, apmd.getClassName(true)));
                        FieldMetaData omittedFmd = new FieldMetaData(this, cls_fields[i].getName());
                        members.add(omittedFmd);
                        Collections.sort(members);
                    }
                }
            }
        }
        catch (Exception e)
        {
            NucleusLogger.METADATA.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        }

        // add properties of interface, only if interface
        if (embCmd instanceof InterfaceMetaData)
        {
            try
            {
                // Get all (reflected) fields in the populating class
                Method[] clsMethods = embeddedClass.getDeclaredMethods();
                for (int i=0; i<clsMethods.length; i++)
                {
                    // Limit to methods in this class, that aren't enhanced fields
                    // that aren't inner class fields, and that aren't static
                    if (clsMethods[i].getDeclaringClass().getName().equals(embeddedType) &&
                        (clsMethods[i].getName().startsWith("get") || clsMethods[i].getName().startsWith("is")) &&
                        !ClassUtils.isInnerClass(clsMethods[i].getName()))
                    {
                        String fieldName = ClassUtils.getFieldNameForJavaBeanGetter(clsMethods[i].getName() );
                        // Find if there is a PropertyMetaData for this field.
                        // This is possible as PropertyMetaData implements Comparable
                        if (Collections.binarySearch(members,fieldName) < 0)
                        {
                            // Add a default PropertyMetaData for this field.
                            NucleusLogger.METADATA.debug(LOCALISER.msg("044060",
                                fieldName,apmd.getClassName(true)));
                            PropertyMetaData fmd=new PropertyMetaData(this, fieldName);
                            members.add(fmd);
                            Collections.sort(members);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                NucleusLogger.METADATA.error(e.getMessage(), e);
                throw new RuntimeException(e.getMessage());
            }
        }
        Collections.sort(members);

        memberIter = members.iterator();
        while (memberIter.hasNext())
        {
            AbstractMemberMetaData fieldFmd = (AbstractMemberMetaData)memberIter.next();
            if (fieldFmd instanceof FieldMetaData)
            {
                Field cls_field = null;
                try
                {
                    cls_field = embeddedClass.getDeclaredField(fieldFmd.getName());
                }
                catch (Exception e)
                {
                    // MetaData field doesn't exist in the class!
                    throw new InvalidMetaDataException(LOCALISER,
                        "044071", embeddedClass.getName(), fieldFmd.getFullFieldName());
                }
                fieldFmd.populate(clr, cls_field, null, primary);
            }
            else
            {
                Method cls_method = null;
                try
                {
                    try
                    {
                        cls_method = embeddedClass.getDeclaredMethod(
                            ClassUtils.getJavaBeanGetterName(fieldFmd.getName(),true),null);
                    }
                    catch(Exception e)
                    {
                        cls_method = embeddedClass.getDeclaredMethod(
                            ClassUtils.getJavaBeanGetterName(fieldFmd.getName(),false),null);
                    }
                }
                catch (Exception e)
                {
                    // MetaData field doesn't exist in the class!
                    throw new InvalidMetaDataException(LOCALISER,
                        "044071", embeddedClass.getName(),fieldFmd.getFullFieldName());
                }
                fieldFmd.populate(clr, null, cls_method, primary);
            }
        }
    }

    /**
     * Method to initialise the object, creating all internal convenience
     * arrays.
     */
    public void initialise()
    {
        fieldMetaData = new AbstractMemberMetaData[members.size()];
        for (int i=0; i<fieldMetaData.length; i++)
        {
            fieldMetaData[i] = (AbstractMemberMetaData) members.get(i);
            fieldMetaData[i].initialise();
        }

        setInitialised();
    }

    // ----------------------------- Accessors ---------------------------------

    /**
     * Accessor for fieldMetaData
     * @return Returns the fieldMetaData.
     */
    public final AbstractMemberMetaData[] getFieldMetaData()
    {
        return fieldMetaData;
    }
    
    /**
     * Accessor for the owner field/property to contain the FK back to the owner
     * @return The owner field/property
     */
    public final String getOwnerMember()
    {
        return ownerMember;
    }

    /**
     * Accessor for the column to check for null values.
     * @return The column name
     */
    public final String getNullIndicatorColumn()
    {
        return nullIndicatorColumn;
    }

    /**
     * Accessor for the value of the null indicator column when null
     * @return The value
     */
    public final String getNullIndicatorValue()
    {
        return nullIndicatorValue;
    }

    // ----------------------------- Mutators ----------------------------------

    /**
     * Method to add a field/property to the embedded definition.
     * Rejects the addition of duplicate named fields/properties.
     * @param mmd Meta-Data for the field/property.
     */
    public void addMember(AbstractMemberMetaData mmd)
    {
        if (mmd == null)
        {
            return;
        }

        if (isInitialised())
        {
            throw new RuntimeException(LOCALISER.msg("044108", 
                mmd.getName(), mmd.getAbstractClassMetaData().getFullClassName()));
        }
        Iterator iter=members.iterator();
        while (iter.hasNext())
        {
            AbstractMemberMetaData md = (AbstractMemberMetaData)iter.next();
            if (mmd.getName().equals(md.getName()))
            {
                throw new RuntimeException(LOCALISER.msg("044112", 
                    mmd.getName(), mmd.getAbstractClassMetaData().getFullClassName()));
            }
        }
        members.add(mmd);
    }

    // ------------------------------- Utilities -------------------------------

    /**
     * Returns a string representation of the object using a prefix
     * This can be used as part of a facility to output a MetaData file. 
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        // Field needs outputting so generate metadata
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<embedded");
        if (ownerMember != null)
        {
            sb.append(" owner-field=\"" + ownerMember + "\"");
        }
        if (nullIndicatorColumn != null)
        {
            sb.append(" null-indicator-column=\"" + nullIndicatorColumn + "\"");
        }
        if (nullIndicatorValue != null)
        {
            sb.append(" null-indicator-value=\"" + nullIndicatorValue + "\"");
        }
        sb.append(">\n");

        // Add fields
        for (int i=0; i<members.size(); i++)
        {
            AbstractMemberMetaData f = (AbstractMemberMetaData)members.get(i);
            sb.append(f.toString(prefix + indent,indent));
        }
        
        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix + "</embedded>\n");
        return sb.toString();
    }
}