/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.util.StringUtils;

/**
 * FetchPlan defined in MetaData.
 * 
 * @since 1.1
 * @version $Revision: 1.12 $
 */
public class FetchPlanMetaData extends MetaData
{
    /** Name of the FetchPlan. */
    final String name;

    /** Max fetch depth for this FetchPlan. */
    protected int maxFetchDepth = -1;

    /** Fetch Size for use when querying using this FetchPlan. */
    protected int fetchSize = -1;

    /** Series of Fetch Groups used in this FetchPlan. Only used during construction. */
    protected List fetchGroups = new ArrayList();    

    /**
     * Constructor.
     * @param parent The parent MetaData
     * @param name Name of fetch plan
     * @param depth Max fetch depth
     * @param size Fetch size for this fetch plan
     */
    public FetchPlanMetaData(MetaData parent, String name, String depth, String size)
    {
        super(parent);

        this.name = name;
        if (!StringUtils.isWhitespace(depth))
        {
            try
            {
                this.maxFetchDepth = (new Integer(depth)).intValue();
            }
            catch (NumberFormatException nfe)
            {
                this.maxFetchDepth = -1;
            }
        }
        if (!StringUtils.isWhitespace(size))
        {
            try
            {
                this.fetchSize = (new Integer(size)).intValue();
            }
            catch (NumberFormatException nfe)
            {
                this.fetchSize = -1;
            }
        }
    }

    /**
     * Accessor for name
     * @return Returns the name.
     */
    public final String getName()
    {
        return name;
    }

    /**
     * Accessor for the maximum fetch depth.
     * @return Returns the max fetch depth
     */
    public final int getMaxFetchDepth()
    {
        return maxFetchDepth;
    }

    /**
     * Accessor for the fetch size.
     * @return Returns the fetch size
     */
    public final int getFetchSize()
    {
        return fetchSize;
    }
    
    /**
     * Accessor for fetchGroupMetaData
     * @return Returns the fetchGroupMetaData.
     */
    public final FetchGroupMetaData[] getFetchGroupMetaData()
    {
        return (FetchGroupMetaData[])fetchGroups.toArray(new FetchGroupMetaData[fetchGroups.size()]);
    }

    /**
     * Add a new FetchGroupMetaData
     * @param fgmd the fetch group
     */
    public void addFetchGroup(FetchGroupMetaData fgmd)
    {
        fetchGroups.add(fgmd);
    }

    // ----------------------------- Utilities ------------------------------------

    /**
     * Returns a string representation of the object.
     * This can be used as part of a facility to output a MetaData file. 
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix, String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<fetch-plan name=\"" + name + "\"" + 
            " max-fetch-depth=\"" + maxFetchDepth + "\"" +
            " fetch-size=\"" + fetchSize + "\"\n");

        // Add fetch-groups
        Iterator iter = fetchGroups.iterator();
        while (iter.hasNext())
        {
            FetchGroupMetaData fgmd = (FetchGroupMetaData)iter.next();
            sb.append(fgmd.toString(prefix + indent, indent));
        }

        sb.append(prefix + "</fetch-plan>\n");
        return sb.toString();
    }
}