/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import org.datanucleus.util.StringUtils;

/**
 * Representation of the Meta-Data for a field of a class.
 *
 * <H3>Lifecycle state</H3>
 * An object of this type has 2 lifecycle states. The first is the raw
 * constructed object which represents pure MetaData (maybe from a MetaData
 * file). The second is a "populated" object which represents MetaData for a
 * Field of a class with the metadata aligned to be appropriate for that Field. 
 *
 * <H3>Containers</H3>
 * Each field can represent a container. The container can be an array, a
 * Collection or a Map. The field type must be of the correct type to represent
 * these.
 *
 * <H3>JPOX Management</H3>
 * Each field can be managed by JPOX or not. The class provides a method for
 * identifying if a field is managed by JPOX (<I>isJdoField()</I>). If a field
 * is managed by JPOX, it will have a field "id" (within its class). In a class
 * the field "id" will start at 0 (for the first field, in alphabetical order).
 *
 * <H3>MetaData Element</H3> 
 * The MetaData element represented here is as follows
 * <PRE> 
 * &lt;!ELEMENT field (extension*, (collection|map|array|(column*))?, join?, element?, 
 *      key?, value?, fetch-group*, order?, embedded?, index?, unique?, foreign-key?,
 *      delete-action?, extension*)?&gt;
 * &lt;!ATTLIST field name CDATA #REQUIRED&gt;
 * &lt;!ATTLIST field persistence-modifier (persistent|transactional|none)
 *      #IMPLIED&gt;
 * &lt;!ATTLIST field table CDATA #IMPLIED&gt;
 * &lt;!ATTLIST field null-value (exception|default|none) 'none'&gt;
 * &lt;!ATTLIST field default-fetch-group (true|false) #IMPLIED&gt;
 * &lt;!ATTLIST field embedded (true|false) #IMPLIED&gt;
 * &lt;!ATTLIST field serialized (true|false) #IMPLIED&gt;
 * &lt;!ATTLIST field dependent (true|false) #IMPLIED&gt;
 * &lt;!ATTLIST field indexed (true|false|unique) #IMPLIED&gt;
 * &lt;!ATTLIST field unique (true|false) #IMPLIED&gt;
 * &lt;!ATTLIST field load-fetch-group CDATA #IMPLIED&gt;
 * &lt;!ATTLIST field recursion-depth CDATA #IMPLIED&gt;
 * &lt;!ATTLIST field primary-key (true|false) 'false'&gt;
 * &lt;!ATTLIST field mapped-by CDATA #IMPLIED&gt;
 * &lt;!ATTLIST field value-strategy CDATA #IMPLIED&gt;
 * &lt;!ATTLIST field delete-action (restrict|cascade|null|default|none) #IMPLIED&gt;
 * &lt;!ATTLIST field sequence CDATA #IMPLIED&gt;
 * &lt;!ATTLIST field field-type CDATA #IMPLIED&gt;
 * </PRE>
 *
 * @since 1.1 
 * @version $Revision: 1.83 $
 */
public class FieldMetaData extends AbstractMemberMetaData
{
    /**
     * Convenience constructor taking defaults
     * @param parent Parent component
     * @param name Name of the field
     */
    public FieldMetaData(MetaData parent, final String name)
    {
        super(parent, name);
    }

    /**
     * Convenience constructor to copy the specification from the passed field.
     * This is used when we have an overriding field and we make a copy of the baseline
     * field as a starting point.
     * @param parent The parent
     * @param fmd The field to copy
     */
    public FieldMetaData(MetaData parent, AbstractMemberMetaData fmd)
    {
        super(parent, fmd);
    }

    /**
     * Constructor. Saves the MetaData with the specified values. The object is
     * then in an "unpopulated" state. It can become "populated" by calling the
     * <B>populate()</B> method which compares it against the field it is to
     * represent and updates any unset attributes and flags up any errors.
     * @param parent parent MetaData instance
     * @param name field name 
     * @param pk attribute primary-key value
     * @param modifier attribute persistence-modifier value
     * @param defaultFetchGroup attribute default-fetch-group value
     * @param nullValue attribute null-value value 
     * @param embedded attribute embedded value
     * @param serialized attribute serialized value
     * @param dependent attribute dependent value
     * @param mappedBy attribute mapped-by value
     * @param column attribute column value
     * @param table attribute table value
     * @param catalog attribute catalog value
     * @param schema attribute schema value
     * @param deleteAction attribute delete-action value
     * @param indexed Whether this is indexed
     * @param unique Apply a unique constraint
     * @param recursionDepth The depth of fetch to use when recursing
     * @param loadFetchGroup Name of the additional fetch group to use when loading
     * @param valueStrategy attribute value-strategy value
     * @param sequence attribute sequence value
     * @param fieldType Implementation type(s) for field.
     */
    public FieldMetaData(MetaData parent,
                         final String name,
                         final String pk,
                         final String modifier,
                         final String defaultFetchGroup,
                         final String nullValue,
                         final String embedded,
                         final String serialized,
                         final String dependent,
                         final String mappedBy,
                         final String column,
                         final String table,
                         final String catalog,
                         final String schema,
                         final String deleteAction,
                         final String indexed,
                         final String unique,
                         final String recursionDepth,
                         final String loadFetchGroup,
                         final String valueStrategy,
                         final String sequence,
                         final String fieldType)
    {
        super(parent, name, pk, modifier, defaultFetchGroup, nullValue, embedded, serialized, dependent, mappedBy, 
            column, table, catalog, schema,
            deleteAction, indexed, unique, recursionDepth, loadFetchGroup, valueStrategy, sequence, fieldType);
    }

    /**
     * Whether this uses getter/setter accessors (Property) or
     * used field based access (Field)
     * @return true if this is a property
     */
    public boolean isProperty()
    {
        return false;
    }

    /**
     * Returns a string representation of the object using a prefix
     * This can be used as part of a facility to output a MetaData file. 
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        // If this field is static or final, don't bother with MetaData since JPOX will ignore it anyway.
        if (isStatic() || isFinal())
        {
            return "";
        }

        // Field needs outputting so generate metadata
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<field name=\"" + name + "\"");
        if (persistenceModifier != null && !StringUtils.isWhitespace(persistenceModifier.toString()))
        {
            sb.append("\n").append(prefix).append("       persistence-modifier=\"" + persistenceModifier + "\"");
        }
        if (!StringUtils.isWhitespace(table))
        {
            sb.append("\n").append(prefix).append("       table=\"" + table + "\"");
        }
        if (primaryKey != null && primaryKey.booleanValue())
        {
            sb.append("\n").append(prefix).append("       primary-key=\"" + primaryKey + "\"");
        }
        sb.append("\n").append(prefix).append("       null-value=\"" + nullValue + "\"");
        if (defaultFetchGroup != null && !StringUtils.isWhitespace(defaultFetchGroup.toString()))
        {
            sb.append("\n").append(prefix).append("       default-fetch-group=\"" + defaultFetchGroup + "\"");
        }
        if (embedded != null && !StringUtils.isWhitespace(embedded.toString()))
        {
            sb.append("\n").append(prefix).append("       embedded=\"" + embedded + "\"");
        }
        if (serialized != null && !StringUtils.isWhitespace(serialized.toString()))
        {
            sb.append("\n").append(prefix).append("       serialized=\"" + serialized + "\"");
        }
        if (dependent != null)
        {
            sb.append("\n").append(prefix).append("       dependent=\"" + dependent + "\"");
        }
        if (mappedBy != null)
        {
            sb.append("\n").append(prefix).append("       mapped-by=\"" + mappedBy + "\"");
        }
        if (fieldTypes != null)
        {
            sb.append("\n").append(prefix).append("       field-type=\"");
            for (int i=0;i<fieldTypes.length;i++)
            {
                sb.append(fieldTypes[i]);
            }
            sb.append("\"");
        }
        if (loadFetchGroup != null)
        {
            sb.append("\n").append(prefix).append("       load-fetch-group=\"" + loadFetchGroup + "\"");
        }
        if (recursionDepth != DEFAULT_RECURSION_DEPTH && recursionDepth != UNDEFINED_RECURSION_DEPTH)
        {
            sb.append("\n").append(prefix).append("       recursion-depth=\"" + recursionDepth + "\"");
        }
        if (valueStrategy != null)
        {
            sb.append("\n").append(prefix).append("       value-strategy=\"" + valueStrategy + "\"");
        }
        if (sequence != null)
        {
            sb.append("\n").append(prefix).append("       sequence=\"" + sequence + "\"");
        }
        sb.append(">\n");

        // Add field containers
        if (container != null)
        {
            if (container instanceof CollectionMetaData)
            {
                CollectionMetaData c = (CollectionMetaData)container;
                sb.append(c.toString(prefix + indent,indent));
            }
            else if (container instanceof ArrayMetaData)
            {
                ArrayMetaData c = (ArrayMetaData)container;
                sb.append(c.toString(prefix + indent,indent));
            }
            else if (container instanceof MapMetaData)
            {
                MapMetaData c = (MapMetaData)container;
                sb.append(c.toString(prefix + indent,indent));
            }
        }

        // Add columns
        if (columnMetaData != null)
        {
            for (int i=0; i<columnMetaData.length; i++)
            {
                sb.append(columnMetaData[i].toString(prefix + indent,indent));
            }
        }

        // Add join
        if (joinMetaData != null)
        {
            sb.append(joinMetaData.toString(prefix + indent,indent));
        }

        // Add element
        if (elementMetaData != null)
        {
            sb.append(elementMetaData.toString(prefix + indent,indent));
        }

        // Add key
        if (keyMetaData != null)
        {
            sb.append(keyMetaData.toString(prefix + indent,indent));
        }

        // Add value
        if (valueMetaData != null)
        {
            sb.append(valueMetaData.toString(prefix + indent,indent));
        }

        // TODO Add fetch-groups

        // Add order
        if (orderMetaData != null)
        {
            sb.append(orderMetaData.toString(prefix + indent,indent));
        }

        // Add embedded
        if (embeddedMetaData != null)
        {
            sb.append(embeddedMetaData.toString(prefix + indent,indent));
        }

        // Add index
        if (indexMetaData != null)
        {
            sb.append(indexMetaData.toString(prefix + indent,indent));
        }

        // Add unique
        if (uniqueMetaData != null)
        {
            sb.append(uniqueMetaData.toString(prefix + indent,indent));
        }

        // Add foreign-key
        if (foreignKeyMetaData != null)
        {
            sb.append(foreignKeyMetaData.toString(prefix + indent,indent));
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</field>\n");
        return sb.toString();
    }

    /**
     * Comparator method. This allows the ClassMetaData to search for a
     * FieldMetaData with a particular name.
     * @param o The object to compare against
     * @return The comparison result
     */ 
    public int compareTo(Object o)
    {
        if (o instanceof AbstractMemberMetaData)
        {
            AbstractMemberMetaData c = (AbstractMemberMetaData)o;
            return this.name.compareTo(c.name);
        }        
        else if (o instanceof String)
        {
            return this.name.compareTo((String)o);
        }
        else if (o == null)
        {
            throw new ClassCastException("object is null");
        }
        throw new ClassCastException(this.getClass().getName() + " != " + o.getClass().getName());
    }
}