/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

/**
 * Representation of a Meta-Data file. 
 * Contains a list of package meta-data, together with any named queries and fetch plans etc.
 *
 * @since 1.1 
 * @version $Revision: 1.24 $
 */
public class FileMetaData extends MetaData
{
    /** JDO file **/
    public static final int JDO_FILE = 1;
    /** ORM file **/
    public static final int ORM_FILE = 2;
    /** JDOQUERY file **/
    public static final int JDOQUERY_FILE = 3;
    /** annotations **/
    public static final int ANNOTATIONS = 4;
    /** JPA file **/
    public static final int JPA_FILE = 5;

    /** Manager for this MetaData object. */
    transient protected final MetaDataManager metaDataManager;

    /** Type of file (JDO, ORM, JDOQUERY, etc) */
    protected int type;

    /** Name of file */
    protected String filename;

    /** Catalog name for all classes in this file */
    protected String catalog;

    /** Schema name for all classes in this file */
    protected String schema;

    /** Named queries defined in this file. */
    protected Collection queries = null;

    /** List of query result MetaData defined for this file. */
    protected Collection queryResultMetaData = null;

    /** Named FetchPlans in this file. */
    protected Collection fetchPlans = null;

    /** List of packages in this file (uses List to retain file positioning) */
    protected List packages = null;

    /** List of event listeners defined for this file. */
    protected List listeners = null;

    /**
     * Constructor.
     * @param filename The file where this is stored (or null).
     * @param metaDataManager the manager for this MetaData object.
     * @param catalog Name of the catalog for all classes in this file
     * @param schema Name of the schema for all classes in this file
     */
    public FileMetaData(String filename, MetaDataManager metaDataManager, String catalog, String schema)
    {
        super(null);

        this.filename = filename;
        this.metaDataManager = metaDataManager;
    }

    // ------------------------------ Accessors --------------------------------

    /**
     * Accessor for the file type (JDO, ORM, etc)
     * @return The file type
     */
    public int getType()
    {
        return type;
    }

    /**
     * Accessor for the number of named queries.
     * @return no of named queries
     */
    public int getNoOfQueries()
    {
        return queries != null ? queries.size() : 0;
    }

    /**
     * Accessor for the metadata of the named queries.
     * @return Meta-Data for the named queries.
     */
    public QueryMetaData[] getQueries()
    {
        return (queries == null ? null : 
            (QueryMetaData[])queries.toArray(new QueryMetaData[queries.size()]));
    }

    /**
     * Accessor for the number of named fetch plans.
     * @return no of named fetch plans
     */
    public int getNoOfFetchPlans()
    {
        return fetchPlans != null ? fetchPlans.size() : 0;
    }

    /**
     * Accessor for the metadata of the named fetch plans.
     * @return Meta-Data for the named fetch plans.
     */
    public FetchPlanMetaData[] getFetchPlans()
    {
        return (fetchPlans == null ? null : 
            (FetchPlanMetaData[])fetchPlans.toArray(new FetchPlanMetaData[fetchPlans.size()]));
    }

    /**
     * Accessor for the MetaDataManager for this file.
     * @return The MetaDataManager for this file.
     */
    public MetaDataManager getMetaDataManager()
    {
        return metaDataManager;
    }

    /**
     * Accessor for the filename
     * @return The filename of this MetaData file.
     */
    public String getFilename()
    {
        return filename;
    }

    /**
     * Accessor for the catalog name for all classes in this file
     * @return Name of the catalog to use.
     */
    public String getCatalog()
    {
        return catalog;
    }

    /**
     * Accessor for the schema name for all classes in this file
     * @return Name of the schema to use
     */
    public String getSchema()
    {
        return schema;
    }

    /**
     * Accessor for the number of packages.
     * @return no of packages.
     */
    public int getNoOfPackages()
    {
        return packages != null ? packages.size() : 0;
    }

    /**
     * Accessor for the meta-data of a package.
     * @param i index number
     * @return Meta-Data for a package.
     */
    public PackageMetaData getPackage(int i)
    {
        if (packages == null)
        {
            return null;
        }

        return (PackageMetaData)packages.get(i);
    }

    /**
     * Accessor for the Meta-Data of a package with a given name.
     * @param name Name of the package
     * @return Meta-Data for the package
     */
    public PackageMetaData getPackage(String name)
    {
        if (packages == null)
        {
            return null;
        }

        Iterator iter = packages.iterator();
        while (iter.hasNext())
        {
            PackageMetaData p = (PackageMetaData)iter.next();
            if (p.name.equals(name))
            {
                return p;
            }
        }
        return null;
    }

    /**
     * Utility method to check if the MetaData for a class is contained in this
     * file.
     * @param pkg_name Name of package
     * @param class_name Name of class
     * @return The MetaData for the class
     */
    public ClassMetaData getClass(String pkg_name, String class_name)
    {
        if (pkg_name == null || class_name == null)
        {
            return null;
        }

        PackageMetaData pmd=getPackage(pkg_name);
        if (pmd != null)
        {
            return pmd.getClass(class_name);
        }

        return null;
    }

    // -------------------------------- Mutators -------------------------------

    /**
     * Mutator for the file type.
     * @param type The file type
     */
    public void setType(int type)
    {
        this.type = type;
    }

    /**
     * Method to add a named query to this class. Rejects the addition of
     * duplicate named queries.
     * @param qmd Meta-Data for the query.
     */
    public void addQuery(QueryMetaData qmd)
    {
        if (qmd == null)
        {
            return;
        }
        if (queries == null)
        {
            queries = new HashSet();
        }

        queries.add(qmd);
    }

    /**
     * Method to add a named fetch plan to this class. 
     * Rejects the addition of duplicate named fetch plans.
     * @param fpmd Meta-Data for the fetch plan.
     */
    public void addFetchPlan(FetchPlanMetaData fpmd)
    {
        if (fpmd == null)
        {
            return;
        }
        if (fetchPlans == null)
        {
            fetchPlans = new HashSet();
        }

        fetchPlans.add(fpmd);
    }

    /**
     * Mutator for the filename for this MetaData file.
     * @param filename The filename of this MetaData file.
     */
    public void setFilename(String filename)
    {
        this.filename = filename;
    }

    /**
     * Mutator for the catalog for all classes in this file
     * @param catalog Catalog name to use
     */
    public void setCatalog(String catalog)
    {
        this.catalog = catalog;
    }

    /**
     * Mutator for the schema for all classes in this file
     * @param schema Schema name to use
     */
    public void setSchema(String schema)
    {
        this.schema = schema;
    }

    /**
     * Method to add a package
     * @param pkg The PackageMetaData to add.
     **/
    public void addPackage(PackageMetaData pkg)
    {
        if (pkg == null)
        {
            return;
        }
        if (packages == null)
        {
            packages = new ArrayList();
        }
        else
        {
            // Check if already exists
            Iterator iter = packages.iterator();
            while (iter.hasNext())
            {
                PackageMetaData p=(PackageMetaData)iter.next();
                if (pkg.getName().equals(p.getName()))
                {
                    return;
                }
            }
        }
        packages.add(pkg);
    }

    // -------------------------------- Utilities ------------------------------

    /**
     * Returns a string representation of the object.
     * @param prefix Any prefix for the output
     * @param indent The indent
     * @return a string representation of the object.
     */
    public String toString(String prefix, String indent)
    {
        if (indent == null)
        {
            indent = "";
        }

        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<jdo");
        if (catalog != null)
        {
            sb.append(" catalog=\"" + catalog + "\"");
        }
        if (schema != null)
        {
            sb.append(" schema=\"" + schema + "\"");
        }
        sb.append(">\n");

        // Add packages
        if (packages != null)
        {
            Iterator iter = packages.iterator();
            while (iter.hasNext())
            {
                sb.append(((PackageMetaData)iter.next()).toString(indent,indent));
            }
        }

        // Add queries
        if (queries != null)
        {
            Iterator iter = queries.iterator();
            while (iter.hasNext())
            {
                sb.append(((QueryMetaData)iter.next()).toString(indent, indent));
            }
        }

        // Add fetch plans
        if (fetchPlans != null)
        {
            Iterator iter = fetchPlans.iterator();
            while (iter.hasNext())
            {
                sb.append(((FetchPlanMetaData)iter.next()).toString(indent, indent));
            }
        }

        // Add extensions
        sb.append(super.toString(indent,indent)); 

        sb.append("</jdo>");
        return sb.toString();
    }

    /**
     * Add a listener class name
     * @param listener the listener metadata. Duplicated classes are ignored
     */
    public void addListener(EventListenerMetaData listener)
    {
        if (listeners == null)
        {
            listeners = new ArrayList();
        }
        if (!listeners.contains(listener))
        {
            listeners.add(listener);
        }
    }

    /**
     * Get the event listeners registered against the file.
     * @return the event listeners
     */
    public List getListeners()
    {
        return listeners;
    }

    /**
     * Method to register a query-result MetaData.
     * @param resultMetaData Query-Result MetaData to register
     */
    public void addQueryResultMetaData(QueryResultMetaData resultMetaData)
    {
        if (queryResultMetaData == null)
        {
            queryResultMetaData = new HashSet();
        }
        if (!queryResultMetaData.contains(resultMetaData))
        {
            queryResultMetaData.add(resultMetaData);
        }
    }

    /**
     * Get the query result MetaData.
     * @return Query Result MetaData
     */
    public QueryResultMetaData[] getQueryResultMetaData()
    {
        if (queryResultMetaData == null)
        {
            return null;
        }
        return (QueryResultMetaData[])queryResultMetaData.toArray(new QueryResultMetaData[queryResultMetaData.size()]);
    }
}