/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2004 Andy Jefferson- added toString(), "column", MetaData, javadocs
    ...
**********************************************************************/
package org.datanucleus.metadata;

/**
 * Foreign keys in metadata serve two quite different purposes. First, when
 * generating schema, the foreign key element identifies foreign keys to be
 * generated. Second, when using the database, foreign key elements identify
 * foreign keys that are assumed to exist in the database. This is important for
 * the runtime to properly order insert, update, and delete statements to avoid
 * constraint violations. A foreign-key element can be contained by a field,
 * element, key, value, or join element, if all of the columns mapped are to be
 * part of the same foreign key. A foreign-key element can be contained within a
 * class element. In this case, the column elements are mapped elsewhere, and
 * the column elements contained in the foreign-key element have only the column
 * name.
 * 
 * @since 1.1
 * @version $Revision: 1.21 $
 */
public class ForeignKeyMetaData extends AbstractConstraintMetaData implements ColumnMetaDataContainer
{
    /**
     * The unique attribute specifies whether the foreign key constraint is
     * defined to be a unique constraint as well. This is most often used with
     * one-to-one mappings.
     */
    protected boolean unique;

    /**
     * The deferred attribute specifies whether the foreign key constraint is
     * defined to be checked only at commit time.
     */
    protected boolean deferred;

    /**
     * Foreign keys represent a consistency constraint in the database that must
     * be maintained. The user can specify by the value of the delete-action
     * attribute what happens if the target row of a foreign key is deleted.
     */
    protected ForeignKeyAction deleteAction;

    /**
     * Foreign keys represent a consistency constraint in the database that must
     * be maintained. The user can specify by the update-action attribute what
     * happens if the target row of a foreign key is updated.
     */
    protected ForeignKeyAction updateAction;

    /**
     * Constructor to create a copy of the passed metadata using the provided parent.
     * @param fkmd The metadata to copy
     */
    public ForeignKeyMetaData(ForeignKeyMetaData fkmd)
    {
        super(null, fkmd.name, fkmd.table); // Ignore parent

        this.deferred = fkmd.deferred;
        this.deleteAction = fkmd.deleteAction;
        this.updateAction = fkmd.updateAction;
        this.unique = fkmd.unique;

        if (fkmd.isInitialised())
        {
            if (fkmd.getMemberMetaData() != null)
            {
                for (int i=0;i<fkmd.getMemberMetaData().length;i++)
                {
                    if (fkmd.getMemberMetaData()[i] instanceof PropertyMetaData)
                    {
                        addMember(new PropertyMetaData(this,(PropertyMetaData)fkmd.getMemberMetaData()[i]));
                    }
                    else
                    {
                        addMember(new FieldMetaData(this,fkmd.getMemberMetaData()[i]));
                    }
                }
            }
            if (fkmd.getColumnMetaData() != null)
            {
                for (int i=0;i<fkmd.getColumnMetaData().length;i++)
                {
                    addColumn(new ColumnMetaData(this,fkmd.getColumnMetaData()[i]));
                }
            }
        }
        else
        {
            for (int i=0;i<fkmd.members.size();i++)
            {
                if (fkmd.members.get(i) instanceof PropertyMetaData)
                {
                    addMember(new PropertyMetaData(this,(PropertyMetaData)fkmd.members.get(i)));
                }
                else
                {
                    addMember(new FieldMetaData(this,(AbstractMemberMetaData) fkmd.members.get(i)));
                }
            }
            for (int i=0;i<fkmd.columns.size();i++)
            {
                addColumn(new ColumnMetaData(this,(ColumnMetaData)fkmd.columns.get(i)));
            }
        }
    }

    /**
     * Constructor. 
     * @param name Name of the foreign key
     * @param table Table to apply the FK to
     * @param unique Whether the key is unique
     * @param deferred Whether the key is deferred
     * @param delete_action Action to perform on deletion
     * @param update_action Action to perform on update
     */
    public ForeignKeyMetaData(final String name,
                              final String table,
                              final String unique,
                              final String deferred,
                              final String delete_action,
                              final String update_action)
    {
        super(null, name, table);    // Ignore parent

        if (unique != null)
        {
            if (unique.equalsIgnoreCase("true"))
            {
                this.unique = true;
            }
            else if (unique.equalsIgnoreCase("false"))
            {
                this.unique = false;
            }
        }
        if (deferred != null)
        {
            if (deferred.equalsIgnoreCase("true"))
            {
                this.deferred = true;
            }
            else if (deferred.equalsIgnoreCase("false"))
            {
                this.deferred = false;
            }
        }

        this.deleteAction = ForeignKeyAction.getForeignKeyAction(delete_action);
        this.updateAction = ForeignKeyAction.getForeignKeyAction(update_action);
    }

    /**
     * Method to initialise the object, setting up all internal arrays.
     * Initialises all sub-objects.
     */
    public void initialise()
    {
        if (isInitialised())
        {
            return;
        }

        // Set up the columnMetaData
        if (members.size() == 0)
        {
            memberMetaData = null;
        }
        else
        {
            memberMetaData = new AbstractMemberMetaData[members.size()];
            for (int i=0; i<memberMetaData.length; i++)
            {
                memberMetaData[i] = (AbstractMemberMetaData) members.get(i);
                memberMetaData[i].initialise();
            }
        }

        // Set up the columnMetaData
        if (columns.size() == 0)
        {
            columnMetaData = null;
        }
        else
        {
            columnMetaData = new ColumnMetaData[columns.size()];
            for (int i=0; i<columnMetaData.length; i++)
            {
                columnMetaData[i] = (ColumnMetaData) columns.get(i);
                columnMetaData[i].initialise();
            }
        }

        // Clear out parsing data
        columns.clear();
        columns = null;
        members.clear();
        members = null;

        setInitialised();
    }

    // ---------------------------- Mutators -----------------------------------

    /**
     * Accessor for whether the FK is deferred
     * @return Returns the deferred.
     */
    public final boolean isDeferred()
    {
        return deferred;
    }

    /**
     * Accessor for the delete action
     * @return Returns the deleteAction.
     */
    public final ForeignKeyAction getDeleteAction()
    {
        return deleteAction;
    }

    /**
     * Accessor for the unique flag 
     * @return Returns the unique.
     */
    public final boolean isUnique()
    {
        return unique;
    }

    /**
     * Accessor for the update action
     * @return Returns the updateAction.
     */
    public final ForeignKeyAction getUpdateAction()
    {
        return updateAction;
    }

    // ---------------------------- Utilities ----------------------------------

    /**
     * Returns a string representation of the object using a prefix
     * This can be used as part of a facility to output a MetaData file. 
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        // Field needs outputting so generate metadata
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<foreign-key deferred=\"" + deferred + "\"\n");
        sb.append(prefix).append("       unique=\"" + unique + "\"");
        if (updateAction != null)
        {
            sb.append("\n").append(prefix).append("       update-action=\"" + updateAction + "\"");
        }
        if (deleteAction != null)
        {
            sb.append("\n").append(prefix).append("       delete-action=\"" + deleteAction + "\"");
        }
        if (table != null)
        {
            sb.append("\n").append(prefix).append("       table=\"" + table + "\"");
        }
        if (name != null)
        {
            sb.append("\n").append(prefix).append("       name=\"" + name + "\"");
        }
        sb.append(">\n");

        // Add fields
        if (memberMetaData != null)
        {
            for (int i=0;i<memberMetaData.length;i++)
            {
                sb.append(memberMetaData[i].toString(prefix + indent,indent));
            }
        }

        // Add columns
        if (columnMetaData != null)
        {
            for (int i=0;i<columnMetaData.length;i++)
            {
                sb.append(columnMetaData[i].toString(prefix + indent,indent));
            }
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</foreign-key>\n");
        return sb.toString();
    }
}