/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2004 Andy Jefferson - addition of table name
2005 Andy Jefferson - added primary-key, unique, and others
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.List;

import org.datanucleus.util.StringUtils;

/**
 * Secondary tables and join tables are mapped using a join condition that
 * associates a column or columns in the secondary or join table with a column
 * or columns in the primary table, typically the primary tables primary key
 * columns. Column elements used for relationship mapping or join conditions
 * specify the column name and optionally the target column name. The target
 * column name is the name of the column in the associated table corresponding
 * to the named column. The target column name is optional when the target
 * column is the single primary key column of the associated table.
 *
 * @since 1.1
 * @version $Revision: 1.24 $
 */
public class JoinMetaData extends MetaData implements ColumnMetaDataContainer
{
    /** the foreign-key element. */
    ForeignKeyMetaData foreignKeyMetaData;

    /** the index element. */
    IndexMetaData indexMetaData;

    /** the unique element. */
    UniqueMetaData uniqueMetaData;

    /** PrimaryKey MetaData */
    protected PrimaryKeyMetaData primaryKeyMetaData;

    /** column elements */
    final List columns = new ArrayList();
       
    /** if is outer join. Outer joins return all elements from at least one of the sides joined. */
    Boolean outer;

    /** the table name. */
    String table;

    /** the catalog name. */
    String catalog;

    /** the schema name. */
    String schema;

    /** the column name. */
    String columnName;

    /** the foreign-key column name. */
    String foreignKey;

    /** The indexing value */
    protected IndexedValue indexed=null;

    /** Whether to add a unique constraint. */
    protected final boolean uniqueConstraint;

    // -------------------------------------------------------------------------
    // Fields below here are not represented in the output MetaData. They are
    // for use internally in the operation of the JDO system. The majority are
    // for convenience to save iterating through the fields since the fields
    // are fixed once initialised.

    /**
     * Contains the metadata for column
     */
    protected ColumnMetaData[] columnMetaData;

    /**
     * Constructor to create a copy of the passed JoinMetaData but with the passed parent.
     * @param parent Parent
     * @param joinmd Existing join metadata to copy
     */
    public JoinMetaData(MetaData parent, JoinMetaData joinmd)
    {
        super(parent);
        this.table = joinmd.table;
        this.catalog = joinmd.catalog;
        this.schema = joinmd.schema;
        this.columnName = joinmd.columnName;
        this.outer = joinmd.outer;
        this.indexed = joinmd.indexed;
        this.uniqueConstraint = joinmd.uniqueConstraint;
        for (int i=0;i<joinmd.columns.size();i++)
        {
            addColumn(new ColumnMetaData(this,(ColumnMetaData)joinmd.columns.get(i)));
        }
    }

    /**
     * Constructor for the Join.
     * @param parent The parent
     * @param table The table name
     * @param catalog The catalog name
     * @param schema The schema name
     * @param columnName The column name
     * @param outer Whether it is an outer join
     * @param fkDeleteAction Delete action on the FK
     * @param indexed whether the join is indexed
     * @param unique whether the join is unique
     */
    public JoinMetaData(final MetaData parent,
                        final String table,
                        final String catalog,
                        final String schema,
                        final String columnName,
                        final String outer,
                        final String fkDeleteAction,
                        String indexed,
                        String unique)
    {
        super(parent);

        if (table != null && parent instanceof AbstractMemberMetaData)
        {
            // "table" has been specified but this join is within <field> or <property> so is not applicable
            //TODO fix message for property and field
            throw new InvalidMetaDataException(LOCALISER, "044130",
                ((AbstractMemberMetaData)parent).getFullFieldName());
        }

        this.table = (StringUtils.isWhitespace(table) ? null : table);
        this.catalog = (StringUtils.isWhitespace(catalog) ? null : catalog);
        this.schema = (StringUtils.isWhitespace(schema) ? null : schema);
        this.columnName = (StringUtils.isWhitespace(columnName) ? null : columnName);

        if (outer != null)
        {
            if (outer.equalsIgnoreCase("true"))
            {
                this.outer = Boolean.TRUE;
            }
            else if (outer.equalsIgnoreCase("false"))
            {
                this.outer = Boolean.FALSE;
            }
        }

        if (!StringUtils.isWhitespace(fkDeleteAction))
        {
            this.foreignKeyMetaData = new ForeignKeyMetaData(null, null, null, null, fkDeleteAction, null);
        }

        this.indexed = IndexedValue.getIndexedValue(indexed);

        if (unique != null && unique.equalsIgnoreCase("true"))
        {
            uniqueConstraint = true;
        }
        else
        {
            uniqueConstraint = false;
        }
    }

    /**
     * Method to initialise the object, creating internal convenience arrays.
     * Initialises all sub-objects.
     */
    public void initialise()
    {
        // Cater for user specifying column name, or columns
        if (columns.size() == 0 && columnName != null)
        {
            columnMetaData = new ColumnMetaData[1];
            columnMetaData[0] = new ColumnMetaData(this, columnName);
            columnMetaData[0].initialise();
        }
        else
        {
            columnMetaData = new ColumnMetaData[columns.size()];
            for (int i=0; i<columnMetaData.length; i++)
            {
                columnMetaData[i] = (ColumnMetaData) columns.get(i);
                columnMetaData[i].initialise();
            }
        }

        if (foreignKeyMetaData != null)
        {
            foreignKeyMetaData.initialise();
        }

        // Interpret the "indexed" value to create our IndexMetaData where it wasn't specified that way
        if (indexMetaData == null && columnMetaData != null && indexed != null && indexed != IndexedValue.FALSE)
        {
            indexMetaData = new IndexMetaData(null, null, (indexed == IndexedValue.UNIQUE) ? "true" : "false");
            for (int i=0;i<columnMetaData.length;i++)
            {
                indexMetaData.addColumn(columnMetaData[i]);
            }
        }
        if (indexMetaData != null)
        {
            indexMetaData.initialise();
        }

        if (uniqueMetaData == null && uniqueConstraint)
        {
            uniqueMetaData = new UniqueMetaData(null, columnName, null);
            for (int i=0;i<columnMetaData.length;i++)
            {
                uniqueMetaData.addColumn(columnMetaData[i]);
            }
        }
        if (uniqueMetaData != null)
        {
            uniqueMetaData.initialise();
        }

        if (primaryKeyMetaData != null)
        {
            primaryKeyMetaData.initialise();
        }

        setInitialised();
    }

    // ------------------------------- Accessors -------------------------------
 
    /**
     * Add a new ColumnMetaData element
     * @param colmd The ColumnMetaData to add
     */
    public void addColumn(ColumnMetaData colmd)
    {
        columns.add(colmd);
        colmd.parent = this;
    }

    /**
     * Accessor for the table name 
     * @return Returns the table name
     */
    public final String getTable()
    {
        return table;
    }

    /**
     * Accessor for the catalog name 
     * @return Returns the catalog name
     */
    public final String getCatalog()
    {
        return catalog;
    }

    /**
     * Accessor for the schema name 
     * @return Returns the schema name
     */
    public final String getSchema()
    {
        return schema;
    }

    /**
     * Accessor for the column name 
     * @return Returns the column.
     */
    public final String getColumnName()
    {
        return columnName;
    }

    /**
     * Accessor for columnMetaData
     * @return Returns the columnMetaData.
     */
    public final ColumnMetaData[] getColumnMetaData()
    {
        return columnMetaData;
    }

    /**
     * @return Returns the outer.
     */
    public final boolean isOuter()
    {
        if (outer == null)
        {
            return false;
        }
        else
        {
            return outer.booleanValue();
        }
    }

    /**
     * @return Returns the indexMetaData.
     */
    public final IndexMetaData getIndexMetaData()
    {
        return indexMetaData;
    }

    /**
     * @return Returns the uniquexMetaData.
     */
    public final UniqueMetaData getUniqueMetaData()
    {
        return uniqueMetaData;
    }

    /**
     * @return Returns the foreignKeyMetaData.
     */
    public final ForeignKeyMetaData getForeignKeyMetaData()
    {
        return foreignKeyMetaData;
    }

    /**
     * Accessor for primaryKeyMetaData
     * @return Returns the primaryKey MetaData.
     */
    public final PrimaryKeyMetaData getPrimaryKeyMetaData()
    {
        return primaryKeyMetaData;
    }

    // ----------------------------- Mutators ----------------------------------
 
    /**
     * @param foreignKeyMetaData The foreignKeyMetaData to set.
     */
    public final void setForeignKeyMetaData(
            ForeignKeyMetaData foreignKeyMetaData)
    {
        this.foreignKeyMetaData = foreignKeyMetaData;
    }

    /**
     * @param indexMetaData The indexMetaData to set.
     */
    public final void setIndexMetaData(IndexMetaData indexMetaData)
    {
        this.indexMetaData = indexMetaData;
    }

    /**
     * @param uniqueMetaData The uniqueMetaData to set.
     */
    public final void setUniqueMetaData(UniqueMetaData uniqueMetaData)
    {
        this.uniqueMetaData = uniqueMetaData;
    }

    /**
     * Mutator for the PrimaryKey MetaData.
     * @param primaryKeyMetaData The PrimaryKey MetaData to set.
     */
    public final void setPrimaryKeyMetaData(PrimaryKeyMetaData primaryKeyMetaData)
    {
        this.primaryKeyMetaData = primaryKeyMetaData;
    }

    // ----------------------------- Utilities ---------------------------------

    /**
     * Returns a string representation of the object using a prefix
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<join");
        if (table != null)
        {
            sb.append(" table=\"" + table + "\"");
        }
        if (columnName != null)
        {
            sb.append(" column=\"" + columnName + "\"");
        }
        if (outer != null)
        {
            sb.append(" outer=\"" + outer + "\"");
        }
        sb.append(">\n");

        // Primary-key
        if (primaryKeyMetaData != null)
        {
            sb.append(primaryKeyMetaData.toString(prefix + indent, indent));
        }

        // Add columns
        if (columnMetaData != null)
        {
            for (int i=0;i<columnMetaData.length;i++)
            {
                sb.append(columnMetaData[i].toString(prefix + indent,indent));
            }
        }

        // Foreign-key
        if (foreignKeyMetaData != null)
        {
            sb.append(foreignKeyMetaData.toString(prefix + indent, indent));
        }

        // Index
        if (indexMetaData != null)
        {
            sb.append(indexMetaData.toString(prefix + indent, indent));
        }

        // Unique
        if (uniqueMetaData != null)
        {
            sb.append(uniqueMetaData.toString(prefix + indent, indent));
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</join>\n");
        return sb.toString();
    }
}