/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2004 Andy Jefferson - toString(), MetaData, javadocs
2004 Andy Jefferson - added index, and indexed
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.datanucleus.util.StringUtils;

/**
 * Representation of Order MetaData - the ordering of the elements of a List.
 * This caters for 2 types of List.
 * <ul>
 * <li><b>indexed list</b> like in JDO2 where we add an index column</li>
 * <li><b>ordered list</b> like in JPA1 where we use some ordering clause when retrieving</li>
 * </ul>
 *
 * @version $Revision: 1.20 $
 */
public class OrderMetaData extends MetaData implements ColumnMetaDataContainer
{
    /** The name of the column (if specified as input) */
    final String columnName;

    /** the columns  */
    final List columns = new ArrayList();

    /** IndexMetaData. */
    protected IndexMetaData indexMetaData;

    /** The indexing value specified as input. */
    protected IndexedValue indexed=null;

    /** Name of the field in the element that is the ordering field. */
    protected final String mappedBy;

    /**
     * Ordering when using an "ordered list" where the elements are retrieved in a particular order.
     * Only used until initialise().
     */
    protected String ordering;

    /** Ordering of fields (when using "ordered List"). */
    protected FieldOrder[] fieldOrders = null;

    // -------------------------------------------------------------------------
    // Fields below here are not represented in the output MetaData. They are
    // for use internally in the operation of the JDO system. The majority are
    // for convenience to save iterating through the fields since the fields
    // are fixed once initialised.

    /**
     * Contains the metadata for column
     */
    protected ColumnMetaData[] columnMetaData;

    /**
     * Constructor to create a copy of the passed metadata using the provided parent.
     * @param parent The parent
     * @param omd The metadata to copy
     */
    public OrderMetaData(MetaData parent, OrderMetaData omd)
    {
        super(parent);
        this.indexed = omd.indexed;
        this.columnName = omd.columnName;

        // TODO Change these to copy rather than reference
        if (omd.indexMetaData != null)
        {
            this.indexMetaData = omd.indexMetaData;
        }
        for (int i=0;i<omd.columns.size();i++)
        {
            addColumn((ColumnMetaData)omd.columns.get(i));
        }
        this.mappedBy = omd.mappedBy;
        this.ordering = omd.ordering;
    }

    /**
     * Constructor when defining an "indexed list" (like JDO2).
     * @param column Name of column
     * @param indexed The indexed value
     * @param mappedBy The field in the element that provides the ordering
     */
    public OrderMetaData(final String column,
                         final String indexed,
                         final String mappedBy)
    {
        super(null); // Ignore parent
        this.columnName = (StringUtils.isWhitespace(column) ? null : column);
        this.indexed = IndexedValue.getIndexedValue(indexed);
        this.mappedBy = (StringUtils.isWhitespace(mappedBy) ? null : mappedBy);
        this.ordering = null;
    }

    /**
     * Constructor when defining an "ordered list" (like JPA1)
     * @param ordering ordering when using an ordered list where the elements are retrieved in a particular order
     */
    public OrderMetaData(final String ordering)
    {
        super(null); // Ignore parent
        this.columnName = null;
        this.mappedBy = null;
        this.ordering = (StringUtils.isWhitespace(ordering) ? null : ordering);
    }

    /**
     * Method to initialise the object, creating internal convenience arrays.
     * Initialises all sub-objects.
     */
    public void initialise()
    {
        if (hasExtension("list-ordering"))
        {
            // User has provided JPOX extension "list-ordering" meaning that we use an ordered list
            // for this collection (like in JPA)
            String val = getValueForExtension("list-ordering");
            if (!StringUtils.isWhitespace(val))
            {
                this.ordering = val;
            }
        }

        columnMetaData = new ColumnMetaData[columns.size()];
        if (columns.size() == 0 && columnName != null)
        {
            columnMetaData = new ColumnMetaData[1];
            columnMetaData[0] = new ColumnMetaData(this, columnName);
            columnMetaData[0].initialise();
        }
        else
        {
            columnMetaData = new ColumnMetaData[columns.size()];
            for (int i=0; i<columnMetaData.length; i++)
            {
                columnMetaData[i] = (ColumnMetaData) columns.get(i);
                columnMetaData[i].initialise();
            }
        }

        // Interpret the "indexed" value to create our IndexMetaData where it wasn't specified that way
        if (indexMetaData == null && columnMetaData != null && indexed != null && indexed != IndexedValue.FALSE)
        {
            indexMetaData = new IndexMetaData(null, null, (indexed == IndexedValue.UNIQUE) ? "true" : "false");
            for (int i=0;i<columnMetaData.length;i++)
            {
                indexMetaData.addColumn(columnMetaData[i]);
            }
        }
        if (indexMetaData != null)
        {
            indexMetaData.initialise();
        }

        if (mappedBy != null)
        {
            // Check that the "mapped-by" field exists in the element class
            AbstractMemberMetaData fmd = (AbstractMemberMetaData)parent;
            AbstractClassMetaData elementCmd = fmd.getCollection().element.classMetaData;
            if (elementCmd != null && !elementCmd.hasMember(mappedBy))
            {
                throw new InvalidMetaDataException(LOCALISER, "044137", 
                    fmd.getFullFieldName(), elementCmd.getFullClassName(), mappedBy);
            }
        }

        if (ordering != null)
        {
            // "ordered List", so split the ordering into its components
            AbstractMemberMetaData fmd = (AbstractMemberMetaData)parent;
            AbstractClassMetaData elementCmd = fmd.getCollection().element.classMetaData;
            if (elementCmd != null && ordering.equals("#PK"))
            {
                // Order using the PK of the element PC
                fieldOrders = new FieldOrder[elementCmd.getNoOfPrimaryKeyMembers()];
                String[] pkFieldNames = elementCmd.getPrimaryKeyMemberNames();
                int i = 0;
                for (int pkFieldNum=0;pkFieldNum<fieldOrders.length;pkFieldNum++)
                {
                    fieldOrders[i++] = new FieldOrder(pkFieldNames[pkFieldNum]);
                }
            }
            else if (elementCmd != null)
            {
                // Order using the provided definition of element PC fields
                StringTokenizer tokeniser = new StringTokenizer(ordering, ",");
                int num = tokeniser.countTokens();
                fieldOrders = new FieldOrder[num];
                int i = 0;
                while (tokeniser.hasMoreTokens())
                {
                    String nextToken = tokeniser.nextToken().trim();
                    String fieldName = null;
                    boolean forward = true;
                    int spacePos = nextToken.indexOf(' ');
                    if (spacePos > 0)
                    {
                        // Of the form "{field} {ordering}"
                        fieldName = nextToken.substring(0, spacePos);
                        String direction = nextToken.substring(spacePos+1).trim();
                        if (direction.equalsIgnoreCase("DESC"))
                        {
                            forward = false;
                        }
                        else if (!direction.equalsIgnoreCase("ASC"))
                        {
                            throw new InvalidMetaDataException(LOCALISER, "044139",
                                fmd.getFullFieldName(), direction);
                        }
                    }
                    else
                    {
                        // Of the form "{field}"
                        fieldName = nextToken;
                    }

                    if (elementCmd != null && !elementCmd.hasMember(fieldName))
                    {
                        throw new InvalidMetaDataException(LOCALISER, "044138",
                            fmd.getFullFieldName(), elementCmd.getFullClassName(), fieldName);
                    }

                    // Add the field order
                    fieldOrders[i] = new FieldOrder(fieldName);
                    if (!forward)
                    {
                        fieldOrders[i].setBackward();
                    }
                    i++;
                }
            }
            else
            {
                // List<NonPC> so use natural ordering
                fieldOrders = new FieldOrder[0];
                // TODO Set this to some special value?
            }
        }

        setInitialised();
    }

    /**
     * Add a new ColumnMetaData element
     * @param colmd The Column MetaData 
     */
    public void addColumn(ColumnMetaData colmd)
    {
        columns.add(colmd);
        colmd.parent = this;
    }

    /**
     * Mutator for the index MetaData 
     * @param indexMetaData The indexMetaData to set.
     */
    public final void setIndexMetaData(IndexMetaData indexMetaData)
    {
        this.indexMetaData = indexMetaData;
    }

    /**
     * Convenience method to return if the List is an "indexed List" like in JDO2.
     * @return Whether the List is indexed (if false means that it is "ordered" (like in JPA1)
     */
    public boolean isIndexedList()
    {
        return (fieldOrders == null);
    }

    /**
     * Accessor for the field in the element that provides the ordering.
     * @return Field in the value that provides the ordering.
     */
    public String getMappedBy()
    {
        return mappedBy;
    }

    /**
     * Accessor for field ordering (if using "ordered List".
     * @return Field orders
     */
    public FieldOrder[] getFieldOrders()
    {
        return fieldOrders;
    }

    /**
     * Accessor for the Column MetaData for the columns
     * @return Returns the columnMetaData.
     */
    public final ColumnMetaData[] getColumnMetaData()
    {
        return columnMetaData;
    }

    /**
     * Accessor for the column name 
     * @return Returns the column.
     */
    public final String getColumnName()
    {
        return columnName;
    }

    /**
     * Accessor for indexMetaData
     * @return Returns the indexMetaData.
     */
    public final IndexMetaData getIndexMetaData()
    {
        return indexMetaData;
    }

    // ------------------------------ Utilities --------------------------------

    /**
     * Returns a string representation of the object using a prefix
     * This can be used as part of a facility to output a MetaData file. 
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        // Field needs outputting so generate metadata
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<order");
        if (columnName != null)
        {
            sb.append(" column=\"" + columnName + "\"");
        }
        if (indexed != null)
        {
            sb.append(" indexed=\"" + indexed.toString() + "\"");
        }
        if (mappedBy != null)
        {
            sb.append(" mapped-by=\"" + mappedBy + "\"");
        }
        sb.append(">\n");

        // Add columns
        for (int i=0; i<columns.size(); i++)
        {
            ColumnMetaData c = (ColumnMetaData)columns.get(i);
            sb.append(c.toString(prefix + indent,indent));
        }

        // Add index
        if (indexMetaData != null)
        {
            sb.append(indexMetaData.toString(prefix + indent,indent));
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</order>\n");
        return sb.toString();
    }

    /**
     * Definition of ordering using a field.
     * Used by "ordered lists".
     *
     * @version $Revision: 1.20 $
     */
    public static class FieldOrder
    {
        String fieldName;
        boolean forward = true;

        public FieldOrder(String name)
        {
            fieldName = name;
        }

        public void setBackward()
        {
            forward = false;
        }

        public String getFieldName()
        {
            return fieldName;
        }

        public boolean isForward()
        {
            return forward;
        }
    }
}