/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.util.StringUtils;

/**
 * Representation of the Meta-Data for a package.
 *
 * @since 1.1
 * @version $Revision: 1.15 $
 */
public class PackageMetaData extends MetaData
{
    /** List of interfaces (uses List to retain positioning). */
    protected List interfaces = null;

    /** List of classes (uses List to retain positioning). */
    protected List classes = null;

    /** Sequence generators. */
    protected Collection sequences = null;

    /** Table generators. */
    protected Collection tableGenerators = null;

    /** Package name */
    protected final String name;

    /** Catalog name for all classes in this package */
    protected String catalog;

    /** Schema name for all classes in this package */
    protected String schema;

    /**
     * Constructor.
     * @param parent The FileMetaData owner
     * @param name Name of package
     * @param catalog Catalog name for all classes in the package
     * @param schema Schema name for all classes in the package
     */
    public PackageMetaData(FileMetaData parent, final String name, final String catalog, final String schema)
    {
        super(parent);

        this.name = name;
        if (this.name == null)
        {
            throw new NucleusUserException(LOCALISER.msg("044041",
                "name", getFileMetaData().getFilename(), "package"));
        }

        this.catalog = (StringUtils.isWhitespace(catalog) ? null : catalog);
        this.schema = (StringUtils.isWhitespace(schema) ? null : schema);
        if (this.catalog == null && parent.getCatalog() != null)
        {
            // Nothing specified for this package, but file has a value
            this.catalog = parent.getCatalog();
        }
        if (this.schema == null && parent.getSchema() != null)
        {
            // Nothing specified for this package, but file has a value
            this.schema = parent.getSchema();
        }
    }

    // -------------------------------- Accessors ------------------------------

    /**
     * Accessor for the parent FileMetaData.
     * @return File MetaData.
     */
    public FileMetaData getFileMetaData()
    {
        if (parent != null)
        {
            return (FileMetaData)parent;
        }
        return null;
    }

    /**
     * Accessor for the name of the package
     * @return package name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Accessor for the catalog name for all classes in this package
     * @return Catalog name to use.
     */
    public String getCatalog()
    {
        return catalog;
    }

    /**
     * Accessor for the schema name for all classes in this package
     * @return Schema name to use.
     */
    public String getSchema()
    {
        return schema;
    }

    /**
     * Accessor for the number of interfaces.
     * @return Number of interfaces.
     */
    public int getNoOfInterfaces()
    {
        return interfaces != null ? interfaces.size() : 0;
    }
 
    /**
     * Accessor for the Meta-Data of a interface in this package.
     * @param i interface index
     * @return Meta-Data for the interface
     */
    public InterfaceMetaData getInterface(int i)
    {
        return (InterfaceMetaData)interfaces.get(i);
    }

    /**
     * Accessor for the Meta-Data of an interface with the specified name.
     * @param name the name of the interface
     * @return Meta-Data for the interface
     */
    public InterfaceMetaData getInterface(String name)
    {
        Iterator iter=interfaces.iterator();
        while (iter.hasNext())
        {
            InterfaceMetaData imd = (InterfaceMetaData)iter.next();
            if (imd.getName().equals(name))
            {
                return imd;
            }
        }
        return null;
    }

    /**
     * Accessor for the number of classes.
     * @return Number of classes.
     */
    public int getNoOfClasses()
    {
        return classes != null ? classes.size() : 0;
    }

    /**
     * Accessor for the Meta-Data of a class in this package.
     * @param i class index
     * @return Meta-Data for the class
     */
    public ClassMetaData getClass(int i)
    {
        return (ClassMetaData)classes.get(i);
    }

    /**
     * Accessor for the Meta-Data of a class with the specified name.
     * @param name the name of the class
     * @return Meta-Data for the class.
     */
    public ClassMetaData getClass(String name)
    {
        Iterator iter=classes.iterator();
        while (iter.hasNext())
        {
            ClassMetaData cmd = (ClassMetaData)iter.next();
            if (cmd.getName().equals(name))
            {
                return cmd;
            }
        }
        return null;
    }

    /**
     * Accessor for the number of sequences.
     * @return Number of sequences.
     */
    public int getNoOfSequences()
    {
        return sequences != null ? sequences.size() : 0;
    }
 
    /**
     * Accessor for the Meta-Data for the sequences in this package.
     * @return Meta-Data for the sequences
     */
    public SequenceMetaData[] getSequences()
    {
        return (sequences == null ? null : (SequenceMetaData[])sequences.toArray(new SequenceMetaData[sequences.size()]));
    }

    /**
     * Accessor for the Meta-Data of an sequence with the specified name.
     * @param name the name of the sequence
     * @return Meta-Data for the sequence
     */
    public SequenceMetaData getSequence(String name)
    {
        Iterator iter = sequences.iterator();
        while (iter.hasNext())
        {
            SequenceMetaData seqmd = (SequenceMetaData)iter.next();
            if (seqmd.getName().equals(name))
            {
                return seqmd;
            }
        }
        return null;
    }

    /**
     * Accessor for the number of table generators.
     * @return Number of table generators.
     */
    public int getNoOfTableGenerators()
    {
        return tableGenerators != null ? tableGenerators.size() : 0;
    }
 
    /**
     * Accessor for the Meta-Data for the table generators in this package.
     * @return Meta-Data for the table generators
     */
    public TableGeneratorMetaData[] getTableGenerators()
    {
        return (tableGenerators == null ? null : (TableGeneratorMetaData[])tableGenerators.toArray(new TableGeneratorMetaData[tableGenerators.size()]));
    }

    /**
     * Accessor for the Meta-Data of a table generator with the specified name.
     * @param name the name of the table generator
     * @return Meta-Data for the table generator
     */
    public TableGeneratorMetaData getTableGenerator(String name)
    {
        Iterator iter = tableGenerators.iterator();
        while (iter.hasNext())
        {
            TableGeneratorMetaData tgmd = (TableGeneratorMetaData)iter.next();
            if (tgmd.getName().equals(name))
            {
                return tgmd;
            }
        }
        return null;
    }

    // -------------------------------- Mutators -------------------------------

    /**
     * Method to add a class Meta-Data to the package.
     * @param cmd Meta-Data for the class
     */
    public void addClass(ClassMetaData cmd)
    {
        if (cmd == null)
        {
            return;
        }
        if (classes == null)
        {
            classes = new ArrayList();
        }
        classes.add(cmd);
    }

    /**
     * Method to add a interface Meta-Data to the package.
     * @param imd Meta-Data for the interface
     */
    public void addInterface(InterfaceMetaData imd)
    {
        if (imd == null)
        {
            return;
        }
        if (interfaces == null)
        {
            interfaces = new ArrayList();
        }
        interfaces.add(imd);
    }

    /**
     * Method to add a sequence Meta-Data to the package.
     * @param seqmd Meta-Data for the sequence
     */
    public void addSequence(SequenceMetaData seqmd)
    {
        if (seqmd == null)
        {
            return;
        }
        if (sequences == null)
        {
            sequences = new HashSet();
        }
        sequences.add(seqmd);
        seqmd.parent = this;
    }

    /**
     * Method to add a table generator Meta-Data to the package.
     * @param tgmd Meta-Data for the table generator
     */
    public void addTableGenerator(TableGeneratorMetaData tgmd)
    {
        if (tgmd == null)
        {
            return;
        }
        if (tableGenerators == null)
        {
            tableGenerators = new HashSet();
        }
        tableGenerators.add(tgmd);
        tgmd.parent = this;
    }

    // ------------------------------ Utilities --------------------------------

    /**
     * Returns a string representation of the object.
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<package name=\"" + name + "\"");
        if (catalog != null)
        {
            sb.append(" catalog=\"" + catalog + "\"");
        }
        if (schema != null)
        {
            sb.append(" schema=\"" + schema + "\"");
        }
        sb.append(">\n");

        // Add interfaces
        if (interfaces != null)
        {
            Iterator int_iter = interfaces.iterator();
            while (int_iter.hasNext())
            {
                sb.append(((InterfaceMetaData)int_iter.next()).toString(prefix + indent,indent));
            }
        }

        // Add classes
        if (classes != null)
        {
            Iterator cls_iter = classes.iterator();
            while (cls_iter.hasNext())
            {
                sb.append(((ClassMetaData)cls_iter.next()).toString(prefix + indent,indent));
            }
        }

        // Add sequences
        if (sequences != null)
        {
            Iterator seq_iter = sequences.iterator();
            while (seq_iter.hasNext())
            {
                sb.append(((SequenceMetaData)seq_iter.next()).toString(prefix + indent,indent));
            }
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix).append("</package>\n");
        return sb.toString();
    }
}