/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import org.datanucleus.util.StringUtils;

/**
 * Representation of the MetaData of a named Query.
 */
public class QueryMetaData extends MetaData
{
    /** Scope of this query (if any). */
    protected String scope;

    /** Name of the query. */
    protected final String name;

    /** Query language. */
    protected String language;

    /** Whether the query is unmodifiable. */
    protected boolean unmodifiable=false;

    /** The single string query */
    protected String query;

    /** The result class to use. Only applies to SQL. */
    protected String resultClass = null;

    /** Name for the MetaData defining the mapping of the result set (for JPA SQL). */
    protected String resultMetaDataName = null;

    /** Whether the query returns unique. Only applies to SQL. */
    protected boolean unique = false;

    /** Name of any fetch-plan to use. */
    protected String fetchPlanName = null;

    /**
     * Constructor.
     * @param parent the parent of the Query
     * @param scope Scope of the query (if any)
     * @param name The Query name
     * @param language The language name
     * @param unmodifiable The unmodifiable tag
     * @param resultClass The result class (optional)
     * @param resultMetaDataName name of the result MetaData to use (optional)
     * @param unique The unique tag
     * @param fetchPlanName Name of any named fetchPlan to use with this query
     */
    public QueryMetaData(final MetaData parent,
                         final String scope,
                         final String name,
                         final String language,
                         final String unmodifiable,
                         final String resultClass,
                         final String resultMetaDataName,
                         final String unique,
                         final String fetchPlanName)
    {
        super(parent);

        if (StringUtils.isWhitespace(name))
        {
            throw new InvalidMetaDataException(LOCALISER,
                "044154");
        }

        this.scope = (StringUtils.isWhitespace(scope) ? null : scope);
        this.name = name;

        // include-subclasses
        if (unmodifiable != null && unmodifiable.equalsIgnoreCase("true"))
        {
            this.unmodifiable = true;
        }
        else
        {
            this.unmodifiable = false;
        }

        this.language = language;
        if (this.language != null)
        {
            if (this.language.equals("javax.jdo.query.JDOQL")) // Convert to JDOQL
            {
                this.language = QueryLanguage.JDOQL.toString();
            }
            else if (this.language.equals("javax.jdo.query.SQL")) // Convert to SQL
            {
                this.language = QueryLanguage.SQL.toString();
            }
            else if (this.language.equals("javax.jdo.query.JPQL")) // Convert to JPQL
            {
                this.language = QueryLanguage.JPQL.toString();
            }
        }
        this.resultClass = (StringUtils.isWhitespace(resultClass) ? null : resultClass);
        this.resultMetaDataName = resultMetaDataName;

        if (unique != null && unique.equalsIgnoreCase("true"))
        {
            this.unique = true;
        }
        if (!StringUtils.isWhitespace(fetchPlanName))
        {
            this.fetchPlanName = fetchPlanName;
        }
    }

    /**
     * Accessor for the scope of the query
     * @return scope of the query (if any).
     */
    public String getScope()
    {
        return scope;
    }

    /**
     * Accessor for the query name.
     * @return query name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Accessor for the language.
     * @return language tag value
     */
    public String getLanguage()
    {
        if (language == null)
        {
            // Default to JDOQL
            language = QueryLanguage.JDOQL.toString();
        }
        return language;
    }

    /**
     * Accessor for the unmodifiable tag value.
     * @return unmodifiable tag value
     */
    public boolean isUnmodifiable()
    {
        return unmodifiable;
    }

    /**
     * Accessor for the query
     * @return The query
     */
    public String getQuery()
    {
        return query;
    }

    /**
     * Accessor for the result class
     * @return result class
     */
    public String getResultClass()
    {
        return resultClass;
    }

    /**
     * Accessor for the name of the QueryResult MetaData to use.
     * @return name of the QueryResult MetaData
     */
    public String getResultMetaDataName()
    {
        return resultMetaDataName;
    }

    /**
     * Accessor for the unique tag value.
     * @return unique tag value
     */
    public boolean isUnique()
    {
        return unique;
    }

    /**
     * Mutator for the query
     * @param query The query
     */
    public void setQuery(String query)
    {
        this.query = query;
    }

    /**
     * Accessor for the name of any FetchPlan to use.
     * @return name of the FetchPlan.
     */
    public String getFetchPlanName()
    {
        return fetchPlanName;
    }

    /**
     * Returns a string representation of the object.
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix,String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<query name=\"" + name + "\"\n");
        sb.append(prefix).append("       language=\"" + language + "\"\n");
        if (unique)
        {
            sb.append(prefix).append("       unique=\"true\"\n");
        }
        if (resultClass != null)
        {
            sb.append(prefix).append("       result-class=\"" + resultClass + "\"\n");
        }
        if (fetchPlanName != null)
        {
            sb.append(prefix).append("       fetch-plan=\"" + fetchPlanName + "\"\n");
        }
        sb.append(prefix).append("       unmodifiable=\"" + unmodifiable + "\">\n");
        sb.append(prefix).append(query).append("\n");

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix + "</query>\n");
        return sb.toString();
    }
}