/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import org.datanucleus.util.StringUtils;

/**
 * Representation of the MetaData of a named Sequence (JDO, or JPA).
 *
 * @since 1.1
 * @version $Revision: 1.12 $
 */
public class SequenceMetaData extends MetaData
{
    /** Name under which this sequence generator is known. */
    protected final String name;

    /** Datastore Sequence name */
    protected String datastoreSequence;

    /** factory class name (JDO). */
    protected String factoryClass;

    /** Strategy for this sequence (JDO). */
    protected SequenceStrategy strategy;

    /** Initial value of the sequence. */
    protected long initialValue = -1;

    /** Allocation size for the sequence. */
    protected long allocationSize = -1;

    /**
     * Constructor.
     * @param parent The parent of this element
     * @param name The sequence name
     * @param datastoreSequence The datastore sequence
     * @param factoryClass The factory class
     * @param strategyValue The strategy value
     * @param initialValue Initial value
     * @param allocationSize Allocation size
     */
    public SequenceMetaData(final MetaData parent,
                         final String name,
                         final String datastoreSequence,
                         final String factoryClass,
                         final String strategyValue,
                         final String initialValue,
                         final String allocationSize)
    {
        super(parent);

        this.name = name;
        if (StringUtils.isWhitespace(name))
        {
            throw new InvalidMetaDataException(LOCALISER,
                "044155");
        }

        this.datastoreSequence = (StringUtils.isWhitespace(datastoreSequence) ? null : datastoreSequence);
        this.factoryClass = factoryClass;

        // strategy
        this.strategy = SequenceStrategy.getStrategy(strategyValue);

        if (!StringUtils.isWhitespace(initialValue))
        {
            this.initialValue = new Long(initialValue).longValue();
        }
        if (!StringUtils.isWhitespace(allocationSize))
        {
            this.allocationSize = new Long(allocationSize).longValue();
        }
    }

    /**
     * Convenience accessor for the fully-qualified name of the sequence.
     * @return Fully qualfiied name of the sequence (including the package name).
     */
    public String getFullyQualifiedName()
    {
        PackageMetaData pmd = (PackageMetaData)getParent();
        return pmd.getName() + "." + name;
    }

    /**
     * Accessor for the class name.
     * @return class name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Accessor for the strategy
     * @return strategy tag value
     */
    public SequenceStrategy getStrategy()
    {
        return strategy;
    }

    /**
     * Accessor for the sequence name
     * @return The sequence name
     */
    public String getDatastoreSequence()
    {
        return datastoreSequence;
    }

    /**
     * Accessor for the factory class
     * @return factory class
     */
    public String getFactoryClass()
    {
        return factoryClass;
    }

    /**
     * Accessor for the initial value of the sequence.
     * @return The initial value
     */
    public long getInitialValue()
    {
        return initialValue;
    }

    /**
     * Accessor for the allocation size of the sequence.
     * @return The allocation size
     */
    public long getAllocationSize()
    {
        return allocationSize;
    }

    /**
     * Returns a string representation of the object.
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix, String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<sequence name=\"" + name + "\"\n");
        if (datastoreSequence != null)
        {
            sb.append(prefix).append("       datastore-sequence=\"" + datastoreSequence + "\"\n");
        }
        if (factoryClass != null)
        {
            sb.append(prefix).append("       factory-class=\"" + factoryClass + "\"\n");
        }
        if (strategy != null)
        {
            sb.append(prefix).append("       strategy=\"" + strategy.toString() + "\">\n");
        }

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix + "</sequence>\n");
        return sb.toString();
    }
}