/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.metadata;

import org.datanucleus.util.StringUtils;

/**
 * Representation of the MetaData of a TableGenerator (JPA).
 *
 * @since 1.1
 * @version $Revision: 1.5 $
 */
public class TableGeneratorMetaData extends MetaData
{
    /** Name under which this table generator is known. */
    protected final String name;

    /** Name of the table to use for sequences */
    protected String tableName;

    /** Name of the catalog to use for the table */
    protected String catalogName;

    /** Name of the schema to use for the table */
    protected String schemaName;

    /** Name of the primary-key column name */
    protected String pkColumnName;

    /** Name of the value column name */
    protected String valueColumnName;

    /** Name of the primary-key column value */
    protected String pkColumnValue;

    /** Initial value in the table. */
    protected long initialValue = 0;

    /** Allocation size for ids from the table. */
    protected long allocationSize = 50;

    /**
     * Constructor.
     * @param parent The parent of this element
     * @param name The generator name
     * @param tableName The table name
     * @param catalogName Catalog name for the table
     * @param schemaName Schema name for the table
     * @param pkColumnName Name of PK column in table
     * @param valueColumnName Name of value column in table
     * @param pkColumnValue Value of the pk in the table
     * @param initialValue Initial value
     * @param allocationSize Allocation size
     */
    public TableGeneratorMetaData(final MetaData parent,
                         final String name,
                         final String tableName,
                         final String catalogName,
                         final String schemaName,
                         final String pkColumnName,
                         final String valueColumnName,
                         final String pkColumnValue,
                         final String initialValue,
                         final String allocationSize)
    {
        super(parent);

        if (StringUtils.isWhitespace(name))
        {
            throw new InvalidMetaDataException(LOCALISER,
                "044155");
        }

        this.name = name;
        this.tableName = (StringUtils.isWhitespace(tableName) ? null : tableName);
        this.catalogName = (StringUtils.isWhitespace(catalogName) ? null : catalogName);
        this.schemaName = (StringUtils.isWhitespace(schemaName) ? null : schemaName);
        this.pkColumnName = (StringUtils.isWhitespace(pkColumnName) ? null : pkColumnName);
        this.valueColumnName = (StringUtils.isWhitespace(valueColumnName) ? null : valueColumnName);
        this.pkColumnValue = (StringUtils.isWhitespace(pkColumnValue) ? null : pkColumnValue);
        if (!StringUtils.isWhitespace(initialValue))
        {
            this.initialValue = new Long(initialValue).longValue();
        }
        if (!StringUtils.isWhitespace(allocationSize))
        {
            this.allocationSize = new Long(allocationSize).longValue();
        }
    }

    /**
     * Convenience accessor for the fully-qualified name of the sequence.
     * @return Fully qualfiied name of the sequence (including the package name).
     */
    public String getFullyQualifiedName()
    {
        PackageMetaData pmd = (PackageMetaData)getParent();
        return pmd.getName() + "." + name;
    }

    /**
     * Accessor for the generator name.
     * @return generator name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Accessor for the table name.
     * @return table name
     */
    public String getTableName()
    {
        return tableName;
    }

    /**
     * Accessor for the catalog name.
     * @return catalog name
     */
    public String getCatalogName()
    {
        return catalogName;
    }

    /**
     * Accessor for the schema name.
     * @return schema name
     */
    public String getSchemaName()
    {
        return schemaName;
    }

    /**
     * Accessor for the PK column name.
     * @return PK column name
     */
    public String getPKColumnName()
    {
        return pkColumnName;
    }

    /**
     * Accessor for the value column name.
     * @return Value column name
     */
    public String getValueColumnName()
    {
        return valueColumnName;
    }

    /**
     * Accessor for the PK column value.
     * @return PK column value
     */
    public String getPKColumnValue()
    {
        return pkColumnValue;
    }

    /**
     * Accessor for the initial value of the sequence.
     * @return The initial value
     */
    public long getInitialValue()
    {
        return initialValue;
    }

    /**
     * Accessor for the allocation size of the sequence.
     * @return The allocation size
     */
    public long getAllocationSize()
    {
        return allocationSize;
    }

    /**
     * Returns a string representation of the object.
     * @param prefix prefix string
     * @param indent indent string
     * @return a string representation of the object.
     */
    public String toString(String prefix, String indent)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(prefix).append("<table-generator name=\"" + name + "\"\n");

        // Add extensions
        sb.append(super.toString(prefix + indent,indent));

        sb.append(prefix + "</table-generator>\n");
        return sb.toString();
    }
}